/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FieldList;
import com.google.cloud.bigquery.StandardSQLTypeName;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.RowType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimeWithTimeZoneType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

public enum BigQueryType {
    BOOLEAN((Type)BooleanType.BOOLEAN, BigQueryType::simpleToStringConverter),
    BYTES((Type)VarbinaryType.VARBINARY, BigQueryType::bytesToStringConverter),
    DATE((Type)DateType.DATE, BigQueryType::dateToStringConverter),
    DATETIME((Type)TimestampType.TIMESTAMP_MICROS, BigQueryType::datetimeToStringConverter),
    FLOAT((Type)DoubleType.DOUBLE, BigQueryType::floatToStringConverter),
    GEOGRAPHY((Type)VarcharType.VARCHAR, BigQueryType.unsupportedToStringConverter()),
    INTEGER((Type)BigintType.BIGINT, BigQueryType::simpleToStringConverter),
    NUMERIC(null, BigQueryType::numericToStringConverter),
    BIGNUMERIC(null, BigQueryType::numericToStringConverter),
    RECORD(null, BigQueryType.unsupportedToStringConverter()),
    STRING((Type)VarcharType.createUnboundedVarcharType(), BigQueryType::stringToStringConverter),
    TIME((Type)TimeType.TIME_MICROS, BigQueryType::timeToStringConverter),
    TIMESTAMP((Type)TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS, BigQueryType::timestampToStringConverter);

    private static final int[] NANO_FACTOR;
    private static final DateTimeFormatter TIME_FORMATTER;
    private static final DateTimeFormatter DATETIME_FORMATTER;
    private final Type nativeType;
    private final OptionalToStringConverter toStringConverter;

    private BigQueryType(Type nativeType, ToStringConverter toStringConverter) {
        this(nativeType, (Object value) -> Optional.of(toStringConverter.convertToString(value)));
        Objects.requireNonNull(toStringConverter, "toStringConverter is null");
    }

    private BigQueryType(Type nativeType, OptionalToStringConverter toStringConverter) {
        this.nativeType = nativeType;
        this.toStringConverter = toStringConverter;
    }

    static RowType.Field toRawTypeField(Map.Entry<String, Adaptor> entry) {
        return BigQueryType.toRawTypeField(entry.getKey(), entry.getValue());
    }

    private static RowType.Field toRawTypeField(String name, Adaptor typeAdaptor) {
        Type trinoType = typeAdaptor.getTrinoType();
        return RowType.field((String)name, (Type)trinoType);
    }

    @VisibleForTesting
    public static LocalDateTime toLocalDateTime(String datetime) {
        int dotPosition = datetime.indexOf(46);
        if (dotPosition == -1) {
            return LocalDateTime.from(DateTimeFormatter.ISO_LOCAL_DATE_TIME.parse(datetime));
        }
        LocalDateTime result = LocalDateTime.from(DateTimeFormatter.ISO_LOCAL_DATE_TIME.parse(datetime.substring(0, dotPosition)));
        String nanosStr = datetime.substring(dotPosition + 1);
        int nanoOfSecond = Integer.parseInt(nanosStr) * NANO_FACTOR[nanosStr.length()];
        return result.withNano(nanoOfSecond);
    }

    public static long toTrinoTimestamp(String datetime) {
        Instant instant = BigQueryType.toLocalDateTime(datetime).toInstant(ZoneOffset.UTC);
        return instant.getEpochSecond() * 1000000L + (long)(instant.getNano() / 1000);
    }

    private static String floatToStringConverter(Object value) {
        return String.format("CAST('%s' AS float64)", value);
    }

    private static String simpleToStringConverter(Object value) {
        return String.valueOf(value);
    }

    private static OptionalToStringConverter unsupportedToStringConverter() {
        return value -> Optional.empty();
    }

    @VisibleForTesting
    public static String dateToStringConverter(Object value) {
        LocalDate date = LocalDate.ofEpochDay((Long)value);
        return BigQueryType.quote(date.toString());
    }

    private static String datetimeToStringConverter(Object value) {
        long epochMicros = (Long)value;
        long epochSeconds = Math.floorDiv(epochMicros, 1000000);
        int nanoAdjustment = Math.floorMod(epochMicros, 1000000) * 1000;
        return BigQueryType.formatTimestamp(epochSeconds, nanoAdjustment, ZoneOffset.UTC);
    }

    @VisibleForTesting
    public static String timeToStringConverter(Object value) {
        long time = (Long)value;
        Verify.verify((0L <= time ? 1 : 0) != 0, (String)"Invalid time value: %s", (long)time);
        long epochSeconds = time / 1000000000000L;
        long nanoAdjustment = time % 1000000000000L / 1000L;
        return TIME_FORMATTER.format(BigQueryType.toZonedDateTime(epochSeconds, nanoAdjustment, ZoneOffset.UTC));
    }

    @VisibleForTesting
    public static String timestampToStringConverter(Object value) {
        LongTimestampWithTimeZone timestamp = (LongTimestampWithTimeZone)value;
        long epochMillis = timestamp.getEpochMillis();
        long epochSeconds = Math.floorDiv(epochMillis, 1000);
        int nanoAdjustment = Math.floorMod(epochMillis, 1000) * 1000000 + timestamp.getPicosOfMilli() / 1000;
        ZoneId zoneId = TimeZoneKey.getTimeZoneKey((short)timestamp.getTimeZoneKey()).getZoneId();
        return BigQueryType.formatTimestamp(epochSeconds, nanoAdjustment, zoneId);
    }

    private static String formatTimestamp(long epochSeconds, long nanoAdjustment, ZoneId zoneId) {
        return DATETIME_FORMATTER.format(BigQueryType.toZonedDateTime(epochSeconds, nanoAdjustment, zoneId));
    }

    private static ZonedDateTime toZonedDateTime(long epochSeconds, long nanoAdjustment, ZoneId zoneId) {
        Instant instant = Instant.ofEpochSecond(epochSeconds, nanoAdjustment);
        return ZonedDateTime.ofInstant(instant, zoneId);
    }

    static String stringToStringConverter(Object value) {
        Slice slice = (Slice)value;
        return BigQueryType.quote(slice.toStringUtf8().replace("'", "\\'"));
    }

    static String numericToStringConverter(Object value) {
        Slice slice = (Slice)value;
        return Decimals.toString((Slice)slice, (int)9);
    }

    static String bytesToStringConverter(Object value) {
        Slice slice = (Slice)value;
        return String.format("FROM_BASE64('%s')", Base64.getEncoder().encodeToString(slice.getBytes()));
    }

    public static Field toField(String name, Type type) {
        if (type instanceof ArrayType) {
            Type elementType = ((ArrayType)type).getElementType();
            return BigQueryType.toInnerField(name, elementType, true);
        }
        return BigQueryType.toInnerField(name, type, false);
    }

    private static Field toInnerField(String name, Type type, boolean repeated) {
        Field.Builder builder = type instanceof RowType ? Field.newBuilder((String)name, (StandardSQLTypeName)StandardSQLTypeName.STRUCT, (FieldList)BigQueryType.toFieldList((RowType)type)) : Field.newBuilder((String)name, (StandardSQLTypeName)BigQueryType.toStandardSqlTypeName(type), (Field[])new Field[0]);
        if (repeated) {
            builder = builder.setMode(Field.Mode.REPEATED);
        }
        return builder.build();
    }

    private static FieldList toFieldList(RowType rowType) {
        ImmutableList.Builder fields = new ImmutableList.Builder();
        for (RowType.Field field : rowType.getFields()) {
            String fieldName = (String)field.getName().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "ROW type does not have field names declared: " + rowType));
            fields.add((Object)BigQueryType.toField(fieldName, field.getType()));
        }
        return FieldList.of((Iterable)fields.build());
    }

    private static StandardSQLTypeName toStandardSqlTypeName(Type type) {
        if (type == BooleanType.BOOLEAN) {
            return StandardSQLTypeName.BOOL;
        }
        if (type == TinyintType.TINYINT || type == SmallintType.SMALLINT || type == IntegerType.INTEGER || type == BigintType.BIGINT) {
            return StandardSQLTypeName.INT64;
        }
        if (type == DoubleType.DOUBLE) {
            return StandardSQLTypeName.FLOAT64;
        }
        if (type instanceof DecimalType) {
            return StandardSQLTypeName.NUMERIC;
        }
        if (type == DateType.DATE) {
            return StandardSQLTypeName.DATE;
        }
        if (type == TimeWithTimeZoneType.createTimeWithTimeZoneType((int)3)) {
            return StandardSQLTypeName.TIME;
        }
        if (type == TimestampType.TIMESTAMP_MICROS) {
            return StandardSQLTypeName.DATETIME;
        }
        if (type == TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS) {
            return StandardSQLTypeName.TIMESTAMP;
        }
        if (type instanceof CharType || type instanceof VarcharType) {
            return StandardSQLTypeName.STRING;
        }
        if (type == VarbinaryType.VARBINARY) {
            return StandardSQLTypeName.BYTES;
        }
        if (type instanceof ArrayType) {
            return StandardSQLTypeName.ARRAY;
        }
        if (type instanceof RowType) {
            return StandardSQLTypeName.STRUCT;
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported column type: " + type.getDisplayName());
    }

    private static String quote(String value) {
        return "'" + value + "'";
    }

    public Optional<String> convertToString(Type type, Object value) {
        if (type instanceof ArrayType) {
            return Optional.empty();
        }
        if (type instanceof DecimalType) {
            String bigqueryTypeName = this.toString();
            Verify.verify((bigqueryTypeName.equals("NUMERIC") || bigqueryTypeName.equals("BIGNUMERIC") ? 1 : 0) != 0, (String)"Expected NUMERIC or BIGNUMERIC: %s", (Object)bigqueryTypeName);
            if (Decimals.isShortDecimal((Type)type)) {
                return Optional.of(String.format("%s '%s'", bigqueryTypeName, Decimals.toString((long)((Long)value), (int)((DecimalType)type).getScale())));
            }
            return Optional.of(String.format("%s '%s'", bigqueryTypeName, Decimals.toString((Slice)((Slice)value), (int)((DecimalType)type).getScale())));
        }
        return this.toStringConverter.convertToString(value);
    }

    public Type getNativeType(Adaptor typeAdaptor) {
        switch (this) {
            case NUMERIC: 
            case BIGNUMERIC: {
                Long precision = typeAdaptor.getPrecision();
                Long scale = typeAdaptor.getScale();
                if (precision != null && scale != null) {
                    return DecimalType.createDecimalType((int)Math.toIntExact(precision), (int)Math.toIntExact(scale));
                }
                if (precision != null) {
                    return DecimalType.createDecimalType((int)Math.toIntExact(precision));
                }
                return DecimalType.createDecimalType((int)38, (int)9);
            }
            case RECORD: {
                Map<String, Adaptor> subTypes = typeAdaptor.getBigQuerySubTypes();
                Preconditions.checkArgument((!subTypes.isEmpty() ? 1 : 0) != 0, (Object)"a record or struct must have sub-fields");
                List fields = subTypes.entrySet().stream().map(BigQueryType::toRawTypeField).collect(Collectors.toList());
                return RowType.from(fields);
            }
        }
        return this.nativeType;
    }

    static {
        NANO_FACTOR = new int[]{-1, 100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10, 1};
        TIME_FORMATTER = DateTimeFormatter.ofPattern("''HH:mm:ss.SSSSSS''");
        DATETIME_FORMATTER = DateTimeFormatter.ofPattern("''yyyy-MM-dd HH:mm:ss.SSSSSS''");
    }

    @FunctionalInterface
    static interface OptionalToStringConverter {
        public Optional<String> convertToString(Object var1);
    }

    @FunctionalInterface
    static interface ToStringConverter {
        public String convertToString(Object var1);
    }

    static interface Adaptor {
        public BigQueryType getBigQueryType();

        public Long getPrecision();

        public Long getScale();

        public Map<String, Adaptor> getBigQuerySubTypes();

        public Field.Mode getMode();

        default public Type getTrinoType() {
            Type rawType = this.getBigQueryType().getNativeType(this);
            return this.getMode() == Field.Mode.REPEATED ? new ArrayType(rawType) : rawType;
        }
    }
}

