/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.lambda.service;

import com.amazonaws.auth.policy.Policy;
import com.amazonaws.auth.policy.Statement;
import com.amazonaws.services.lambda.AWSLambda;
import com.amazonaws.services.lambda.model.AliasConfiguration;
import com.amazonaws.services.lambda.model.EventSourceMappingConfiguration;
import com.amazonaws.services.lambda.model.FunctionConfiguration;
import com.amazonaws.services.lambda.model.GetFunctionRequest;
import com.amazonaws.services.lambda.model.GetFunctionResult;
import com.amazonaws.services.lambda.model.GetPolicyRequest;
import com.amazonaws.services.lambda.model.GetPolicyResult;
import com.amazonaws.services.lambda.model.ListAliasesRequest;
import com.amazonaws.services.lambda.model.ListAliasesResult;
import com.amazonaws.services.lambda.model.ListEventSourceMappingsRequest;
import com.amazonaws.services.lambda.model.ListEventSourceMappingsResult;
import com.amazonaws.services.lambda.model.ListFunctionsRequest;
import com.amazonaws.services.lambda.model.ListFunctionsResult;
import com.amazonaws.services.lambda.model.ListVersionsByFunctionRequest;
import com.amazonaws.services.lambda.model.ListVersionsByFunctionResult;
import com.amazonaws.services.lambda.model.ResourceNotFoundException;
import com.amazonaws.services.lambda.model.ServiceException;
import com.amazonaws.services.lambda.model.TooManyRequestsException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.netflix.spinnaker.clouddriver.aws.data.ArnUtils;
import com.netflix.spinnaker.clouddriver.aws.security.AmazonClientProvider;
import com.netflix.spinnaker.clouddriver.aws.security.NetflixAmazonCredentials;
import com.netflix.spinnaker.clouddriver.core.limits.ServiceLimitConfiguration;
import com.netflix.spinnaker.clouddriver.lambda.service.config.LambdaServiceConfig;
import com.netflix.spinnaker.kork.exceptions.SpinnakerException;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.Nullable;

public class LambdaService {
    private final AmazonClientProvider amazonClientProvider;
    private final NetflixAmazonCredentials account;
    private final String region;
    private final int TIMEOUT_MINUTES;
    private final int RETRIES;
    private final Clock clock = Clock.systemDefaultZone();
    private final ObjectMapper mapper;
    private final ExecutorService executorService;

    public LambdaService(AmazonClientProvider amazonClientProvider, NetflixAmazonCredentials account, String region, ObjectMapper mapper, LambdaServiceConfig lambdaServiceConfig, ServiceLimitConfiguration serviceLimitConfiguration) {
        this.amazonClientProvider = amazonClientProvider;
        this.account = account;
        this.region = region;
        this.mapper = mapper;
        this.TIMEOUT_MINUTES = lambdaServiceConfig.getRetry().getTimeout();
        this.RETRIES = lambdaServiceConfig.getRetry().getRetries();
        this.executorService = Executors.newFixedThreadPool(this.computeThreads(serviceLimitConfiguration, lambdaServiceConfig));
    }

    public List<Map<String, Object>> getAllFunctions() throws InterruptedException {
        List<FunctionConfiguration> functions = this.listAllFunctionConfigurations();
        List functionTasks = Collections.synchronizedList(new ArrayList());
        List hydratedFunctionList = Collections.synchronizedList(new ArrayList());
        functions.stream().forEach(f -> {
            ConcurrentHashMap functionAttributes = new ConcurrentHashMap();
            functionTasks.add(() -> this.addBaseAttributes(functionAttributes, f.getFunctionName()));
            functionTasks.add(() -> this.addRevisionsAttributes(functionAttributes, f.getFunctionName()));
            functionTasks.add(() -> this.addAliasAndEventSourceMappingConfigurationAttributes(functionAttributes, f.getFunctionName()));
            functionTasks.add(() -> this.addTargetGroupAttributes(functionAttributes, f.getFunctionName()));
            hydratedFunctionList.add(functionAttributes);
        });
        this.executorService.invokeAll(functionTasks);
        return hydratedFunctionList.stream().filter(lf -> lf.get("functionName") != null).collect(Collectors.toList());
    }

    public Map<String, Object> getFunctionByName(String functionName) throws InterruptedException {
        List<Callable<Void>> functionTasks = Collections.synchronizedList(new ArrayList());
        ConcurrentHashMap<String, Object> functionAttributes = new ConcurrentHashMap<String, Object>();
        this.addBaseAttributes(functionAttributes, functionName);
        if (functionAttributes.isEmpty()) {
            return null;
        }
        functionTasks.add(() -> this.addRevisionsAttributes(functionAttributes, functionName));
        functionTasks.add(() -> this.addAliasAndEventSourceMappingConfigurationAttributes(functionAttributes, functionName));
        functionTasks.add(() -> this.addTargetGroupAttributes(functionAttributes, functionName));
        this.executorService.invokeAll(functionTasks);
        return functionAttributes;
    }

    public List<FunctionConfiguration> listAllFunctionConfigurations() {
        ListFunctionsResult listFunctionsResult;
        AWSLambda lambda = this.amazonClientProvider.getAmazonLambda(this.account, this.region);
        String nextMarker = null;
        ArrayList<FunctionConfiguration> lstFunction = new ArrayList<FunctionConfiguration>();
        do {
            ListFunctionsRequest listFunctionsRequest = new ListFunctionsRequest();
            if (nextMarker != null) {
                listFunctionsRequest.setMarker(nextMarker);
            }
            if ((listFunctionsResult = this.retry("listFunctions", () -> lambda.listFunctions(listFunctionsRequest), this.RETRIES, this.TIMEOUT_MINUTES)) == null) break;
            lstFunction.addAll(listFunctionsResult.getFunctions());
        } while ((nextMarker = listFunctionsResult.getNextMarker()) != null && nextMarker.length() != 0);
        return lstFunction;
    }

    private Void addBaseAttributes(Map<String, Object> functionAttributes, String functionName) {
        GetFunctionResult result = this.getFunctionResult(functionName);
        if (result == null) {
            return null;
        }
        Map attr = (Map)this.mapper.convertValue((Object)result.getConfiguration(), Map.class);
        attr.put("account", this.account.getName());
        attr.put("region", this.region);
        attr.put("code", result.getCode());
        attr.put("tags", result.getTags());
        attr.put("concurrency", result.getConcurrency());
        attr.values().removeAll(Collections.singleton(null));
        functionAttributes.putAll(attr);
        return null;
    }

    private GetFunctionResult getFunctionResult(String functionName) {
        AWSLambda lambda = this.amazonClientProvider.getAmazonLambda(this.account, this.region);
        GetFunctionRequest request = new GetFunctionRequest().withFunctionName(functionName);
        return this.retry("getFunctionRequest", () -> lambda.getFunction(request), this.RETRIES, this.TIMEOUT_MINUTES);
    }

    private Void addRevisionsAttributes(Map<String, Object> functionAttributes, String functionName) {
        Map<String, String> revisions = this.listFunctionRevisions(functionName);
        functionAttributes.put("revisions", revisions);
        return null;
    }

    private Map<String, String> listFunctionRevisions(String functionName) {
        ListVersionsByFunctionResult listVersionsByFunctionResult;
        AWSLambda lambda = this.amazonClientProvider.getAmazonLambda(this.account, this.region);
        String nextMarker = null;
        HashMap<String, String> listRevionIds = new HashMap<String, String>();
        do {
            ListVersionsByFunctionRequest listVersionsByFunctionRequest = new ListVersionsByFunctionRequest();
            listVersionsByFunctionRequest.setFunctionName(functionName);
            if (nextMarker != null) {
                listVersionsByFunctionRequest.setMarker(nextMarker);
            }
            if ((listVersionsByFunctionResult = this.retry("listVersionsByFunction", () -> lambda.listVersionsByFunction(listVersionsByFunctionRequest), this.RETRIES, this.TIMEOUT_MINUTES)) == null) {
                return listRevionIds;
            }
            for (FunctionConfiguration x : listVersionsByFunctionResult.getVersions()) {
                listRevionIds.put(x.getRevisionId(), x.getVersion());
            }
        } while ((nextMarker = listVersionsByFunctionResult.getNextMarker()) != null && nextMarker.length() != 0);
        return listRevionIds;
    }

    private Void addAliasAndEventSourceMappingConfigurationAttributes(Map<String, Object> functionAttributes, String functionName) {
        List<AliasConfiguration> aliasConfigurationList = this.listAliasConfiguration(functionName);
        functionAttributes.put("aliasConfigurations", aliasConfigurationList);
        List<EventSourceMappingConfiguration> eventSourceMappingConfigurationsList = this.listEventSourceMappingConfiguration(functionName);
        for (AliasConfiguration currAlias : aliasConfigurationList) {
            List<EventSourceMappingConfiguration> currAliasEvents = this.listEventSourceMappingConfiguration(currAlias.getAliasArn());
            eventSourceMappingConfigurationsList.addAll(currAliasEvents);
        }
        functionAttributes.put("eventSourceMappings", eventSourceMappingConfigurationsList);
        return null;
    }

    private List<AliasConfiguration> listAliasConfiguration(String functionName) {
        ListAliasesResult listAliasesResult;
        AWSLambda lambda = this.amazonClientProvider.getAmazonLambda(this.account, this.region);
        String nextMarker = null;
        ArrayList<AliasConfiguration> aliasConfigurations = new ArrayList<AliasConfiguration>();
        do {
            ListAliasesRequest listAliasesRequest = new ListAliasesRequest();
            listAliasesRequest.setFunctionName(functionName);
            if (nextMarker != null) {
                listAliasesRequest.setMarker(nextMarker);
            }
            if ((listAliasesResult = this.retry("listAliases", () -> lambda.listAliases(listAliasesRequest), this.RETRIES, this.TIMEOUT_MINUTES)) == null) {
                return aliasConfigurations;
            }
            for (AliasConfiguration x : listAliasesResult.getAliases()) {
                aliasConfigurations.add(x);
            }
        } while ((nextMarker = listAliasesResult.getNextMarker()) != null && nextMarker.length() != 0);
        return aliasConfigurations;
    }

    private List<EventSourceMappingConfiguration> listEventSourceMappingConfiguration(String functionName) {
        ListEventSourceMappingsResult listEventSourceMappingsResult;
        ArrayList<EventSourceMappingConfiguration> eventSourceMappingConfigurations = new ArrayList<EventSourceMappingConfiguration>();
        AWSLambda lambda = this.amazonClientProvider.getAmazonLambda(this.account, this.region);
        String nextMarker = null;
        do {
            ListEventSourceMappingsRequest listEventSourceMappingsRequest = new ListEventSourceMappingsRequest();
            listEventSourceMappingsRequest.setFunctionName(functionName);
            if (nextMarker != null) {
                listEventSourceMappingsRequest.setMarker(nextMarker);
            }
            if ((listEventSourceMappingsResult = this.retry("listEventSourceMappings", () -> lambda.listEventSourceMappings(listEventSourceMappingsRequest), this.RETRIES, this.TIMEOUT_MINUTES)) == null) {
                return eventSourceMappingConfigurations;
            }
            for (EventSourceMappingConfiguration x : listEventSourceMappingsResult.getEventSourceMappings()) {
                eventSourceMappingConfigurations.add(x);
            }
        } while ((nextMarker = listEventSourceMappingsResult.getNextMarker()) != null && nextMarker.length() != 0);
        return eventSourceMappingConfigurations;
    }

    private Void addTargetGroupAttributes(Map<String, Object> functionAttributes, String functionName) {
        List<String> targetGroups = this.getTargetGroupNames(functionName);
        functionAttributes.put("targetGroups", targetGroups);
        return null;
    }

    private List<String> getTargetGroupNames(String functionName) {
        AWSLambda lambda = this.amazonClientProvider.getAmazonLambda(this.account, this.region);
        List<String> targetGroupNames = new ArrayList<String>();
        Predicate<Statement> isAllowStatement = statement -> statement.getEffect().toString().equals(Statement.Effect.Allow.toString());
        Predicate<Statement> isLambdaInvokeAction = statement -> statement.getActions().stream().anyMatch(action -> action.getActionName().equals("lambda:InvokeFunction"));
        Predicate<Statement> isElbPrincipal = statement -> statement.getPrincipals().stream().anyMatch(principal -> principal.getId().equals("elasticloadbalancing.amazonaws.com"));
        try {
            GetPolicyResult result = this.retry("getPolicy", () -> lambda.getPolicy(new GetPolicyRequest().withFunctionName(functionName)), this.RETRIES, this.TIMEOUT_MINUTES);
            if (result == null) {
                return targetGroupNames;
            }
            String json = result.getPolicy();
            Policy policy = Policy.fromJson((String)json);
            targetGroupNames = policy.getStatements().stream().filter(isAllowStatement.and(isLambdaInvokeAction).and(isElbPrincipal)).flatMap(statement -> statement.getConditions().stream()).filter(condition -> condition.getType().equals("ArnLike") && condition.getConditionKey().equals("AWS:SourceArn")).flatMap(condition -> condition.getValues().stream()).flatMap(value -> ArnUtils.extractTargetGroupName((String)value).stream()).collect(Collectors.toList());
        }
        catch (ResourceNotFoundException resourceNotFoundException) {
            // empty catch block
        }
        return targetGroupNames;
    }

    /*
     * Loose catch block
     */
    @Nullable
    private <T> T retry(String requestName, Supplier<T> fn, int maxRetries, int timeoutMinutes) {
        int retries = 0;
        long startTime = this.clock.instant().toEpochMilli();
        while (true) {
            long currentTime;
            if ((currentTime = this.clock.instant().toEpochMilli()) > startTime + TimeUnit.MINUTES.toMillis(timeoutMinutes)) {
                throw new SpinnakerException("Failed to complete sdk method 'lambda:" + requestName + "' before the timeout.");
            }
            try {
                return fn.get();
            }
            catch (ResourceNotFoundException notFoundException) {
                return null;
            }
            catch (ServiceException | TooManyRequestsException e) {
                if (retries >= maxRetries - 1) {
                    throw e;
                }
                if (!(e instanceof ServiceException)) continue;
                ++retries;
                continue;
            }
            break;
        }
        catch (Exception e) {
            throw e;
        }
    }

    private int computeThreads(ServiceLimitConfiguration serviceLimitConfiguration, LambdaServiceConfig lambdaServiceConfig) {
        int serviceLimit = serviceLimitConfiguration.getLimit("rateLimit", AWSLambda.class.getSimpleName(), this.account.getName(), "aws", Double.valueOf(5.0)).intValue();
        return Math.min(serviceLimit * 2, lambdaServiceConfig.getConcurrency().getThreads());
    }
}

