package io.vertx.mutiny.core.file;

import java.util.Map;
import java.util.stream.Collectors;
import io.smallrye.mutiny.Multi;
import io.smallrye.mutiny.Uni;
import java.util.function.Consumer;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Publisher;
import io.smallrye.mutiny.vertx.TypeArg;
import io.vertx.codegen.annotations.Fluent;
import io.smallrye.common.annotation.CheckReturnValue;
import io.vertx.core.file.OpenOptions;
import io.vertx.core.Future;
import java.util.List;
import io.vertx.core.file.CopyOptions;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Contains a broad set of operations for manipulating files on the file system.
 * <p>
 * A (potential) blocking and non blocking version of each operation is provided.
 * <p>
 * The non blocking versions take a handler which is called when the operation completes or an error occurs.
 * <p>
 * The blocking versions are named <code>xxxBlocking</code> and return the results, or throw exceptions directly.
 * In many cases, depending on the operating system and file system some of the potentially blocking operations
 * can return quickly, which is why we provide them, but it's highly recommended that you test how long they take to
 * return in your particular application before using them on an event loop.
 * <p>
 * Please consult the documentation for more information on file system support.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.file.FileSystem original} non Mutiny-ified interface using Vert.x codegen.
 */

@io.smallrye.mutiny.vertx.MutinyGen(io.vertx.core.file.FileSystem.class)
public class FileSystem {

  public static final io.smallrye.mutiny.vertx.TypeArg<FileSystem> __TYPE_ARG = new io.smallrye.mutiny.vertx.TypeArg<>(    obj -> new FileSystem((io.vertx.core.file.FileSystem) obj),
    FileSystem::getDelegate
  );

  private final io.vertx.core.file.FileSystem delegate;
  
  public FileSystem(io.vertx.core.file.FileSystem delegate) {
    this.delegate = delegate;
  }

  public FileSystem(Object delegate) {
    this.delegate = (io.vertx.core.file.FileSystem)delegate;
  }

  /**
   * Empty constructor used by CDI, do not use this constructor directly.
   **/
  FileSystem() {
    this.delegate = null;
  }

  public io.vertx.core.file.FileSystem getDelegate() {
    return delegate;
  }

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    FileSystem that = (FileSystem) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The copy will fail if the destination already exists.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> copy(String from, String to) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.copy(from, to, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param from the path to copy from
   * @param to the path to copy to
   * @return the Void instance produced by the operation.
   */
  public Void copyAndAwait(String from, String to) { 
    return (Void) copy(from, to).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String)} but you don't need to compose it with other operations.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem copyAndForget(String from, String to) { 
    copy(from, to).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> copy(String from, String to, io.vertx.core.file.CopyOptions options) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.copy(from, to, options, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String,CopyOptions)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return the Void instance produced by the operation.
   */
  public Void copyAndAwait(String from, String to, io.vertx.core.file.CopyOptions options) { 
    return (Void) copy(from, to, options).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String,CopyOptions)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String,CopyOptions)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#copy(String,String,CopyOptions)} but you don't need to compose it with other operations.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem copyAndForget(String from, String to, io.vertx.core.file.CopyOptions options) { 
    copy(from, to, options).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param from 
   * @param to 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem copyBlocking(String from, String to) { 
    delegate.copyBlocking(from, to);
    return this;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * If <code>recursive</code> is <code>true</code> and <code>from</code> represents a directory, then the directory and its contents
   * will be copied recursively to the destination <code>to</code>.
   * <p>
   * The copy will fail if the destination if the destination already exists.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param recursive 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> copyRecursive(String from, String to, boolean recursive) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.copyRecursive(from, to, recursive, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#copyRecursive(String,String,boolean)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param from the path to copy from
   * @param to the path to copy to
   * @param recursive 
   * @return the Void instance produced by the operation.
   */
  public Void copyRecursiveAndAwait(String from, String to, boolean recursive) { 
    return (Void) copyRecursive(from, to, recursive).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#copyRecursive(String,String,boolean)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#copyRecursive(String,String,boolean)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#copyRecursive(String,String,boolean)} but you don't need to compose it with other operations.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param recursive 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem copyRecursiveAndForget(String from, String to, boolean recursive) { 
    copyRecursive(from, to, recursive).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param from 
   * @param to 
   * @param recursive 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem copyRecursiveBlocking(String from, String to, boolean recursive) { 
    delegate.copyRecursiveBlocking(from, to, recursive);
    return this;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The move will fail if the destination already exists.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> move(String from, String to) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.move(from, to, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#move(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param from the path to copy from
   * @param to the path to copy to
   * @return the Void instance produced by the operation.
   */
  public Void moveAndAwait(String from, String to) { 
    return (Void) move(from, to).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#move(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#move(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#move(String,String)} but you don't need to compose it with other operations.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem moveAndForget(String from, String to) { 
    move(from, to).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> move(String from, String to, io.vertx.core.file.CopyOptions options) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.move(from, to, options, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#move(String,String,CopyOptions)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return the Void instance produced by the operation.
   */
  public Void moveAndAwait(String from, String to, io.vertx.core.file.CopyOptions options) { 
    return (Void) move(from, to, options).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#move(String,String,CopyOptions)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#move(String,String,CopyOptions)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#move(String,String,CopyOptions)} but you don't need to compose it with other operations.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem moveAndForget(String from, String to, io.vertx.core.file.CopyOptions options) { 
    move(from, to, options).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param from 
   * @param to 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem moveBlocking(String from, String to) { 
    delegate.moveBlocking(from, to);
    return this;
  }

  /**
   * Truncate the file represented by <code>path</code> to length <code>len</code> in bytes, asynchronously.
   * <p>
   * The operation will fail if the file does not exist or <code>len</code> is less than <code>zero</code>.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path the path to the file
   * @param len the length to truncate it to
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> truncate(String path, long len) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.truncate(path, len, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#truncate(String,long)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path the path to the file
   * @param len the length to truncate it to
   * @return the Void instance produced by the operation.
   */
  public Void truncateAndAwait(String path, long len) { 
    return (Void) truncate(path, len).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#truncate(String,long)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#truncate(String,long)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#truncate(String,long)} but you don't need to compose it with other operations.
   * @param path the path to the file
   * @param len the length to truncate it to
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem truncateAndForget(String path, long len) { 
    truncate(path, len).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param len 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem truncateBlocking(String path, long len) { 
    delegate.truncateBlocking(path, len);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.
   * <p>
   * The permission String takes the form rwxr-x--- as
   * specified <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path the path to the file
   * @param perms the permissions string
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> chmod(String path, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.chmod(path, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#chmod(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path the path to the file
   * @param perms the permissions string
   * @return the Void instance produced by the operation.
   */
  public Void chmodAndAwait(String path, String perms) { 
    return (Void) chmod(path, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#chmod(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#chmod(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#chmod(String,String)} but you don't need to compose it with other operations.
   * @param path the path to the file
   * @param perms the permissions string
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem chmodAndForget(String path, String perms) { 
    chmod(path, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param perms 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem chmodBlocking(String path, String perms) { 
    delegate.chmodBlocking(path, perms);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.<p>
   * The permission String takes the form rwxr-x--- as
   * specified in {<a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>}.
   * <p>
   * If the file is directory then all contents will also have their permissions changed recursively. Any directory permissions will
   * be set to <code>dirPerms</code>, whilst any normal file permissions will be set to <code>perms</code>.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path the path to the file
   * @param perms the permissions string
   * @param dirPerms the directory permissions
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> chmodRecursive(String path, String perms, String dirPerms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.chmodRecursive(path, perms, dirPerms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#chmodRecursive(String,String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path the path to the file
   * @param perms the permissions string
   * @param dirPerms the directory permissions
   * @return the Void instance produced by the operation.
   */
  public Void chmodRecursiveAndAwait(String path, String perms, String dirPerms) { 
    return (Void) chmodRecursive(path, perms, dirPerms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#chmodRecursive(String,String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#chmodRecursive(String,String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#chmodRecursive(String,String,String)} but you don't need to compose it with other operations.
   * @param path the path to the file
   * @param perms the permissions string
   * @param dirPerms the directory permissions
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem chmodRecursiveAndForget(String path, String perms, String dirPerms) { 
    chmodRecursive(path, perms, dirPerms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param perms 
   * @param dirPerms 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem chmodRecursiveBlocking(String path, String perms, String dirPerms) { 
    delegate.chmodRecursiveBlocking(path, perms, dirPerms);
    return this;
  }

  /**
   * Change the ownership on the file represented by <code>path</code> to <code>user</code> and {code group}, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path the path to the file
   * @param user the user name, <code>null</code> will not change the user name
   * @param group the user group, <code>null</code> will not change the user group name
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> chown(String path, String user, String group) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.chown(path, user, group, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#chown(String,String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path the path to the file
   * @param user the user name, <code>null</code> will not change the user name
   * @param group the user group, <code>null</code> will not change the user group name
   * @return the Void instance produced by the operation.
   */
  public Void chownAndAwait(String path, String user, String group) { 
    return (Void) chown(path, user, group).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#chown(String,String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#chown(String,String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#chown(String,String,String)} but you don't need to compose it with other operations.
   * @param path the path to the file
   * @param user the user name, <code>null</code> will not change the user name
   * @param group the user group, <code>null</code> will not change the user group name
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem chownAndForget(String path, String user, String group) { 
    chown(path, user, group).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param user 
   * @param group 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem chownBlocking(String path, String user, String group) { 
    delegate.chownBlocking(path, user, group);
    return this;
  }

  /**
   * Obtain properties for the file represented by <code>path</code>, asynchronously.
   * <p>
   * If the file is a link, the link will be followed.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path the path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.file.FileProps> props(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.props(path, new Handler<AsyncResult<io.vertx.core.file.FileProps>>() {
      public void handle(AsyncResult<io.vertx.core.file.FileProps> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.mutiny.core.file.FileProps.newInstance((io.vertx.core.file.FileProps)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#props(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path the path to the file
   * @return the FileProps instance produced by the operation.
   */
  public io.vertx.mutiny.core.file.FileProps propsAndAwait(String path) { 
    return (io.vertx.mutiny.core.file.FileProps) props(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#props(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#props(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#props(String)} but you don't need to compose it with other operations.
   * @param path the path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem propsAndForget(String path) { 
    props(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return 
   */
  public io.vertx.mutiny.core.file.FileProps propsBlocking(String path) { 
    io.vertx.mutiny.core.file.FileProps ret = io.vertx.mutiny.core.file.FileProps.newInstance((io.vertx.core.file.FileProps)delegate.propsBlocking(path));
    return ret;
  }

  /**
   * Obtain properties for the link represented by <code>path</code>, asynchronously.
   * <p>
   * The link will not be followed.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path the path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.file.FileProps> lprops(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.lprops(path, new Handler<AsyncResult<io.vertx.core.file.FileProps>>() {
      public void handle(AsyncResult<io.vertx.core.file.FileProps> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.mutiny.core.file.FileProps.newInstance((io.vertx.core.file.FileProps)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#lprops(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path the path to the file
   * @return the FileProps instance produced by the operation.
   */
  public io.vertx.mutiny.core.file.FileProps lpropsAndAwait(String path) { 
    return (io.vertx.mutiny.core.file.FileProps) lprops(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#lprops(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#lprops(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#lprops(String)} but you don't need to compose it with other operations.
   * @param path the path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem lpropsAndForget(String path) { 
    lprops(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return 
   */
  public io.vertx.mutiny.core.file.FileProps lpropsBlocking(String path) { 
    io.vertx.mutiny.core.file.FileProps ret = io.vertx.mutiny.core.file.FileProps.newInstance((io.vertx.core.file.FileProps)delegate.lpropsBlocking(path));
    return ret;
  }

  /**
   * Create a hard link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param link the link
   * @param existing the link destination
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> link(String link, String existing) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.link(link, existing, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#link(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param link the link
   * @param existing the link destination
   * @return the Void instance produced by the operation.
   */
  public Void linkAndAwait(String link, String existing) { 
    return (Void) link(link, existing).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#link(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#link(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#link(String,String)} but you don't need to compose it with other operations.
   * @param link the link
   * @param existing the link destination
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem linkAndForget(String link, String existing) { 
    link(link, existing).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param link 
   * @param existing 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem linkBlocking(String link, String existing) { 
    delegate.linkBlocking(link, existing);
    return this;
  }

  /**
   * Create a symbolic link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param link the link
   * @param existing the link destination
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> symlink(String link, String existing) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.symlink(link, existing, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#symlink(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param link the link
   * @param existing the link destination
   * @return the Void instance produced by the operation.
   */
  public Void symlinkAndAwait(String link, String existing) { 
    return (Void) symlink(link, existing).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#symlink(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#symlink(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#symlink(String,String)} but you don't need to compose it with other operations.
   * @param link the link
   * @param existing the link destination
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem symlinkAndForget(String link, String existing) { 
    symlink(link, existing).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param link 
   * @param existing 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem symlinkBlocking(String link, String existing) { 
    delegate.symlinkBlocking(link, existing);
    return this;
  }

  /**
   * Unlinks the link on the file system represented by the path <code>link</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param link the link
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> unlink(String link) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.unlink(link, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#unlink(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param link the link
   * @return the Void instance produced by the operation.
   */
  public Void unlinkAndAwait(String link) { 
    return (Void) unlink(link).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#unlink(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#unlink(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#unlink(String)} but you don't need to compose it with other operations.
   * @param link the link
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem unlinkAndForget(String link) { 
    unlink(link).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param link 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem unlinkBlocking(String link) { 
    delegate.unlinkBlocking(link);
    return this;
  }

  /**
   * Returns the path representing the file that the symbolic link specified by <code>link</code> points to, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param link the link
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<String> readSymlink(String link) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.readSymlink(link, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#readSymlink(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param link the link
   * @return the String instance produced by the operation.
   */
  public String readSymlinkAndAwait(String link) { 
    return (String) readSymlink(link).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#readSymlink(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#readSymlink(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#readSymlink(String)} but you don't need to compose it with other operations.
   * @param link the link
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem readSymlinkAndForget(String link) { 
    readSymlink(link).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param link 
   * @return 
   */
  public String readSymlinkBlocking(String link) { 
    String ret = delegate.readSymlinkBlocking(link);
    return ret;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> delete(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.delete(path, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#delete(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @return the Void instance produced by the operation.
   */
  public Void deleteAndAwait(String path) { 
    return (Void) delete(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#delete(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#delete(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#delete(String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem deleteAndForget(String path) { 
    delete(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem deleteBlocking(String path) { 
    delegate.deleteBlocking(path);
    return this;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * <p>
   * If the path represents a directory and <code>recursive = true</code> then the directory and its contents will be
   * deleted recursively.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @param recursive delete recursively?
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> deleteRecursive(String path, boolean recursive) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.deleteRecursive(path, recursive, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#deleteRecursive(String,boolean)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @param recursive delete recursively?
   * @return the Void instance produced by the operation.
   */
  public Void deleteRecursiveAndAwait(String path, boolean recursive) { 
    return (Void) deleteRecursive(path, recursive).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#deleteRecursive(String,boolean)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#deleteRecursive(String,boolean)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#deleteRecursive(String,boolean)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @param recursive delete recursively?
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem deleteRecursiveAndForget(String path, boolean recursive) { 
    deleteRecursive(path, recursive).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param recursive 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem deleteRecursiveBlocking(String path, boolean recursive) { 
    delegate.deleteRecursiveBlocking(path, recursive);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The operation will fail if the directory already exists.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> mkdir(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.mkdir(path, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @return the Void instance produced by the operation.
   */
  public Void mkdirAndAwait(String path) { 
    return (Void) mkdir(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirAndForget(String path) { 
    mkdir(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirBlocking(String path) { 
    delegate.mkdirBlocking(path);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the directory already exists.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @param perms the permissions string
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> mkdir(String path, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.mkdir(path, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @param perms the permissions string
   * @return the Void instance produced by the operation.
   */
  public Void mkdirAndAwait(String path, String perms) { 
    return (Void) mkdir(path, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#mkdir(String,String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @param perms the permissions string
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirAndForget(String path, String perms) { 
    mkdir(path, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param perms 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirBlocking(String path, String perms) { 
    delegate.mkdirBlocking(path, perms);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> mkdirs(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.mkdirs(path, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @return the Void instance produced by the operation.
   */
  public Void mkdirsAndAwait(String path) { 
    return (Void) mkdirs(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirsAndForget(String path) { 
    mkdirs(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirsBlocking(String path) { 
    delegate.mkdirsBlocking(path);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @param perms the permissions string
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> mkdirs(String path, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.mkdirs(path, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @param perms the permissions string
   * @return the Void instance produced by the operation.
   */
  public Void mkdirsAndAwait(String path, String perms) { 
    return (Void) mkdirs(path, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#mkdirs(String,String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @param perms the permissions string
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirsAndForget(String path, String perms) { 
    mkdirs(path, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param perms 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem mkdirsBlocking(String path, String perms) { 
    delegate.mkdirsBlocking(path, perms);
    return this;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<List<String>> readDir(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.readDir(path, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#readDir(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @return the List<String> instance produced by the operation.
   */
  public List<String> readDirAndAwait(String path) { 
    return (List<String>) readDir(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#readDir(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#readDir(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#readDir(String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem readDirAndForget(String path) { 
    readDir(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return 
   */
  public List<String> readDirBlocking(String path) { 
    List<String> ret = delegate.readDirBlocking(path);
    return ret;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The parameter <code>filter</code> is a regular expression. If <code>filter</code> is specified then only the paths that
   * match  @{filter}will be returned.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the directory
   * @param filter the filter expression
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<List<String>> readDir(String path, String filter) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.readDir(path, filter, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#readDir(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the directory
   * @param filter the filter expression
   * @return the List<String> instance produced by the operation.
   */
  public List<String> readDirAndAwait(String path, String filter) { 
    return (List<String>) readDir(path, filter).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#readDir(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#readDir(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#readDir(String,String)} but you don't need to compose it with other operations.
   * @param path path to the directory
   * @param filter the filter expression
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem readDirAndForget(String path, String filter) { 
    readDir(path, filter).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param filter 
   * @return 
   */
  public List<String> readDirBlocking(String path, String filter) { 
    List<String> ret = delegate.readDirBlocking(path, filter);
    return ret;
  }

  /**
   * Reads the entire file as represented by the path <code>path</code> as a , asynchronously.
   * <p>
   * Do not use this method to read very large files or you risk running out of available RAM.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.buffer.Buffer> readFile(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.readFile(path, new Handler<AsyncResult<io.vertx.core.buffer.Buffer>>() {
      public void handle(AsyncResult<io.vertx.core.buffer.Buffer> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.mutiny.core.buffer.Buffer.newInstance((io.vertx.core.buffer.Buffer)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#readFile(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @return the Buffer instance produced by the operation.
   */
  public io.vertx.mutiny.core.buffer.Buffer readFileAndAwait(String path) { 
    return (io.vertx.mutiny.core.buffer.Buffer) readFile(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#readFile(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#readFile(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#readFile(String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem readFileAndForget(String path) { 
    readFile(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return 
   */
  public io.vertx.mutiny.core.buffer.Buffer readFileBlocking(String path) { 
    io.vertx.mutiny.core.buffer.Buffer ret = io.vertx.mutiny.core.buffer.Buffer.newInstance((io.vertx.core.buffer.Buffer)delegate.readFileBlocking(path));
    return ret;
  }

  /**
   * Creates the file, and writes the specified <code>Buffer data</code> to the file represented by the path <code>path</code>,
   * asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @param data 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> writeFile(String path, io.vertx.mutiny.core.buffer.Buffer data) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.writeFile(path, data.getDelegate(), handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#writeFile(String,io.vertx.mutiny.core.buffer.Buffer)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @param data 
   * @return the Void instance produced by the operation.
   */
  public Void writeFileAndAwait(String path, io.vertx.mutiny.core.buffer.Buffer data) { 
    return (Void) writeFile(path, data).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#writeFile(String,io.vertx.mutiny.core.buffer.Buffer)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#writeFile(String,io.vertx.mutiny.core.buffer.Buffer)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#writeFile(String,io.vertx.mutiny.core.buffer.Buffer)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @param data 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem writeFileAndForget(String path, io.vertx.mutiny.core.buffer.Buffer data) { 
    writeFile(path, data).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param data 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem writeFileBlocking(String path, io.vertx.mutiny.core.buffer.Buffer data) { 
    delegate.writeFileBlocking(path, data.getDelegate());
    return this;
  }

  /**
   * Open the file represented by <code>path</code>, asynchronously.
   * <p>
   * The file is opened for both reading and writing. If the file does not already exist it will be created.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @param options options describing how the file should be opened
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.file.AsyncFile> open(String path, io.vertx.core.file.OpenOptions options) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.open(path, options, new Handler<AsyncResult<io.vertx.core.file.AsyncFile>>() {
      public void handle(AsyncResult<io.vertx.core.file.AsyncFile> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.mutiny.core.file.AsyncFile.newInstance((io.vertx.core.file.AsyncFile)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#open(String,OpenOptions)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @param options options describing how the file should be opened
   * @return the AsyncFile instance produced by the operation.
   */
  public io.vertx.mutiny.core.file.AsyncFile openAndAwait(String path, io.vertx.core.file.OpenOptions options) { 
    return (io.vertx.mutiny.core.file.AsyncFile) open(path, options).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#open(String,OpenOptions)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#open(String,OpenOptions)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#open(String,OpenOptions)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @param options options describing how the file should be opened
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem openAndForget(String path, io.vertx.core.file.OpenOptions options) { 
    open(path, options).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param options 
   * @return 
   */
  public io.vertx.mutiny.core.file.AsyncFile openBlocking(String path, io.vertx.core.file.OpenOptions options) { 
    io.vertx.mutiny.core.file.AsyncFile ret = io.vertx.mutiny.core.file.AsyncFile.newInstance((io.vertx.core.file.AsyncFile)delegate.openBlocking(path, options));
    return ret;
  }

  /**
   * Creates an empty file with the specified <code>path</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> createFile(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createFile(path, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createFile(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @return the Void instance produced by the operation.
   */
  public Void createFileAndAwait(String path) { 
    return (Void) createFile(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createFile(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createFile(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createFile(String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createFileAndForget(String path) { 
    createFile(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createFileBlocking(String path) { 
    delegate.createFileBlocking(path);
    return this;
  }

  /**
   * Creates an empty file with the specified <code>path</code> and permissions <code>perms</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @param perms the permissions string
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> createFile(String path, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createFile(path, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createFile(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @param perms the permissions string
   * @return the Void instance produced by the operation.
   */
  public Void createFileAndAwait(String path, String perms) { 
    return (Void) createFile(path, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createFile(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createFile(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createFile(String,String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @param perms the permissions string
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createFileAndForget(String path, String perms) { 
    createFile(path, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @param perms 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createFileBlocking(String path, String perms) { 
    delegate.createFileBlocking(path, perms);
    return this;
  }

  /**
   * Determines whether the file as specified by the path <code>path</code> exists, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to the file
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Boolean> exists(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.exists(path, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#exists(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to the file
   * @return the Boolean instance produced by the operation.
   */
  public Boolean existsAndAwait(String path) { 
    return (Boolean) exists(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#exists(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#exists(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#exists(String)} but you don't need to compose it with other operations.
   * @param path path to the file
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem existsAndForget(String path) { 
    exists(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return 
   */
  public boolean existsBlocking(String path) { 
    boolean ret = delegate.existsBlocking(path);
    return ret;
  }

  /**
   * Returns properties of the file-system being used by the specified <code>path</code>, asynchronously.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param path path to anywhere on the filesystem
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.file.FileSystemProps> fsProps(String path) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.fsProps(path, new Handler<AsyncResult<io.vertx.core.file.FileSystemProps>>() {
      public void handle(AsyncResult<io.vertx.core.file.FileSystemProps> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.mutiny.core.file.FileSystemProps.newInstance((io.vertx.core.file.FileSystemProps)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#fsProps(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param path path to anywhere on the filesystem
   * @return the FileSystemProps instance produced by the operation.
   */
  public io.vertx.mutiny.core.file.FileSystemProps fsPropsAndAwait(String path) { 
    return (io.vertx.mutiny.core.file.FileSystemProps) fsProps(path).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#fsProps(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#fsProps(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#fsProps(String)} but you don't need to compose it with other operations.
   * @param path path to anywhere on the filesystem
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem fsPropsAndForget(String path) { 
    fsProps(path).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param path 
   * @return 
   */
  public io.vertx.mutiny.core.file.FileSystemProps fsPropsBlocking(String path) { 
    io.vertx.mutiny.core.file.FileSystemProps ret = io.vertx.mutiny.core.file.FileSystemProps.newInstance((io.vertx.core.file.FileSystemProps)delegate.fsPropsBlocking(path));
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<String> createTempDirectory(String prefix) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createTempDirectory(prefix, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @return the String instance produced by the operation.
   */
  public String createTempDirectoryAndAwait(String prefix) { 
    return (String) createTempDirectory(prefix).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String)} but you don't need to compose it with other operations.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createTempDirectoryAndForget(String prefix) { 
    createTempDirectory(prefix).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param prefix 
   * @return 
   */
  public String createTempDirectoryBlocking(String prefix) { 
    String ret = delegate.createTempDirectoryBlocking(prefix);
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<String> createTempDirectory(String prefix, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createTempDirectory(prefix, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return the String instance produced by the operation.
   */
  public String createTempDirectoryAndAwait(String prefix, String perms) { 
    return (String) createTempDirectory(prefix, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String)} but you don't need to compose it with other operations.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createTempDirectoryAndForget(String prefix, String perms) { 
    createTempDirectory(prefix, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param prefix 
   * @param perms 
   * @return 
   */
  public String createTempDirectoryBlocking(String prefix, String perms) { 
    String ret = delegate.createTempDirectoryBlocking(prefix, perms);
    return ret;
  }

  /**
   * Creates a new directory in the directory provided by the path <code>path</code>, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<String> createTempDirectory(String dir, String prefix, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createTempDirectory(dir, prefix, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return the String instance produced by the operation.
   */
  public String createTempDirectoryAndAwait(String dir, String prefix, String perms) { 
    return (String) createTempDirectory(dir, prefix, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createTempDirectory(String,String,String)} but you don't need to compose it with other operations.
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createTempDirectoryAndForget(String dir, String prefix, String perms) { 
    createTempDirectory(dir, prefix, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param dir 
   * @param prefix 
   * @param perms 
   * @return 
   */
  public String createTempDirectoryBlocking(String dir, String prefix, String perms) { 
    String ret = delegate.createTempDirectoryBlocking(dir, prefix, perms);
    return ret;
  }

  /**
   * Creates a new file in the default temporary-file directory, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<String> createTempFile(String prefix, String suffix) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createTempFile(prefix, suffix, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @return the String instance produced by the operation.
   */
  public String createTempFileAndAwait(String prefix, String suffix) { 
    return (String) createTempFile(prefix, suffix).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String)} but you don't need to compose it with other operations.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createTempFileAndForget(String prefix, String suffix) { 
    createTempFile(prefix, suffix).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param prefix 
   * @param suffix 
   * @return 
   */
  public String createTempFileBlocking(String prefix, String suffix) { 
    String ret = delegate.createTempFileBlocking(prefix, suffix);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<String> createTempFile(String prefix, String suffix, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createTempFile(prefix, suffix, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms 
   * @return the String instance produced by the operation.
   */
  public String createTempFileAndAwait(String prefix, String suffix, String perms) { 
    return (String) createTempFile(prefix, suffix, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String)} but you don't need to compose it with other operations.
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms 
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createTempFileAndForget(String prefix, String suffix, String perms) { 
    createTempFile(prefix, suffix, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param prefix 
   * @param suffix 
   * @param perms 
   * @return 
   */
  public String createTempFileBlocking(String prefix, String suffix, String perms) { 
    String ret = delegate.createTempFileBlocking(prefix, suffix, perms);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms the permissions string
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<String> createTempFile(String dir, String prefix, String suffix, String perms) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.createTempFile(dir, prefix, suffix, perms, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms the permissions string
   * @return the String instance produced by the operation.
   */
  public String createTempFileAndAwait(String dir, String prefix, String suffix, String perms) { 
    return (String) createTempFile(dir, prefix, suffix, perms).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.file.FileSystem#createTempFile(String,String,String,String)} but you don't need to compose it with other operations.
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms the permissions string
   * @return the instance of FileSystem to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.file.FileSystem createTempFileAndForget(String dir, String prefix, String suffix, String perms) { 
    createTempFile(dir, prefix, suffix, perms).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * @param dir 
   * @param prefix 
   * @param suffix 
   * @param perms 
   * @return 
   */
  public String createTempFileBlocking(String dir, String prefix, String suffix, String perms) { 
    String ret = delegate.createTempFileBlocking(dir, prefix, suffix, perms);
    return ret;
  }

  public static  FileSystem newInstance(io.vertx.core.file.FileSystem arg) {
    return arg != null ? new FileSystem(arg) : null;
  }

}
