/*
 * Decompiled with CFR 0.152.
 */
package io.sentry.core.cache;

import io.sentry.core.ILogger;
import io.sentry.core.ISerializer;
import io.sentry.core.SentryEvent;
import io.sentry.core.SentryLevel;
import io.sentry.core.SentryOptions;
import io.sentry.core.cache.IEventCache;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import org.jetbrains.annotations.NotNull;

public final class DiskCache
implements IEventCache {
    public static final String FILE_SUFFIX = ".sentry-event";
    private static final Charset UTF8 = Charset.forName("UTF-8");
    private final File directory;
    private final int maxSize;
    private final ISerializer serializer;
    private final SentryOptions options;

    public DiskCache(SentryOptions options) {
        this.directory = new File(options.getCacheDirPath());
        this.maxSize = options.getCacheDirSize();
        this.serializer = options.getSerializer();
        this.options = options;
    }

    @Override
    public void store(SentryEvent event) {
        if (this.getNumberOfStoredEvents() >= this.maxSize) {
            ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.WARNING, "Disk cache full (respecting maxSize). Not storing event {}", event);
            return;
        }
        File eventFile = this.getEventFile(event);
        if (eventFile.exists()) {
            ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.WARNING, "Not adding Event to offline storage because it already exists: %s", eventFile.getAbsolutePath());
            return;
        }
        ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.DEBUG, "Adding Event to offline storage: %s", eventFile.getAbsolutePath());
        try (FileOutputStream fileOutputStream = new FileOutputStream(eventFile);
             OutputStreamWriter wrt = new OutputStreamWriter((OutputStream)fileOutputStream, UTF8);){
            this.serializer.serialize(event, wrt);
        }
        catch (Exception e) {
            ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.ERROR, "Error writing Event to offline storage: %s", event.getEventId());
        }
    }

    @Override
    public void discard(SentryEvent event) {
        File eventFile = this.getEventFile(event);
        if (eventFile.exists()) {
            ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.DEBUG, "Discarding event from cache: %s", eventFile.getAbsolutePath());
            if (!eventFile.delete()) {
                ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.ERROR, "Failed to delete Event: %s", eventFile.getAbsolutePath());
            }
        } else {
            ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.DEBUG, "Event was not cached: %s", eventFile.getAbsolutePath());
        }
    }

    private int getNumberOfStoredEvents() {
        return this.allEventFiles().length;
    }

    private boolean isDirectoryValid() {
        if (!(this.directory.isDirectory() && this.directory.canWrite() && this.directory.canRead())) {
            ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.ERROR, "The directory for caching Sentry events is inaccessible.: %s", this.directory.getAbsolutePath());
            return false;
        }
        return true;
    }

    private File getEventFile(SentryEvent event) {
        return new File(this.directory.getAbsolutePath(), event.getEventId().toString() + FILE_SUFFIX);
    }

    @Override
    @NotNull
    public Iterator<SentryEvent> iterator() {
        File[] allCachedEvents = this.allEventFiles();
        ArrayList<SentryEvent> ret = new ArrayList<SentryEvent>(allCachedEvents.length);
        for (File f : allCachedEvents) {
            try (InputStreamReader rdr = new InputStreamReader((InputStream)new BufferedInputStream(new FileInputStream(f)), UTF8);){
                ret.add(this.serializer.deserializeEvent(rdr));
            }
            catch (FileNotFoundException e) {
                ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.DEBUG, "Event file '%s' disappeared while converting all cached files to events.", f.getAbsolutePath());
            }
            catch (IOException e) {
                ILogger.logIfNotNull(this.options.getLogger(), SentryLevel.ERROR, String.format("Error while reading cached event from file %s", f.getAbsolutePath()), e);
            }
        }
        return ret.iterator();
    }

    private File[] allEventFiles() {
        if (this.isDirectoryValid()) {
            return this.directory.listFiles((__, fileName) -> fileName.endsWith(FILE_SUFFIX));
        }
        return new File[0];
    }
}

