/*
 * Decompiled with CFR 0.152.
 */
package io.r2dbc.pool;

import io.r2dbc.pool.Assert;
import io.r2dbc.pool.ConnectionPoolConfiguration;
import io.r2dbc.pool.ConnectionPoolException;
import io.r2dbc.pool.ConnectionPoolMXBean;
import io.r2dbc.pool.PoolMetrics;
import io.r2dbc.pool.PooledConnection;
import io.r2dbc.pool.Validation;
import io.r2dbc.spi.Closeable;
import io.r2dbc.spi.Connection;
import io.r2dbc.spi.ConnectionFactory;
import io.r2dbc.spi.ConnectionFactoryMetadata;
import io.r2dbc.spi.Lifecycle;
import io.r2dbc.spi.R2dbcTimeoutException;
import io.r2dbc.spi.Wrapped;
import java.lang.management.ManagementFactory;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Function;
import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import org.reactivestreams.Publisher;
import reactor.core.Disposable;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.pool.InstrumentedPool;
import reactor.pool.PoolBuilder;
import reactor.pool.PoolConfig;
import reactor.pool.PoolMetricsRecorder;
import reactor.pool.PooledRef;
import reactor.pool.PooledRefMetadata;
import reactor.util.Logger;
import reactor.util.Loggers;
import reactor.util.annotation.Nullable;

public class ConnectionPool
implements ConnectionFactory,
Disposable,
Closeable,
Wrapped<ConnectionFactory> {
    private static final Logger logger = Loggers.getLogger(ConnectionPool.class);
    private final ConnectionFactory factory;
    private final InstrumentedPool<Connection> connectionPool;
    private final Duration maxAcquireTime;
    private final List<Runnable> destroyHandlers = new ArrayList<Runnable>();
    private final Optional<PoolMetrics> poolMetrics;
    private final Mono<Connection> create;
    @Nullable
    private final Function<? super Connection, ? extends Publisher<Void>> preRelease;

    public ConnectionPool(ConnectionPoolConfiguration configuration) {
        this.connectionPool = this.createConnectionPool(Assert.requireNonNull(configuration, "ConnectionPoolConfiguration must not be null"));
        this.factory = configuration.getConnectionFactory();
        this.maxAcquireTime = configuration.getMaxAcquireTime();
        this.poolMetrics = Optional.ofNullable(this.connectionPool.metrics()).map(x$0 -> new PoolMetricsWrapper((InstrumentedPool.PoolMetrics)x$0));
        this.preRelease = configuration.getPreRelease();
        if (configuration.isRegisterJmx()) {
            this.getMetrics().ifPresent(poolMetrics -> this.registerToJmx((PoolMetrics)poolMetrics, configuration.getName()));
        }
        String acqName = String.format("Connection Acquisition from [%s]", configuration.getConnectionFactory());
        String timeoutMessage = String.format("Connection Acquisition timed out after %dms", this.maxAcquireTime.toMillis());
        Function<Connection, Mono<Void>> allocateValidation = this.getValidationFunction(configuration);
        Mono create = Mono.defer(() -> {
            AtomicReference emitted = new AtomicReference();
            Mono mono = this.connectionPool.acquire().doOnNext(emitted::set).doOnSubscribe(subscription -> {
                if (logger.isDebugEnabled()) {
                    logger.debug("Obtaining new connection from the driver");
                }
            }).flatMap(ref -> {
                Mono prepare = null;
                if (ref.poolable() instanceof Lifecycle) {
                    prepare = Mono.from((Publisher)((Lifecycle)ref.poolable()).postAllocate());
                }
                if (configuration.getPostAllocate() != null) {
                    Mono postAllocate = Mono.defer(() -> Mono.from(configuration.getPostAllocate().apply((Connection)ref.poolable())));
                    prepare = prepare == null ? postAllocate : prepare.then(postAllocate);
                }
                return prepare == null ? Mono.just((Object)ref) : prepare.thenReturn(ref).onErrorResume(throwable -> ref.invalidate().then(Mono.error((Throwable)throwable)));
            }).flatMap(ref -> {
                PooledConnection connection = new PooledConnection((PooledRef<Connection>)ref, this.preRelease);
                return ((Mono)allocateValidation.apply(connection)).thenReturn((Object)connection).onErrorResume(throwable -> ref.invalidate().then(Mono.error((Throwable)throwable)));
            }).doOnCancel(() -> {
                PooledRef ref = (PooledRef)emitted.get();
                if (ref != null && emitted.compareAndSet(ref, null)) {
                    ref.release().subscribe();
                }
            }).name(acqName);
            if (!this.maxAcquireTime.isNegative()) {
                mono = mono.timeout(this.maxAcquireTime).onErrorMap(TimeoutException.class, e -> new R2dbcTimeoutException(timeoutMessage, (Throwable)e));
            }
            return mono;
        });
        this.create = configuration.getAcquireRetry() > 0 ? create.retry((long)configuration.getAcquireRetry()) : create;
    }

    private Function<Connection, Mono<Void>> getValidationFunction(ConnectionPoolConfiguration configuration) {
        String timeoutMessage = String.format("Validation timed out after %dms", this.maxAcquireTime.toMillis());
        if (!this.maxAcquireTime.isNegative()) {
            return this.getValidation(configuration).andThen(mono -> mono.timeout(this.maxAcquireTime).onErrorMap(TimeoutException.class, e -> new R2dbcTimeoutException(timeoutMessage, (Throwable)e)));
        }
        return this.getValidation(configuration);
    }

    private Function<Connection, Mono<Void>> getValidation(ConnectionPoolConfiguration configuration) {
        String validationQuery = configuration.getValidationQuery();
        if (validationQuery != null && !validationQuery.isEmpty()) {
            return connection -> Validation.validate(connection, validationQuery);
        }
        return connection -> Validation.validate(connection, configuration.getValidationDepth());
    }

    public Mono<Integer> warmup() {
        return this.connectionPool.warmup();
    }

    private InstrumentedPool<Connection> createConnectionPool(ConnectionPoolConfiguration configuration) {
        ConnectionFactory factory = configuration.getConnectionFactory();
        Duration maxCreateConnectionTime = configuration.getMaxCreateConnectionTime();
        int initialSize = configuration.getInitialSize();
        int maxSize = configuration.getMaxSize();
        Duration maxIdleTime = configuration.getMaxIdleTime();
        Duration maxLifeTime = configuration.getMaxLifeTime();
        Consumer<PoolBuilder<Connection, ? extends PoolConfig<? extends Connection>>> customizer = configuration.getCustomizer();
        PoolMetricsRecorder metricsRecorder = configuration.getMetricsRecorder();
        if (factory instanceof ConnectionPool) {
            Loggers.getLogger(ConnectionPool.class).warn(String.format("Creating ConnectionPool using another ConnectionPool [%s] as ConnectionFactory", factory));
        }
        Mono allocator = Mono.from((Publisher)factory.create()).name("Connection Allocation");
        if (!maxCreateConnectionTime.isNegative()) {
            allocator = allocator.timeout(maxCreateConnectionTime);
        }
        BiPredicate<Connection, PooledRefMetadata> evictionPredicate = (connection, metadata) -> {
            long maxIdleTimeMills = maxIdleTime.toMillis();
            long maxLifeTimeMillis = maxLifeTime.toMillis();
            if (maxIdleTimeMills == 0L || maxLifeTimeMillis == 0L) {
                return true;
            }
            boolean isIdleTimeExceeded = maxIdleTimeMills > 0L && metadata.idleTime() >= maxIdleTimeMills;
            boolean isLifeTimeExceeded = maxLifeTimeMillis > 0L && metadata.lifeTime() >= maxLifeTimeMillis;
            return isIdleTimeExceeded || isLifeTimeExceeded;
        };
        PoolBuilder builder = PoolBuilder.from((Publisher)allocator).clock(configuration.getClock()).metricsRecorder(metricsRecorder).evictionPredicate(evictionPredicate).destroyHandler(Connection::close).sizeBetween(0, Runtime.getRuntime().availableProcessors());
        if (maxSize == -1 || initialSize > 0) {
            builder.sizeBetween(Math.max(0, initialSize), maxSize == -1 ? Integer.MAX_VALUE : maxSize);
        } else {
            builder.sizeBetween(initialSize, maxSize);
        }
        Duration backgroundEvictionInterval = configuration.getBackgroundEvictionInterval();
        if (!backgroundEvictionInterval.isZero()) {
            if (!backgroundEvictionInterval.isNegative()) {
                builder.evictInBackground(backgroundEvictionInterval);
            } else if (!configuration.getMaxIdleTime().isNegative()) {
                builder.evictInBackground(configuration.getMaxIdleTime());
            }
        }
        customizer.accept((PoolBuilder<Connection, ? extends PoolConfig<? extends Connection>>)builder);
        return builder.buildPool();
    }

    public Mono<Connection> create() {
        return this.create;
    }

    public Mono<Void> close() {
        return this.disposeLater();
    }

    public void dispose() {
        this.disposeLater().block();
    }

    public Mono<Void> disposeLater() {
        if (this.isDisposed()) {
            return Mono.empty();
        }
        ArrayList errors = new ArrayList();
        return Flux.fromIterable(this.destroyHandlers).flatMap(Mono::fromRunnable).concatWith((Publisher)this.connectionPool.disposeLater()).onErrorContinue((throwable, o) -> errors.add(throwable)).then(Mono.defer(() -> {
            if (errors.isEmpty()) {
                return Mono.empty();
            }
            Throwable rootError = (Throwable)errors.get(0);
            if (errors.size() == 1) {
                return Mono.error((Throwable)rootError);
            }
            errors.subList(1, errors.size()).forEach(rootError::addSuppressed);
            return Mono.error((Throwable)rootError);
        }));
    }

    public boolean isDisposed() {
        return this.connectionPool.isDisposed();
    }

    public ConnectionFactoryMetadata getMetadata() {
        return this.factory.getMetadata();
    }

    public ConnectionFactory unwrap() {
        return this.factory;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(this.getClass().getSimpleName());
        sb.append("[");
        sb.append(this.factory.getMetadata().getName());
        sb.append("]");
        return sb.toString();
    }

    public Optional<PoolMetrics> getMetrics() {
        return this.poolMetrics;
    }

    private void registerToJmx(PoolMetrics poolMetrics, String name) {
        MBeanServer mBeanServer = ManagementFactory.getPlatformMBeanServer();
        try {
            ObjectName jmxName = this.getPoolObjectName(name);
            mBeanServer.registerMBean(new ConnectionPoolMXBeanImpl(poolMetrics), jmxName);
            this.destroyHandlers.add(() -> {
                try {
                    mBeanServer.unregisterMBean(jmxName);
                }
                catch (JMException e) {
                    throw new ConnectionPoolException("Failed to unregister from JMX", e);
                }
            });
        }
        catch (JMException e) {
            throw new ConnectionPoolException("Failed to register to JMX", e);
        }
    }

    protected ObjectName getPoolObjectName(String name) throws MalformedObjectNameException {
        Hashtable<String, String> prop = new Hashtable<String, String>();
        prop.put("type", ConnectionPool.class.getSimpleName());
        prop.put("name", name);
        return new ObjectName("io.r2dbc.pool", prop);
    }

    private class ConnectionPoolMXBeanImpl
    implements ConnectionPoolMXBean {
        private final PoolMetrics poolMetrics;

        ConnectionPoolMXBeanImpl(PoolMetrics poolMetrics) {
            this.poolMetrics = poolMetrics;
        }

        @Override
        public int getAcquiredSize() {
            return this.poolMetrics.acquiredSize();
        }

        @Override
        public int getAllocatedSize() {
            return this.poolMetrics.allocatedSize();
        }

        @Override
        public int getIdleSize() {
            return this.poolMetrics.idleSize();
        }

        @Override
        public int getPendingAcquireSize() {
            return this.poolMetrics.pendingAcquireSize();
        }

        @Override
        public int getMaxAllocatedSize() {
            return this.poolMetrics.getMaxAllocatedSize();
        }

        @Override
        public int getMaxPendingAcquireSize() {
            return this.poolMetrics.getMaxPendingAcquireSize();
        }
    }

    private class PoolMetricsWrapper
    implements PoolMetrics {
        private final InstrumentedPool.PoolMetrics delegate;

        PoolMetricsWrapper(InstrumentedPool.PoolMetrics delegate) {
            this.delegate = delegate;
        }

        @Override
        public int acquiredSize() {
            return this.delegate.acquiredSize();
        }

        @Override
        public int allocatedSize() {
            return this.delegate.allocatedSize();
        }

        @Override
        public int idleSize() {
            return this.delegate.idleSize();
        }

        @Override
        public int pendingAcquireSize() {
            return this.delegate.pendingAcquireSize();
        }

        @Override
        public int getMaxAllocatedSize() {
            return this.delegate.getMaxAllocatedSize();
        }

        @Override
        public int getMaxPendingAcquireSize() {
            return this.delegate.getMaxPendingAcquireSize();
        }
    }
}

