/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.docs.generation;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.exc.StreamReadException;
import com.fasterxml.jackson.databind.DatabindException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import io.quarkus.docs.generation.ReferenceIndexGenerator;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class CheckCrossReferences {
    private static final String SOURCE_BLOCK_PREFIX = "[source";
    private static final String SOURCE_BLOCK_DELIMITER = "--";
    private static final Pattern XREF_PATTERN = Pattern.compile("xref:([^\\[]+)\\[[^\\]]*\\]");
    private static final Pattern ANGLE_BRACKETS_WITHOUT_DESCRIPTION_PATTERN = Pattern.compile("<<([a-z0-9_\\-#\\.]+?)>>", 2);
    private static final Pattern ANGLE_BRACKETS_WITH_DESCRIPTION_PATTERN = Pattern.compile("<<([a-z0-9_\\-#\\.]+?),([^>]+?)>>", 2);
    private static final Set<String> IGNORED_GUIDES = Set.of("deploying-to-kubernetes.adoc", "deploying-to-openshift.adoc");
    private final Path srcDir;
    private final ReferenceIndexGenerator.Index referenceIndex;

    public static void main(String[] args) throws Exception {
        CheckCrossReferences checker = new CheckCrossReferences(args.length >= 1 ? Path.of(args[0], new String[0]) : CheckCrossReferences.docsDir().resolve("src/main/asciidoc"), args.length >= 2 ? Path.of(args[1], new String[0]) : CheckCrossReferences.docsDir().resolve("target/referenceIndex.yaml"));
        System.out.println("[INFO] Checking cross references using: " + args[0]);
        Map<String, List<String>> errors = checker.check();
        if (!errors.isEmpty()) {
            StringBuffer errorLog = new StringBuffer("Unable to find cross reference for:\n\n");
            for (Map.Entry<String, List<String>> errorEntry : errors.entrySet()) {
                errorLog.append("- " + errorEntry.getKey() + "\n");
                for (String error : errorEntry.getValue()) {
                    errorLog.append("    . " + error + "\n");
                }
            }
            errorLog.append("See https://quarkus.io/guides/doc-reference#cross-references");
            throw new IllegalStateException(errorLog.toString());
        }
        System.out.println("[INFO] Done");
    }

    public CheckCrossReferences(Path srcDir, Path referenceIndexPath) throws StreamReadException, DatabindException, IOException {
        if (!Files.exists(srcDir, new LinkOption[0]) || !Files.isDirectory(srcDir, new LinkOption[0])) {
            throw new IllegalStateException(String.format("Source directory (%s) does not exist", srcDir.toAbsolutePath()));
        }
        this.srcDir = srcDir;
        if (!Files.exists(referenceIndexPath, new LinkOption[0]) || !Files.isReadable(referenceIndexPath)) {
            throw new IllegalStateException(String.format("Reference index %s does not exist or is not readable", referenceIndexPath.toAbsolutePath()));
        }
        ObjectMapper om = new ObjectMapper((JsonFactory)new YAMLFactory().enable(YAMLGenerator.Feature.MINIMIZE_QUOTES));
        this.referenceIndex = (ReferenceIndexGenerator.Index)om.readValue(referenceIndexPath.toFile(), ReferenceIndexGenerator.Index.class);
    }

    private Map<String, List<String>> check() throws IOException {
        Map<String, String> titlesByReference = this.referenceIndex.getReferences().stream().collect(Collectors.toMap(s -> s.getReference(), s -> s.getTitle()));
        TreeMap<String, List<String>> errors = new TreeMap<String, List<String>>();
        try (Stream<Path> pathStream = Files.list(this.srcDir);){
            pathStream.filter(path -> this.includeFile(path.getFileName().toString())).forEach(path -> {
                List<String> guideLines;
                try {
                    guideLines = Files.readAllLines(path);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
                String fileName = path.getFileName().toString();
                StringBuilder currentBuffer = new StringBuilder();
                boolean inSourceBlock = false;
                boolean findDelimiter = false;
                String currentSourceBlockDelimiter = "----";
                int lineNumber = 0;
                for (String line : guideLines) {
                    ++lineNumber;
                    if (inSourceBlock) {
                        if (findDelimiter) {
                            if (line.isBlank() || line.startsWith(".")) continue;
                            if (!line.startsWith(SOURCE_BLOCK_DELIMITER)) {
                                throw new IllegalStateException("Unable to find source block delimiter in file " + fileName + " at line " + lineNumber);
                            }
                            currentSourceBlockDelimiter = line.stripTrailing();
                            findDelimiter = false;
                            continue;
                        }
                        if (!line.stripTrailing().equals(currentSourceBlockDelimiter)) continue;
                        inSourceBlock = false;
                        continue;
                    }
                    if (line.startsWith(SOURCE_BLOCK_PREFIX)) {
                        inSourceBlock = true;
                        findDelimiter = true;
                        if (currentBuffer.length() <= 0) continue;
                        CheckCrossReferences.checkLinks(titlesByReference, errors, fileName, currentBuffer.toString());
                        currentBuffer.setLength(0);
                        continue;
                    }
                    currentBuffer.append(line + "\n");
                }
                if (currentBuffer.length() > 0) {
                    CheckCrossReferences.checkLinks(titlesByReference, errors, fileName, currentBuffer.toString());
                }
            });
        }
        return errors;
    }

    private static void checkLinks(Map<String, String> titlesByReference, Map<String, List<String>> errors, String fileName, String content) {
        String reference;
        Matcher matcher = XREF_PATTERN.matcher(content);
        while (matcher.find()) {
            reference = CheckCrossReferences.getQualifiedReference(fileName, matcher.group(1));
            if (titlesByReference.containsKey(reference)) continue;
            CheckCrossReferences.addError(errors, fileName, reference + " in link " + matcher.group());
        }
        matcher = ANGLE_BRACKETS_WITHOUT_DESCRIPTION_PATTERN.matcher(content);
        while (matcher.find()) {
            reference = CheckCrossReferences.getQualifiedReference(fileName, matcher.group(1));
            if (titlesByReference.containsKey(reference)) continue;
            CheckCrossReferences.addError(errors, fileName, reference + " in link " + matcher.group());
        }
        matcher = ANGLE_BRACKETS_WITH_DESCRIPTION_PATTERN.matcher(content);
        while (matcher.find()) {
            reference = CheckCrossReferences.getQualifiedReference(fileName, matcher.group(1));
            if (titlesByReference.containsKey(reference)) continue;
            CheckCrossReferences.addError(errors, fileName, reference + " in link " + matcher.group());
        }
    }

    private boolean includeFile(String fileName) {
        if (fileName.startsWith("_attributes") || fileName.equals("README.adoc")) {
            return false;
        }
        if (fileName.startsWith("doc-")) {
            return false;
        }
        if (IGNORED_GUIDES.contains(fileName)) {
            return false;
        }
        return fileName.endsWith(".adoc");
    }

    private static String getQualifiedReference(String fileName, String reference) {
        if ((reference = CheckCrossReferences.normalizeAdoc(reference)).startsWith("#")) {
            return fileName + reference;
        }
        if (reference.contains(".adoc")) {
            return reference;
        }
        if (reference.contains("#")) {
            int hashIndex = reference.indexOf(35);
            return reference.substring(0, hashIndex) + ".adoc" + reference.substring(hashIndex);
        }
        return fileName + "#" + reference;
    }

    private static String normalizeAdoc(String adoc) {
        if (adoc.startsWith("./")) {
            return adoc.substring(2);
        }
        return adoc;
    }

    private static void addError(Map<String, List<String>> errors, String fileName, String error) {
        errors.computeIfAbsent(fileName, f -> new ArrayList()).add(error);
    }

    private static Path docsDir() {
        Path path = Paths.get(System.getProperty("user.dir"), new String[0]);
        if (path.endsWith("docs")) {
            return path;
        }
        return path.resolve("docs");
    }
}

