/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.docs.generation;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.exc.StreamWriteException;
import com.fasterxml.jackson.databind.DatabindException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.asciidoctor.Asciidoctor;
import org.asciidoctor.Options;
import org.asciidoctor.SafeMode;
import org.asciidoctor.ast.Block;
import org.asciidoctor.ast.Cell;
import org.asciidoctor.ast.Document;
import org.asciidoctor.ast.Row;
import org.asciidoctor.ast.Section;
import org.asciidoctor.ast.StructuralNode;
import org.asciidoctor.ast.Table;

public class ReferenceIndexGenerator {
    private static final String YAML_FRONTMATTER = "---\n";
    private static final String SOURCE_BLOCK_PREFIX = "[source";
    private static final String SOURCE_BLOCK_DELIMITER = "--";
    private static final Pattern ICON_PATTERN = Pattern.compile("icon:[^\\[]+\\[[^\\]]*\\] ");
    private final Path srcDir;
    private final Path targetDir;

    public static void main(String[] args) throws Exception {
        System.out.println("[INFO] Creating reference index generator: " + String.valueOf(List.of(args)));
        ReferenceIndexGenerator generator = new ReferenceIndexGenerator(args.length >= 1 ? Path.of(args[0], new String[0]) : ReferenceIndexGenerator.docsDir().resolve("src/main/asciidoc"), args.length >= 2 ? Path.of(args[1], new String[0]) : ReferenceIndexGenerator.docsDir().resolve("target"));
        System.out.println("[INFO] Generating reference index");
        Index index = generator.generateIndex();
        System.out.println("[INFO] Writing reference index file");
        generator.writeYamlFiles(index);
        System.out.println("[INFO] Transforming the source code");
        Map<String, List<String>> errors = generator.transformFiles(index);
        if (!errors.isEmpty()) {
            System.out.println();
            System.out.println("################################################");
            System.out.println("# Errors occurred while transforming references");
            System.out.println("################################################");
            System.out.println();
            for (Map.Entry<String, List<String>> errorEntry : errors.entrySet()) {
                System.out.println("- " + errorEntry.getKey());
                for (String error : errorEntry.getValue()) {
                    System.out.println("    . " + error);
                }
            }
            System.out.println();
            System.exit(1);
        }
        System.out.println("[INFO] Done");
    }

    public ReferenceIndexGenerator(Path srcDir, Path targetDir) {
        if (!Files.exists(srcDir, new LinkOption[0]) || !Files.isDirectory(srcDir, new LinkOption[0])) {
            throw new IllegalStateException(String.format("Source directory (%s) does not exist", srcDir.toAbsolutePath()));
        }
        this.srcDir = srcDir;
        if (!Files.exists(srcDir, new LinkOption[0]) || !Files.isDirectory(srcDir, new LinkOption[0])) {
            throw new IllegalStateException(String.format("Source directory (%s) does not exist", srcDir.toAbsolutePath()));
        }
        this.targetDir = targetDir;
    }

    private void writeYamlFiles(Index index) throws StreamWriteException, DatabindException, IOException {
        ObjectMapper om = new ObjectMapper((JsonFactory)new YAMLFactory().enable(YAMLGenerator.Feature.MINIMIZE_QUOTES));
        om.writeValue(this.targetDir.resolve("referenceIndex.yaml").toFile(), (Object)index);
    }

    private Index generateIndex() throws IOException {
        if (!Files.exists(this.srcDir, new LinkOption[0]) || !Files.isDirectory(this.srcDir, new LinkOption[0])) {
            throw new IllegalStateException(String.format("Source directory (%s) does not exist", this.srcDir.toAbsolutePath()));
        }
        if (!Files.exists(this.targetDir, new LinkOption[0]) || !Files.isDirectory(this.targetDir, new LinkOption[0])) {
            throw new IllegalStateException(String.format("Target directory (%s) does not exist. Exiting.%n", this.targetDir.toAbsolutePath()));
        }
        Options options = Options.builder().docType("book").sourceDir(this.srcDir.toFile()).baseDir(this.srcDir.toFile()).safe(SafeMode.UNSAFE).build();
        Index index = new Index();
        try (Asciidoctor asciidoctor = Asciidoctor.Factory.create();
             Stream<Path> pathStream = Files.list(this.srcDir);){
            pathStream.filter(path -> this.includeFile(path.getFileName().toString())).forEach(path -> {
                String guideContent;
                try {
                    guideContent = Files.readString(path);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
                if (guideContent.startsWith(YAML_FRONTMATTER)) {
                    int end = guideContent.indexOf(YAML_FRONTMATTER, YAML_FRONTMATTER.length());
                    guideContent = guideContent.substring(end + YAML_FRONTMATTER.length());
                }
                Document doc = asciidoctor.load(guideContent, options);
                String fileName = path.getFileName().toString();
                String title = doc.getDoctitle();
                index.add(new IndexReference(fileName, title));
                this.addBlocks(index, fileName, doc.getBlocks());
            });
        }
        return index;
    }

    private void addBlocks(Index index, String fileName, List<StructuralNode> blocks) {
        for (StructuralNode block : blocks) {
            if ((block instanceof Section || block instanceof Table || block instanceof Block) && block.getId() != null) {
                IndexReference reference = new IndexReference(fileName, block.getId(), block.getTitle() != null ? block.getTitle().replace("<code>", "`").replace("</code>", "`").replace('\n', ' ').replace("&#8217;", "'").replace("&amp;", "&").replace("&#8230;&#8203;", "...").replace("<em>", "_").replace("</em>", "_").replace("<b>", "*").replace("</b>", "*").replace("<strong>", "*").replace("</strong>", "*").replaceAll("<a.*</a> ", "") : "~~ unknown title ~~");
                index.add(reference);
            }
            if (block instanceof Table) {
                for (Row row : ((Table)block).getBody()) {
                    for (Cell cell : row.getCells()) {
                        String cellContent = ICON_PATTERN.matcher(cell.getSource()).replaceAll("").trim();
                        if (!cellContent.startsWith("[[") || !cellContent.contains("]]")) continue;
                        IndexReference reference = new IndexReference(fileName, cellContent.substring(2, cellContent.indexOf("]]")), "Configuration property documentation");
                        index.add(reference);
                    }
                }
            }
            this.addBlocks(index, fileName, block.getBlocks());
        }
    }

    private Map<String, List<String>> transformFiles(Index index) throws IOException {
        Map<String, String> titlesByReference = index.getReferences().stream().collect(Collectors.toMap(s -> s.getReference(), s -> s.getTitle()));
        LinkedHashMap<String, List<String>> errors = new LinkedHashMap<String, List<String>>();
        try (Stream<Path> pathStream = Files.list(this.srcDir);){
            pathStream.filter(path -> this.includeFile(path.getFileName().toString())).forEach(path -> {
                List<String> guideLines;
                try {
                    guideLines = Files.readAllLines(path);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
                String fileName = path.getFileName().toString();
                StringBuilder rewrittenGuide = new StringBuilder();
                StringBuilder currentBuffer = new StringBuilder();
                boolean inSourceBlock = false;
                boolean findDelimiter = false;
                String currentSourceBlockDelimiter = "----";
                int lineNumber = 0;
                for (String line : guideLines) {
                    ++lineNumber;
                    if (inSourceBlock) {
                        if (findDelimiter) {
                            rewrittenGuide.append(line + "\n");
                            if (line.isBlank() || line.startsWith(".")) continue;
                            if (!line.startsWith(SOURCE_BLOCK_DELIMITER)) {
                                throw new IllegalStateException("Unable to find source block delimiter in file " + fileName + " at line " + lineNumber);
                            }
                            currentSourceBlockDelimiter = line.stripTrailing();
                            findDelimiter = false;
                            continue;
                        }
                        if (line.stripTrailing().equals(currentSourceBlockDelimiter)) {
                            inSourceBlock = false;
                        }
                        rewrittenGuide.append(line + "\n");
                        continue;
                    }
                    if (line.startsWith(SOURCE_BLOCK_PREFIX)) {
                        inSourceBlock = true;
                        findDelimiter = true;
                        if (currentBuffer.length() > 0) {
                            rewrittenGuide.append(this.rewriteLinks(titlesByReference, errors, fileName, currentBuffer.toString()));
                            currentBuffer.setLength(0);
                        }
                        rewrittenGuide.append(line + "\n");
                        continue;
                    }
                    currentBuffer.append(line + "\n");
                }
                if (currentBuffer.length() > 0) {
                    rewrittenGuide.append(this.rewriteLinks(titlesByReference, errors, fileName, currentBuffer.toString()));
                }
                try {
                    Files.writeString(path, (CharSequence)rewrittenGuide, new OpenOption[0]);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
        }
        return errors;
    }

    private String rewriteLinks(Map<String, String> titlesByReference, Map<String, List<String>> errors, String fileName, String content) {
        return content;
    }

    private boolean includeFile(String fileName) {
        if (fileName.startsWith("_attributes") || fileName.equals("README.adoc")) {
            return false;
        }
        return fileName.endsWith(".adoc");
    }

    private static Path docsDir() {
        Path path = Paths.get(System.getProperty("user.dir"), new String[0]);
        if (path.endsWith("docs")) {
            return path;
        }
        return path.resolve("docs");
    }

    public static class Index {
        private List<IndexReference> references = new ArrayList<IndexReference>();

        public List<IndexReference> getReferences() {
            return this.references;
        }

        public void add(IndexReference reference) {
            this.references.add(reference);
        }
    }

    public static class IndexReference {
        private String reference;
        private String title;

        public IndexReference() {
        }

        public IndexReference(String fileName, String title) {
            this.reference = fileName;
            this.title = title;
        }

        public IndexReference(String fileName, String blockId, String title) {
            this.reference = fileName + "#" + blockId;
            this.title = title;
        }

        public String getReference() {
            return this.reference;
        }

        public String getTitle() {
            return this.title;
        }

        public String toString() {
            return this.reference + " -> " + this.title;
        }
    }
}

