/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.internal.cache;

import java.io.File;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.geode.cache.DiskAccessException;
import org.apache.geode.distributed.internal.DistributionConfig;
import org.apache.geode.distributed.internal.InternalDistributedSystem;
import org.apache.geode.internal.cache.DirectoryHolder;
import org.apache.geode.internal.cache.DiskStoreImpl;
import org.apache.geode.internal.cache.GemFireCacheImpl;
import org.apache.geode.internal.i18n.LocalizedStrings;
import org.apache.geode.internal.logging.LogService;
import org.apache.geode.internal.logging.LoggingThreadGroup;
import org.apache.geode.internal.logging.log4j.LocalizedMessage;
import org.apache.geode.internal.logging.log4j.LogMarker;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;

public class DiskStoreMonitor {
    private static final Logger logger = LogService.getLogger();
    private static final boolean DISABLE_MONITOR = Boolean.getBoolean("gemfire.DISK_USAGE_DISABLE_MONITORING");
    private static final int USAGE_CHECK_INTERVAL = Integer.getInteger("gemfire.DISK_USAGE_POLLING_INTERVAL_MILLIS", 10000);
    private static final float LOG_WARNING_THRESHOLD_PCT = Integer.getInteger("gemfire.DISK_USAGE_LOG_WARNING_PERCENT", 99).intValue();
    private final ScheduledExecutorService exec;
    private final Map<DiskStoreImpl, Set<DirectoryHolderUsage>> disks = new ConcurrentHashMap<DiskStoreImpl, Set<DirectoryHolderUsage>>();
    private final LogUsage logDisk = new LogUsage(this.getLogDir());
    volatile DiskStateAction _testAction;

    public static void checkWarning(float val) {
        if (val < 0.0f || val > 100.0f) {
            throw new IllegalArgumentException(LocalizedStrings.DiskWriteAttributesFactory_DISK_USAGE_WARNING_INVALID_0.toLocalizedString(Float.valueOf(val)));
        }
    }

    public static void checkCritical(float val) {
        if (val < 0.0f || val > 100.0f) {
            throw new IllegalArgumentException(LocalizedStrings.DiskWriteAttributesFactory_DISK_USAGE_CRITICAL_INVALID_0.toLocalizedString(Float.valueOf(val)));
        }
    }

    public DiskStoreMonitor() {
        if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR)) {
            logger.trace(LogMarker.DISK_STORE_MONITOR, "Disk monitoring is {}", (Object)(DISABLE_MONITOR ? "disabled" : "enabled"));
            logger.trace(LogMarker.DISK_STORE_MONITOR, "Log directory usage warning is set to {}%", (Object)Float.valueOf(LOG_WARNING_THRESHOLD_PCT));
            logger.trace(LogMarker.DISK_STORE_MONITOR, "Scheduling disk usage checks every {} ms", (Object)USAGE_CHECK_INTERVAL);
        }
        if (DISABLE_MONITOR) {
            this.exec = null;
        } else {
            final LoggingThreadGroup tg = LoggingThreadGroup.createThreadGroup(LocalizedStrings.DiskStoreMonitor_ThreadGroup.toLocalizedString(), logger);
            this.exec = Executors.newScheduledThreadPool(1, new ThreadFactory(){

                @Override
                public Thread newThread(Runnable r) {
                    Thread t = new Thread(tg, r, "DiskStoreMonitor");
                    t.setDaemon(true);
                    return t;
                }
            });
            this.exec.scheduleWithFixedDelay(new Runnable(){

                @Override
                public void run() {
                    try {
                        DiskStoreMonitor.this.checkUsage();
                    }
                    catch (Exception e) {
                        logger.error((Message)LocalizedMessage.create(LocalizedStrings.DiskStoreMonitor_ERR), (Throwable)e);
                    }
                }
            }, 0L, USAGE_CHECK_INTERVAL, TimeUnit.MILLISECONDS);
        }
    }

    public void addDiskStore(DiskStoreImpl ds) {
        if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR)) {
            logger.trace(LogMarker.DISK_STORE_MONITOR, "Now monitoring disk store {}", (Object)ds.getName());
        }
        HashSet<DirectoryHolderUsage> du = new HashSet<DirectoryHolderUsage>();
        for (DirectoryHolder dir : ds.getDirectoryHolders()) {
            du.add(new DirectoryHolderUsage(ds, dir));
        }
        this.disks.put(ds, du);
    }

    public void removeDiskStore(DiskStoreImpl ds) {
        if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR)) {
            logger.trace(LogMarker.DISK_STORE_MONITOR, "No longer monitoring disk store {}", (Object)ds.getName());
        }
        this.disks.remove(ds);
    }

    public boolean isNormal(DiskStoreImpl ds, DirectoryHolder dir) {
        Set<DirectoryHolderUsage> dirs = this.disks.get(ds);
        if (dirs != null) {
            for (DirectoryHolderUsage du : dirs) {
                if (du.dir != dir) continue;
                return du.getState() == DiskState.NORMAL;
            }
        }
        return true;
    }

    public void close() {
        if (this.exec != null) {
            this.exec.shutdownNow();
        }
        this.disks.clear();
    }

    private void checkUsage() {
        for (Map.Entry<DiskStoreImpl, Set<DirectoryHolderUsage>> entry : this.disks.entrySet()) {
            DiskUsage du;
            DiskState update;
            DiskStoreImpl ds = entry.getKey();
            Iterator<DirectoryHolderUsage> iterator = entry.getValue().iterator();
            while (iterator.hasNext() && (update = (du = (DiskUsage)iterator.next()).update(ds.getDiskUsageWarningPercentage(), ds.getDiskUsageCriticalPercentage())) != DiskState.CRITICAL) {
            }
        }
        this.logDisk.update(LOG_WARNING_THRESHOLD_PCT, 100.0f);
    }

    private File getLogDir() {
        DistributionConfig conf;
        InternalDistributedSystem ds;
        File log = null;
        GemFireCacheImpl internalCache = GemFireCacheImpl.getInstance();
        if (internalCache != null && (ds = internalCache.getInternalDistributedSystem()) != null && (conf = ds.getConfig()) != null && (log = conf.getLogFile()) != null) {
            log = log.getParentFile();
        }
        if (log == null) {
            log = new File(".");
        }
        return log;
    }

    class DirectoryHolderUsage
    extends DiskUsage {
        private final DiskStoreImpl disk;
        private final DirectoryHolder dir;

        public DirectoryHolderUsage(DiskStoreImpl disk, DirectoryHolder dir) {
            this.disk = disk;
            this.dir = dir;
        }

        @Override
        protected void handleStateChange(DiskState next, String pct) {
            if (DiskStoreMonitor.this._testAction != null) {
                logger.info(LogMarker.DISK_STORE_MONITOR, "Invoking test handler for state change to {}", (Object)next);
                DiskStoreMonitor.this._testAction.handleDiskStateChange(next);
            }
            Object[] args = new Object[]{this.dir.getDir(), this.disk.getName(), pct};
            String msg = "Critical disk usage threshold exceeded for volume " + this.dir.getDir().getAbsolutePath() + ": " + pct + " full";
            switch (next) {
                case NORMAL: {
                    logger.warn(LogMarker.DISK_STORE_MONITOR, (Message)LocalizedMessage.create(LocalizedStrings.DiskStoreMonitor_DISK_NORMAL, args));
                    break;
                }
                case WARN: {
                    logger.warn(LogMarker.DISK_STORE_MONITOR, (Message)LocalizedMessage.create(LocalizedStrings.DiskStoreMonitor_DISK_WARNING, args));
                    break;
                }
                case CRITICAL: {
                    logger.error(LogMarker.DISK_STORE_MONITOR, (Message)LocalizedMessage.create(LocalizedStrings.DiskStoreMonitor_DISK_CRITICAL, args));
                    this.disk.handleDiskAccessException(new DiskAccessException(msg, this.disk));
                }
            }
        }

        @Override
        protected File dir() {
            return this.dir.getDir();
        }

        @Override
        protected long getMinimumSpace() {
            return (long)DiskStoreImpl.MIN_DISK_SPACE_FOR_LOGS + this.disk.getMaxOplogSize();
        }

        @Override
        protected void recordStats(long total, long free, long elapsed) {
            this.dir.getDiskDirectoryStats().addVolumeCheck(total, free, elapsed);
        }
    }

    static class LogUsage
    extends DiskUsage {
        private final File dir;

        public LogUsage(File dir) {
            this.dir = dir;
        }

        @Override
        protected void handleStateChange(DiskState next, String pct) {
            Object[] args = new Object[]{this.dir.getAbsolutePath(), pct};
            switch (next) {
                case NORMAL: {
                    logger.info(LogMarker.DISK_STORE_MONITOR, (Message)LocalizedMessage.create(LocalizedStrings.DiskStoreMonitor_LOG_DISK_NORMAL, args));
                    break;
                }
                case WARN: 
                case CRITICAL: {
                    logger.warn(LogMarker.DISK_STORE_MONITOR, (Message)LocalizedMessage.create(LocalizedStrings.DiskStoreMonitor_LOG_DISK_WARNING, args));
                }
            }
        }

        @Override
        protected long getMinimumSpace() {
            return DiskStoreImpl.MIN_DISK_SPACE_FOR_LOGS;
        }

        @Override
        protected File dir() {
            return this.dir;
        }

        @Override
        protected void recordStats(long total, long free, long elapsed) {
        }
    }

    static abstract class DiskUsage {
        private DiskState state = DiskState.NORMAL;

        DiskUsage() {
        }

        public synchronized DiskState getState() {
            return this.state;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public DiskState update(float warning, float critical) {
            boolean belowMin;
            DiskState next;
            DiskState current;
            DiskUsage diskUsage = this;
            synchronized (diskUsage) {
                current = this.state;
            }
            if (!(warning > 0.0f) && !(critical > 0.0f)) {
                return current;
            }
            if (!this.dir().exists()) {
                if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR)) {
                    logger.trace(LogMarker.DISK_STORE_MONITOR, "Skipping check of non-existent directory {}", (Object)this.dir().getAbsolutePath());
                }
                return current;
            }
            long min = this.getMinimumSpace();
            if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR)) {
                logger.trace(LogMarker.DISK_STORE_MONITOR, "Checking usage for directory {}, minimum free space is {} MB", (Object)this.dir().getAbsolutePath(), (Object)min);
            }
            long start = System.nanoTime();
            long remaining = this.dir().getUsableSpace();
            long total = this.dir().getTotalSpace();
            long elapsed = System.nanoTime() - start;
            double use = 100.0 * (double)(total - remaining) / (double)total;
            this.recordStats(total, remaining, elapsed);
            String pct = Math.round(use) + "%";
            if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR)) {
                logger.trace(LogMarker.DISK_STORE_MONITOR, "Directory {} has {} bytes free out of {} ({} usage)", (Object)this.dir().getAbsolutePath(), (Object)remaining, (Object)total, (Object)pct);
            }
            if ((next = DiskState.select(use, warning, critical, belowMin = remaining < 0x100000L * min)) == current) {
                return next;
            }
            DiskUsage diskUsage2 = this;
            synchronized (diskUsage2) {
                this.state = next;
            }
            this.handleStateChange(next, pct);
            return next;
        }

        protected abstract File dir();

        protected abstract long getMinimumSpace();

        protected abstract void recordStats(long var1, long var3, long var5);

        protected abstract void handleStateChange(DiskState var1, String var2);
    }

    static interface DiskStateAction {
        public void handleDiskStateChange(DiskState var1);
    }

    static enum DiskState {
        NORMAL,
        WARN,
        CRITICAL;


        public static DiskState select(double actual, double warn, double critical, boolean belowMinimum) {
            if (critical > 0.0 && (actual > critical || belowMinimum)) {
                return CRITICAL;
            }
            if (warn > 0.0 && actual > warn) {
                return WARN;
            }
            return NORMAL;
        }
    }
}

