/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.instrumentation.testing.junit.http;

import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.instrumentation.testing.InstrumentationTestRunner;
import io.opentelemetry.instrumentation.testing.junit.http.HttpClientResult;
import io.opentelemetry.instrumentation.testing.junit.http.HttpClientTestOptions;
import io.opentelemetry.instrumentation.testing.junit.http.HttpClientTestServer;
import io.opentelemetry.instrumentation.testing.junit.http.HttpClientTypeAdapter;
import io.opentelemetry.instrumentation.testing.junit.http.SingleConnection;
import io.opentelemetry.instrumentation.testing.util.TelemetryDataUtil;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.testing.assertj.AttributeAssertion;
import io.opentelemetry.sdk.testing.assertj.OpenTelemetryAssertions;
import io.opentelemetry.sdk.testing.assertj.SpanDataAssert;
import io.opentelemetry.sdk.testing.assertj.TraceAssert;
import io.opentelemetry.sdk.trace.data.SpanData;
import io.opentelemetry.sdk.trace.data.StatusData;
import io.opentelemetry.semconv.ErrorAttributes;
import io.opentelemetry.semconv.HttpAttributes;
import io.opentelemetry.semconv.NetworkAttributes;
import io.opentelemetry.semconv.ServerAttributes;
import io.opentelemetry.semconv.UrlAttributes;
import io.opentelemetry.semconv.UserAgentAttributes;
import java.net.URI;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.annotation.Nullable;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.condition.DisabledIfSystemProperty;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.junit.jupiter.params.provider.ValueSource;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
public abstract class AbstractHttpClientTest<REQUEST>
implements HttpClientTypeAdapter<REQUEST> {
    public static final Duration CONNECTION_TIMEOUT = Duration.ofSeconds(5L);
    public static final Duration READ_TIMEOUT = Duration.ofSeconds(2L);
    public static final String TEST_REQUEST_HEADER = "X-Test-Request";
    public static final String TEST_RESPONSE_HEADER = "X-Test-Response";
    static final String BASIC_AUTH_KEY = "custom-authorization-header";
    static final String BASIC_AUTH_VAL = "plain text auth token";
    protected InstrumentationTestRunner testing;
    private HttpClientTestServer server;
    private HttpClientTestOptions options;

    protected final Duration connectTimeout() {
        return CONNECTION_TIMEOUT;
    }

    protected final Duration readTimeout() {
        return READ_TIMEOUT;
    }

    @BeforeAll
    void setupOptions() {
        HttpClientTestOptions.Builder builder = HttpClientTestOptions.builder();
        this.configure(builder);
        this.options = builder.build();
    }

    protected void configure(HttpClientTestOptions.Builder optionsBuilder) {
    }

    final void setTesting(InstrumentationTestRunner testing, HttpClientTestServer server) {
        this.testing = testing;
        this.server = server;
    }

    @BeforeEach
    void verifyExtension() {
        if (this.testing == null) {
            throw new AssertionError((Object)"Subclasses of AbstractHttpClientTest must register HttpClientInstrumentationExtension");
        }
    }

    @ParameterizedTest
    @ValueSource(strings={"/success", "/success?with=params"})
    void successfulGetRequest(String path) throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        URI uri = this.resolveAddress(path);
        String method = "GET";
        int responseCode = this.doRequest(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent().hasStatus(StatusData.unset()), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
    }

    @Test
    void requestWithNonStandardHttpMethod() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestNonStandardHttpMethod());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        URI uri = this.resolveAddress("/success");
        String method = "TEST";
        int responseCode = this.doRequest(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo("2".equals(this.options.getHttpProtocolVersion().apply(uri)) ? 400 : 405);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, "_OTHER", responseCode, null).hasNoParent().hasAttribute(HttpAttributes.HTTP_REQUEST_METHOD_ORIGINAL, (Object)method)}));
    }

    @ParameterizedTest
    @ValueSource(strings={"PUT", "POST"})
    void successfulRequestWithParent(String method) throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        URI uri = this.resolveAddress("/success");
        int responseCode = this.testing.runWithSpan("parent", () -> this.doRequest(method, uri));
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("parent").hasKind(SpanKind.INTERNAL).hasNoParent(), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasParent(trace.getSpan(0)), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(1))}));
    }

    @Test
    void successfulRequestWithNotSampledParent() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = this.resolveAddress("/success");
        int responseCode = this.testing.runWithNonRecordingSpan(() -> this.doRequest(method, uri));
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        Thread.sleep(200L);
        OpenTelemetryAssertions.assertThat(this.testing.traces()).isEmpty();
    }

    @ParameterizedTest
    @ValueSource(strings={"PUT", "POST"})
    void shouldSuppressNestedClientSpanIfAlreadyUnderParentClientSpan(String method) throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestWithClientParent());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        URI uri = this.resolveAddress("/success");
        int responseCode = this.testing.runWithHttpClientSpan("parent-client-span", () -> this.doRequest(method, uri));
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertSortedTraces(TelemetryDataUtil.orderByRootSpanName("parent-client-span", "test-http-server"), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("parent-client-span").hasKind(SpanKind.CLIENT).hasNoParent()}), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> AbstractHttpClientTest.assertServerSpan(span)}));
    }

    @Test
    void requestWithCallbackAndParent() throws Throwable {
        Assumptions.assumeTrue((boolean)this.options.getTestCallback());
        Assumptions.assumeTrue((boolean)this.options.getTestCallbackWithParent());
        String method = "GET";
        URI uri = this.resolveAddress("/success");
        HttpClientResult result = this.testing.runWithSpan("parent", () -> this.doRequestWithCallback(method, uri, () -> this.testing.runWithSpan("child", () -> {})));
        OpenTelemetryAssertions.assertThat((int)result.get()).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("parent").hasKind(SpanKind.INTERNAL).hasNoParent(), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasParent(trace.getSpan(0)), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(1)), span -> span.hasName("child").hasKind(SpanKind.INTERNAL).hasParent(trace.getSpan(0))}));
    }

    @Test
    void requestWithCallbackAndNoParent() throws Throwable {
        Assumptions.assumeTrue((boolean)this.options.getTestCallback());
        String method = "GET";
        URI uri = this.resolveAddress("/success");
        HttpClientResult result = this.doRequestWithCallback(method, uri, () -> this.testing.runWithSpan("callback", () -> {}));
        OpenTelemetryAssertions.assertThat((int)result.get()).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("callback").hasKind(SpanKind.INTERNAL).hasNoParent()}));
    }

    @Test
    void basicRequestWith1Redirect() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestRedirects());
        String method = "GET";
        URI uri = this.resolveAddress("/redirect");
        int responseCode = this.doRequest(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        if (this.options.isLowLevelInstrumentation()) {
            this.testing.waitAndAssertSortedTraces(TelemetryDataUtil.comparingRootSpanAttribute(HttpAttributes.HTTP_REQUEST_RESEND_COUNT), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, this.options.getResponseCodeOnRedirectError(), null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri.resolve("/success"), method, responseCode, 1).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
        } else {
            this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0)), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
        }
    }

    @Test
    void basicRequestWith2Redirects() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestRedirects());
        String method = "GET";
        URI uri = this.resolveAddress("/another-redirect");
        int responseCode = this.doRequest(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        if (this.options.isLowLevelInstrumentation()) {
            this.testing.waitAndAssertSortedTraces(TelemetryDataUtil.comparingRootSpanAttribute(HttpAttributes.HTTP_REQUEST_RESEND_COUNT), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, this.options.getResponseCodeOnRedirectError(), null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri.resolve("/redirect"), method, this.options.getResponseCodeOnRedirectError(), 1).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri.resolve("/success"), method, responseCode, 2).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
        } else {
            this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0)), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0)), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
        }
    }

    @Test
    void circularRedirects() {
        Assumptions.assumeTrue((boolean)this.options.getTestRedirects());
        Assumptions.assumeTrue((boolean)this.options.getTestCircularRedirects());
        String method = "GET";
        URI uri = this.resolveAddress("/circular-redirect");
        Throwable thrown = Assertions.catchThrowable(() -> this.doRequest(method, uri));
        Throwable ex = thrown instanceof ExecutionException ? thrown.getCause() : thrown;
        Throwable clientError = this.options.getClientSpanErrorMapper().apply(uri, ex);
        if (this.options.isLowLevelInstrumentation()) {
            this.testing.waitAndAssertSortedTraces(TelemetryDataUtil.comparingRootSpanAttribute(HttpAttributes.HTTP_REQUEST_RESEND_COUNT), IntStream.range(0, this.options.getMaxRedirects()).mapToObj(i -> this.makeCircularRedirectAssertForLolLevelTrace(uri, method, i)).collect(Collectors.toList()));
        } else {
            this.testing.waitAndAssertTraces(trace -> {
                ArrayList<Consumer<SpanDataAssert>> assertions = new ArrayList<Consumer<SpanDataAssert>>();
                assertions.add(span -> this.assertClientSpan((SpanDataAssert)span, uri, method, this.options.getResponseCodeOnRedirectError(), null).hasNoParent().hasException(clientError));
                for (int i = 0; i < this.options.getMaxRedirects(); ++i) {
                    assertions.add(span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0)));
                }
                trace.hasSpansSatisfyingExactly(assertions);
            });
        }
    }

    private Consumer<TraceAssert> makeCircularRedirectAssertForLolLevelTrace(URI uri, String method, int resendNo) {
        Integer resendCountValue = resendNo > 0 ? Integer.valueOf(resendNo) : null;
        return trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, this.options.getResponseCodeOnRedirectError(), resendCountValue), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))});
    }

    @Test
    void redirectToSecuredCopiesAuthHeader() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestRedirects());
        String method = "GET";
        URI uri = this.resolveAddress("/to-secured");
        int responseCode = this.doRequest(method, uri, Collections.singletonMap(BASIC_AUTH_KEY, BASIC_AUTH_VAL));
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        if (this.options.isLowLevelInstrumentation()) {
            this.testing.waitAndAssertSortedTraces(TelemetryDataUtil.comparingRootSpanAttribute(HttpAttributes.HTTP_REQUEST_RESEND_COUNT), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, this.options.getResponseCodeOnRedirectError(), null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri.resolve("/secured"), method, responseCode, 1).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
        } else {
            this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0)), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
        }
    }

    @ParameterizedTest
    @CsvSource(value={"/error,500", "/client-error,400"})
    void errorSpan(String path, int responseCode) {
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = this.resolveAddress(path);
        this.testing.runWithSpan("parent", () -> {
            try {
                this.doRequest(method, uri);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        });
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("parent").hasKind(SpanKind.INTERNAL).hasNoParent(), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasParent(trace.getSpan(0)), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(1))}));
    }

    @Test
    void reuseRequest() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestReusedRequest());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = this.resolveAddress("/success");
        int responseCode = this.doReusedRequest(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}), trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
    }

    @Test
    void requestWithExistingTracingHeaders() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = this.resolveAddress("/success");
        int responseCode = this.doRequestWithExistingTracingHeaders(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
    }

    @Test
    void captureHttpHeaders() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestCaptureHttpHeaders());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        URI uri = this.resolveAddress("/success");
        String method = "GET";
        int responseCode = this.doRequest(method, uri, Collections.singletonMap(TEST_REQUEST_HEADER, "test"));
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent().hasAttributesSatisfying(Arrays.asList(OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.stringArrayKey((String)"http.request.header.x-test-request"), Collections.singletonList("test")), OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.stringArrayKey((String)"http.response.header.x-test-response"), Collections.singletonList("test")))), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
    }

    @Test
    void connectionErrorUnopenedPort() {
        Assumptions.assumeTrue((boolean)this.options.getTestConnectionFailure());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = URI.create("http://localhost:61/");
        Throwable thrown = Assertions.catchThrowable(() -> this.testing.runWithSpan("parent", () -> this.doRequest(method, uri)));
        Throwable ex = thrown instanceof ExecutionException ? thrown.getCause() : thrown;
        Throwable clientError = this.options.getClientSpanErrorMapper().apply(uri, ex);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("parent").hasKind(SpanKind.INTERNAL).hasNoParent().hasStatus(StatusData.error()).hasException(ex), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, null, null).hasParent(trace.getSpan(0)).hasException(clientError)}));
    }

    @Test
    void connectionErrorUnopenedPortWithCallback() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestConnectionFailure());
        Assumptions.assumeTrue((boolean)this.options.getTestCallback());
        Assumptions.assumeTrue((boolean)this.options.getTestErrorWithCallback());
        String method = "GET";
        URI uri = URI.create("http://localhost:61/");
        HttpClientResult result = this.testing.runWithSpan("parent", () -> this.doRequestWithCallback(method, uri, () -> this.testing.runWithSpan("callback", () -> {})));
        Throwable thrown = Assertions.catchThrowable(result::get);
        Throwable ex = thrown instanceof ExecutionException ? thrown.getCause() : thrown;
        Throwable clientError = this.options.getClientSpanErrorMapper().apply(uri, ex);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactlyInAnyOrder(new Consumer[]{span -> span.hasName("parent").hasKind(SpanKind.INTERNAL).hasNoParent(), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, null, null).hasParent(trace.getSpan(0)).hasException(clientError), span -> span.hasName("callback").hasKind(SpanKind.INTERNAL).hasParent(trace.getSpan(0))}));
    }

    @Test
    void connectionErrorNonRoutableAddress() {
        Assumptions.assumeTrue((boolean)this.options.getTestRemoteConnection());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "HEAD";
        URI uri = URI.create(this.options.getTestHttps() ? "https://192.0.2.1/" : "http://192.0.2.1/");
        Throwable thrown = Assertions.catchThrowable(() -> this.testing.runWithSpan("parent", () -> this.doRequest(method, uri)));
        Throwable ex = thrown instanceof ExecutionException ? thrown.getCause() : thrown;
        Throwable clientError = this.options.getClientSpanErrorMapper().apply(uri, ex);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("parent").hasKind(SpanKind.INTERNAL).hasNoParent().hasStatus(StatusData.error()).hasException(ex), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, null, null).hasParent(trace.getSpan(0)).hasException(clientError)}));
    }

    @Test
    void readTimedOut() {
        Assumptions.assumeTrue((boolean)this.options.getTestReadTimeout());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = this.resolveAddress("/read-timeout");
        Throwable thrown = Assertions.catchThrowable(() -> this.testing.runWithSpan("parent", () -> this.doRequest(method, uri)));
        Throwable ex = thrown instanceof ExecutionException ? thrown.getCause() : thrown;
        Throwable clientError = this.options.getClientSpanErrorMapper().apply(uri, ex);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName("parent").hasKind(SpanKind.INTERNAL).hasNoParent().hasStatus(StatusData.error()).hasException(ex), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, null, null).hasParent(trace.getSpan(0)).hasException(clientError), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(1))}));
    }

    @DisabledIfSystemProperty(named="java.vm.name", matches=".*IBM J9 VM.*", disabledReason="IBM JVM has different protocol support for TLS")
    @Test
    void httpsRequest() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getTestRemoteConnection());
        Assumptions.assumeTrue((boolean)this.options.getTestHttps());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = URI.create("https://localhost:" + this.server.httpsPort() + "/success");
        int responseCode = this.doRequest(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))}));
    }

    @Test
    void httpClientMetrics() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        URI uri = this.resolveAddress("/success");
        String method = "GET";
        int responseCode = this.doRequest(method, uri);
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        AtomicReference instrumentationName = new AtomicReference();
        this.testing.waitAndAssertTraces(trace -> {
            instrumentationName.set(trace.getSpan(0).getInstrumentationScopeInfo().getName());
            trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, responseCode, null).hasNoParent(), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))});
        });
        this.testing.waitAndAssertMetrics((String)instrumentationName.get(), "http.client.request.duration", metrics -> metrics.anySatisfy(metric -> OpenTelemetryAssertions.assertThat((MetricData)metric).hasDescription("Duration of HTTP client requests.").hasUnit("s").hasHistogramSatisfying(histogram -> histogram.hasPointsSatisfying(new Consumer[]{point -> point.hasSumGreaterThan(0.0)}))));
    }

    @Test
    void highConcurrency() {
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        int count = 50;
        String method = "GET";
        URI uri = this.resolveAddress("/success");
        CountDownLatch latch = new CountDownLatch(1);
        ExecutorService pool = Executors.newFixedThreadPool(4);
        int i = 0;
        while (i < count) {
            int index = i++;
            Runnable job = () -> {
                try {
                    latch.await();
                }
                catch (InterruptedException e) {
                    throw new AssertionError((Object)e);
                }
                try {
                    Integer result = this.testing.runWithSpan("Parent span " + index, () -> {
                        Span.current().setAttribute("test.request.id", (long)index);
                        return this.doRequest(method, uri, Collections.singletonMap("test-request-id", String.valueOf(index)));
                    });
                    OpenTelemetryAssertions.assertThat((Integer)result).isEqualTo(200);
                }
                catch (Throwable throwable) {
                    if (throwable instanceof AssertionError) {
                        throw (AssertionError)((Object)throwable);
                    }
                    throw new AssertionError((Object)throwable);
                }
            };
            pool.submit(job);
        }
        latch.countDown();
        ArrayList<Consumer<TraceAssert>> assertions = new ArrayList<Consumer<TraceAssert>>();
        for (int i2 = 0; i2 < count; ++i2) {
            assertions.add(trace -> {
                SpanData rootSpan = trace.getSpan(0);
                int requestId = Integer.parseInt(rootSpan.getName().substring("Parent span ".length()));
                trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName(rootSpan.getName()).hasKind(SpanKind.INTERNAL).hasNoParent().hasAttributesSatisfyingExactly(new AttributeAssertion[]{OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.longKey((String)"test.request.id"), (int)requestId)}), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasParent(rootSpan), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(1)).hasAttributesSatisfyingExactly(new AttributeAssertion[]{OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.longKey((String)"test.request.id"), (int)requestId)})});
            });
        }
        this.testing.waitAndAssertTraces(assertions);
        pool.shutdown();
    }

    @Test
    void highConcurrencyWithCallback() {
        Assumptions.assumeTrue((boolean)this.options.getTestCallback());
        Assumptions.assumeTrue((boolean)this.options.getTestCallbackWithParent());
        int count = 50;
        String method = "GET";
        URI uri = this.resolveAddress("/success");
        CountDownLatch latch = new CountDownLatch(1);
        ExecutorService pool = Executors.newFixedThreadPool(4);
        IntStream.range(0, count).forEach(index -> {
            Runnable job = () -> {
                try {
                    latch.await();
                }
                catch (InterruptedException e) {
                    throw new AssertionError((Object)e);
                }
                try {
                    HttpClientResult result = this.testing.runWithSpan("Parent span " + index, () -> {
                        Span.current().setAttribute("test.request.id", (long)index);
                        return this.doRequestWithCallback(method, uri, Collections.singletonMap("test-request-id", String.valueOf(index)), () -> this.testing.runWithSpan("child", () -> {}));
                    });
                    OpenTelemetryAssertions.assertThat((int)result.get()).isEqualTo(200);
                }
                catch (Throwable throwable) {
                    if (throwable instanceof AssertionError) {
                        throw (AssertionError)((Object)throwable);
                    }
                    throw new AssertionError((Object)throwable);
                }
            };
            pool.submit(job);
        });
        latch.countDown();
        ArrayList<Consumer<TraceAssert>> assertions = new ArrayList<Consumer<TraceAssert>>();
        for (int i = 0; i < count; ++i) {
            assertions.add(trace -> {
                SpanData rootSpan = trace.getSpan(0);
                int requestId = Integer.parseInt(rootSpan.getName().substring("Parent span ".length()));
                trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName(rootSpan.getName()).hasKind(SpanKind.INTERNAL).hasNoParent().hasAttributesSatisfyingExactly(new AttributeAssertion[]{OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.longKey((String)"test.request.id"), (int)requestId)}), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasParent(rootSpan), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(1)).hasAttributesSatisfyingExactly(new AttributeAssertion[]{OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.longKey((String)"test.request.id"), (int)requestId)}), span -> span.hasName("child").hasKind(SpanKind.INTERNAL).hasParent(rootSpan)});
            });
        }
        this.testing.waitAndAssertTraces(assertions);
        pool.shutdown();
    }

    @Test
    void highConcurrencyOnSingleConnection() {
        SingleConnection singleConnection = this.options.getSingleConnectionFactory().apply("localhost", this.server.httpPort());
        Assumptions.assumeTrue((singleConnection != null ? 1 : 0) != 0);
        int count = 50;
        String method = "GET";
        String path = "/success";
        URI uri = this.resolveAddress(path);
        CountDownLatch latch = new CountDownLatch(1);
        ExecutorService pool = Executors.newFixedThreadPool(4);
        int i = 0;
        while (i < count) {
            int index = i++;
            Runnable job = () -> {
                try {
                    latch.await();
                }
                catch (InterruptedException e) {
                    throw new AssertionError((Object)e);
                }
                try {
                    Integer result = this.testing.runWithSpan("Parent span " + index, () -> {
                        Span.current().setAttribute("test.request.id", (long)index);
                        return singleConnection.doRequest(path, Collections.singletonMap("test-request-id", String.valueOf(index)));
                    });
                    OpenTelemetryAssertions.assertThat((Integer)result).isEqualTo(200);
                }
                catch (Throwable throwable) {
                    if (throwable instanceof AssertionError) {
                        throw (AssertionError)((Object)throwable);
                    }
                    throw new AssertionError((Object)throwable);
                }
            };
            pool.submit(job);
        }
        latch.countDown();
        ArrayList<Consumer<TraceAssert>> assertions = new ArrayList<Consumer<TraceAssert>>();
        for (int i2 = 0; i2 < count; ++i2) {
            assertions.add(trace -> {
                SpanData rootSpan = trace.getSpan(0);
                int requestId = Integer.parseInt(rootSpan.getName().substring("Parent span ".length()));
                trace.hasSpansSatisfyingExactly(new Consumer[]{span -> span.hasName(rootSpan.getName()).hasKind(SpanKind.INTERNAL).hasNoParent().hasAttributesSatisfyingExactly(new AttributeAssertion[]{OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.longKey((String)"test.request.id"), (int)requestId)}), span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasParent(rootSpan), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(1)).hasAttributesSatisfyingExactly(new AttributeAssertion[]{OpenTelemetryAssertions.equalTo((AttributeKey)AttributeKey.longKey((String)"test.request.id"), (int)requestId)})});
            });
        }
        this.testing.waitAndAssertTraces(assertions);
        pool.shutdown();
    }

    @Test
    void spanEndsAfterBodyReceived() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.isSpanEndsAfterBody());
        Assumptions.assumeTrue((boolean)this.options.getHasSendRequest());
        String method = "GET";
        URI uri = this.resolveAddress("/long-request");
        int responseCode = this.doRequest(method, uri, Collections.singletonMap("delay", String.valueOf(TimeUnit.SECONDS.toMillis(1L))));
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> {
            trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasNoParent().hasStatus(StatusData.unset()), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))});
            SpanData span2 = trace.getSpan(0);
            ((AbstractBooleanAssert)OpenTelemetryAssertions.assertThat((span2.getEndEpochNanos() - span2.getStartEpochNanos() >= TimeUnit.SECONDS.toNanos(1L) ? 1 : 0) != 0).describedAs("Span duration should be at least 1s", new Object[0])).isTrue();
        });
    }

    @Test
    void spanEndsAfterHeadersReceived() throws Exception {
        Assumptions.assumeTrue((boolean)this.options.isSpanEndsAfterHeaders());
        String method = "GET";
        URI uri = this.resolveAddress("/long-request");
        int responseCode = this.doRequest(method, uri, Collections.singletonMap("delay", String.valueOf(TimeUnit.SECONDS.toMillis(2L))));
        OpenTelemetryAssertions.assertThat((int)responseCode).isEqualTo(200);
        this.testing.waitAndAssertTraces(trace -> {
            trace.hasSpansSatisfyingExactly(new Consumer[]{span -> this.assertClientSpan((SpanDataAssert)span, uri, method, 200, null).hasNoParent().hasStatus(StatusData.unset()), span -> AbstractHttpClientTest.assertServerSpan(span).hasParent(trace.getSpan(0))});
            SpanData span2 = trace.getSpan(0);
            ((AbstractBooleanAssert)OpenTelemetryAssertions.assertThat((span2.getEndEpochNanos() - span2.getStartEpochNanos() <= TimeUnit.SECONDS.toNanos(2L) ? 1 : 0) != 0).describedAs("Span duration should be less than 2s", new Object[0])).isTrue();
        });
    }

    SpanDataAssert assertClientSpan(SpanDataAssert span, URI uri, String method, @Nullable Integer responseCode, @Nullable Integer resendCount) {
        Set<AttributeKey<?>> httpClientAttributes = this.options.getHttpAttributes().apply(uri);
        return span.hasName(this.options.getExpectedClientSpanNameMapper().apply(uri, method)).hasKind(SpanKind.CLIENT).hasAttributesSatisfying(attrs -> {
            OpenTelemetryAssertions.assertThat((Attributes)attrs).doesNotContainKey(NetworkAttributes.NETWORK_TRANSPORT).doesNotContainKey(NetworkAttributes.NETWORK_TYPE).doesNotContainKey(NetworkAttributes.NETWORK_PROTOCOL_NAME);
            if (httpClientAttributes.contains(NetworkAttributes.NETWORK_PROTOCOL_VERSION)) {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(NetworkAttributes.NETWORK_PROTOCOL_VERSION, (Object)this.options.getHttpProtocolVersion().apply(uri));
            }
            if (httpClientAttributes.contains(ServerAttributes.SERVER_ADDRESS)) {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(ServerAttributes.SERVER_ADDRESS, (Object)uri.getHost());
            }
            if (httpClientAttributes.contains(ServerAttributes.SERVER_PORT)) {
                int uriPort = uri.getPort();
                if (uriPort <= 0) {
                    if (attrs.get(ServerAttributes.SERVER_PORT) != null) {
                        int effectivePort = "https".equals(uri.getScheme()) ? 443 : 80;
                        OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(ServerAttributes.SERVER_PORT, effectivePort);
                    }
                } else {
                    OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(ServerAttributes.SERVER_PORT, uriPort);
                }
            }
            if (uri.getPort() != 61 && !uri.getHost().equals("192.0.2.1")) {
                if (attrs.get(NetworkAttributes.NETWORK_PEER_ADDRESS) != null) {
                    OpenTelemetryAssertions.assertThat((Attributes)attrs).hasEntrySatisfying(NetworkAttributes.NETWORK_PEER_ADDRESS, addr -> OpenTelemetryAssertions.assertThat((String)addr).isIn(new Object[]{"127.0.0.1", "0:0:0:0:0:0:0:1"}));
                }
                if (attrs.get(NetworkAttributes.NETWORK_PEER_PORT) != null) {
                    OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(NetworkAttributes.NETWORK_PEER_PORT, Objects.equals(uri.getScheme(), "https") ? this.server.httpsPort() : this.server.httpPort());
                }
            }
            if (httpClientAttributes.contains(UrlAttributes.URL_FULL)) {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(UrlAttributes.URL_FULL, (Object)uri.toString());
            }
            if (httpClientAttributes.contains(HttpAttributes.HTTP_REQUEST_METHOD)) {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(HttpAttributes.HTTP_REQUEST_METHOD, (Object)method);
            }
            OpenTelemetryAssertions.assertThat((Attributes)attrs).doesNotContainKey(UserAgentAttributes.USER_AGENT_ORIGINAL);
            if (responseCode != null) {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(HttpAttributes.HTTP_RESPONSE_STATUS_CODE, (Object)responseCode);
                if (responseCode >= 400) {
                    OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(ErrorAttributes.ERROR_TYPE, (Object)String.valueOf(responseCode));
                }
            } else {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).doesNotContainKey(HttpAttributes.HTTP_RESPONSE_STATUS_CODE);
                OpenTelemetryAssertions.assertThat((Attributes)attrs).containsKey(ErrorAttributes.ERROR_TYPE);
            }
            if (resendCount != null) {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).containsEntry(HttpAttributes.HTTP_REQUEST_RESEND_COUNT, (Object)resendCount);
            } else {
                OpenTelemetryAssertions.assertThat((Attributes)attrs).doesNotContainKey(HttpAttributes.HTTP_REQUEST_RESEND_COUNT);
            }
        });
    }

    static SpanDataAssert assertServerSpan(SpanDataAssert span) {
        return span.hasName("test-http-server").hasKind(SpanKind.SERVER);
    }

    private int doRequest(String method, URI uri) throws Exception {
        return this.doRequest(method, uri, Collections.emptyMap());
    }

    private int doRequest(String method, URI uri, Map<String, String> headers) throws Exception {
        Object request = this.buildRequest(method, uri, headers);
        return this.sendRequest(request, method, uri, headers);
    }

    private int doReusedRequest(String method, URI uri) throws Exception {
        Object request = this.buildRequest(method, uri, Collections.emptyMap());
        this.sendRequest(request, method, uri, Collections.emptyMap());
        return this.sendRequest(request, method, uri, Collections.emptyMap());
    }

    private int doRequestWithExistingTracingHeaders(String method, URI uri) throws Exception {
        HashMap<String, String> headers = new HashMap<String, String>();
        for (String field : this.testing.getOpenTelemetry().getPropagators().getTextMapPropagator().fields()) {
            headers.put(field, "12345789");
        }
        Object request = this.buildRequest(method, uri, headers);
        return this.sendRequest(request, method, uri, headers);
    }

    private HttpClientResult doRequestWithCallback(String method, URI uri, Runnable callback) throws Exception {
        return this.doRequestWithCallback(method, uri, Collections.emptyMap(), callback);
    }

    private HttpClientResult doRequestWithCallback(String method, URI uri, Map<String, String> headers, Runnable callback) throws Exception {
        Object request = this.buildRequest(method, uri, headers);
        HttpClientResult httpClientResult = new HttpClientResult(callback);
        this.sendRequestWithCallback(request, method, uri, headers, httpClientResult);
        return httpClientResult;
    }

    protected final URI resolveAddress(String path) {
        return URI.create("http://localhost:" + this.server.httpPort() + path);
    }
}

