/*
 * Decompiled with CFR 0.152.
 */
package org.opensaml.xml.security.keyinfo.provider;

import java.math.BigInteger;
import java.security.PublicKey;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.security.auth.x500.X500Principal;
import org.opensaml.xml.XMLObject;
import org.opensaml.xml.security.CriteriaSet;
import org.opensaml.xml.security.SecurityException;
import org.opensaml.xml.security.credential.Credential;
import org.opensaml.xml.security.keyinfo.KeyInfoCredentialContext;
import org.opensaml.xml.security.keyinfo.KeyInfoCredentialResolver;
import org.opensaml.xml.security.keyinfo.KeyInfoHelper;
import org.opensaml.xml.security.keyinfo.KeyInfoResolutionContext;
import org.opensaml.xml.security.keyinfo.provider.AbstractKeyInfoProvider;
import org.opensaml.xml.security.x509.BasicX509Credential;
import org.opensaml.xml.security.x509.InternalX500DNHandler;
import org.opensaml.xml.security.x509.X500DNHandler;
import org.opensaml.xml.security.x509.X509Util;
import org.opensaml.xml.signature.X509Data;
import org.opensaml.xml.signature.X509Digest;
import org.opensaml.xml.signature.X509IssuerSerial;
import org.opensaml.xml.signature.X509SKI;
import org.opensaml.xml.signature.X509SubjectName;
import org.opensaml.xml.util.Base64;
import org.opensaml.xml.util.DatatypeHelper;
import org.opensaml.xml.util.LazySet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InlineX509DataProvider
extends AbstractKeyInfoProvider {
    private final Logger log = LoggerFactory.getLogger(InlineX509DataProvider.class);
    private X500DNHandler x500DNHandler = new InternalX500DNHandler();

    public X500DNHandler getX500DNHandler() {
        return this.x500DNHandler;
    }

    public void setX500DNHandler(X500DNHandler handler) {
        if (handler == null) {
            throw new IllegalArgumentException("X500DNHandler may not be null");
        }
        this.x500DNHandler = handler;
    }

    @Override
    public boolean handles(XMLObject keyInfoChild) {
        return keyInfoChild instanceof X509Data;
    }

    @Override
    public Collection<Credential> process(KeyInfoCredentialResolver resolver, XMLObject keyInfoChild, CriteriaSet criteriaSet, KeyInfoResolutionContext kiContext) throws SecurityException {
        KeyInfoCredentialContext credContext;
        X509Certificate entityCert;
        if (!this.handles(keyInfoChild)) {
            return null;
        }
        X509Data x509Data = (X509Data)keyInfoChild;
        this.log.debug("Attempting to extract credential from an X509Data");
        List<X509Certificate> certs = this.extractCertificates(x509Data);
        if (certs.isEmpty()) {
            this.log.info("The X509Data contained no X509Certificate elements, skipping credential extraction");
            return null;
        }
        List<X509CRL> crls = this.extractCRLs(x509Data);
        PublicKey resolvedPublicKey = null;
        if (kiContext != null && kiContext.getKey() != null && kiContext.getKey() instanceof PublicKey) {
            resolvedPublicKey = (PublicKey)kiContext.getKey();
        }
        if ((entityCert = this.findEntityCert(certs, x509Data, resolvedPublicKey)) == null) {
            this.log.warn("The end-entity cert could not be identified, skipping credential extraction");
            return null;
        }
        BasicX509Credential cred = new BasicX509Credential();
        cred.setEntityCertificate(entityCert);
        cred.setCRLs(crls);
        cred.setEntityCertificateChain(certs);
        if (kiContext != null) {
            cred.getKeyNames().addAll(kiContext.getKeyNames());
        }
        if ((credContext = this.buildCredentialContext(kiContext)) != null) {
            cred.getCredentalContextSet().add(credContext);
        }
        LazySet<Credential> credentialSet = new LazySet<Credential>();
        credentialSet.add(cred);
        return credentialSet;
    }

    private List<X509CRL> extractCRLs(X509Data x509Data) throws SecurityException {
        List<X509CRL> crls = null;
        try {
            crls = KeyInfoHelper.getCRLs(x509Data);
        }
        catch (CRLException e) {
            this.log.error("Error extracting CRL's from X509Data", (Throwable)e);
            throw new SecurityException("Error extracting CRL's from X509Data", e);
        }
        this.log.debug("Found {} X509CRLs", (Object)crls.size());
        return crls;
    }

    private List<X509Certificate> extractCertificates(X509Data x509Data) throws SecurityException {
        List<X509Certificate> certs = null;
        try {
            certs = KeyInfoHelper.getCertificates(x509Data);
        }
        catch (CertificateException e) {
            this.log.error("Error extracting certificates from X509Data", (Throwable)e);
            throw new SecurityException("Error extracting certificates from X509Data", e);
        }
        this.log.debug("Found {} X509Certificates", (Object)certs.size());
        return certs;
    }

    protected X509Certificate findEntityCert(List<X509Certificate> certs, X509Data x509Data, PublicKey resolvedKey) {
        if (certs == null || certs.isEmpty()) {
            return null;
        }
        if (certs.size() == 1) {
            this.log.debug("Single certificate was present, treating as end-entity certificate");
            return certs.get(0);
        }
        X509Certificate cert = null;
        cert = this.findCertFromKey(certs, resolvedKey);
        if (cert != null) {
            this.log.debug("End-entity certificate resolved by matching previously resolved public key");
            return cert;
        }
        cert = this.findCertFromSubjectNames(certs, x509Data.getX509SubjectNames());
        if (cert != null) {
            this.log.debug("End-entity certificate resolved by matching X509SubjectName");
            return cert;
        }
        cert = this.findCertFromIssuerSerials(certs, x509Data.getX509IssuerSerials());
        if (cert != null) {
            this.log.debug("End-entity certificate resolved by matching X509IssuerSerial");
            return cert;
        }
        cert = this.findCertFromSubjectKeyIdentifier(certs, x509Data.getX509SKIs());
        if (cert != null) {
            this.log.debug("End-entity certificate resolved by matching X509SKI");
            return cert;
        }
        cert = this.findCertFromDigest(certs, x509Data.getXMLObjects(X509Digest.DEFAULT_ELEMENT_NAME));
        if (cert != null) {
            this.log.debug("End-entity certificate resolved by matching X509Digest");
            return cert;
        }
        this.log.debug("Treating the first certificate in the X509Data as the end-entity certificate");
        return certs.get(0);
    }

    protected X509Certificate findCertFromKey(List<X509Certificate> certs, PublicKey key) {
        if (key != null) {
            for (X509Certificate cert : certs) {
                if (!cert.getPublicKey().equals(key)) continue;
                return cert;
            }
        }
        return null;
    }

    protected X509Certificate findCertFromSubjectNames(List<X509Certificate> certs, List<X509SubjectName> names) {
        for (X509SubjectName subjectName : names) {
            if (DatatypeHelper.isEmpty(subjectName.getValue())) continue;
            X500Principal subjectX500Principal = null;
            try {
                subjectX500Principal = this.x500DNHandler.parse(subjectName.getValue());
            }
            catch (IllegalArgumentException e) {
                this.log.warn("X500 subject name '{}' could not be parsed by configured X500DNHandler '{}'", (Object)subjectName.getValue(), (Object)this.x500DNHandler.getClass().getName());
                return null;
            }
            for (X509Certificate cert : certs) {
                if (!cert.getSubjectX500Principal().equals(subjectX500Principal)) continue;
                return cert;
            }
        }
        return null;
    }

    protected X509Certificate findCertFromIssuerSerials(List<X509Certificate> certs, List<X509IssuerSerial> serials) {
        for (X509IssuerSerial issuerSerial : serials) {
            if (issuerSerial.getX509IssuerName() == null || issuerSerial.getX509SerialNumber() == null) continue;
            String issuerNameValue = issuerSerial.getX509IssuerName().getValue();
            BigInteger serialNumber = issuerSerial.getX509SerialNumber().getValue();
            if (DatatypeHelper.isEmpty(issuerNameValue)) continue;
            X500Principal issuerX500Principal = null;
            try {
                issuerX500Principal = this.x500DNHandler.parse(issuerNameValue);
            }
            catch (IllegalArgumentException e) {
                this.log.warn("X500 issuer name '{}' could not be parsed by configured X500DNHandler '{}'", (Object)issuerNameValue, (Object)this.x500DNHandler.getClass().getName());
                return null;
            }
            for (X509Certificate cert : certs) {
                if (!cert.getIssuerX500Principal().equals(issuerX500Principal) || !cert.getSerialNumber().equals(serialNumber)) continue;
                return cert;
            }
        }
        return null;
    }

    protected X509Certificate findCertFromSubjectKeyIdentifier(List<X509Certificate> certs, List<X509SKI> skis) {
        for (X509SKI ski : skis) {
            if (DatatypeHelper.isEmpty(ski.getValue())) continue;
            byte[] xmlValue = Base64.decode(ski.getValue());
            for (X509Certificate cert : certs) {
                byte[] certValue = X509Util.getSubjectKeyIdentifier(cert);
                if (certValue == null || !Arrays.equals(xmlValue, certValue)) continue;
                return cert;
            }
        }
        return null;
    }

    protected X509Certificate findCertFromDigest(List<X509Certificate> certs, List<XMLObject> digests) {
        for (XMLObject xo : digests) {
            X509Digest digest;
            if (!(xo instanceof X509Digest) || DatatypeHelper.isEmpty((digest = (X509Digest)xo).getValue())) continue;
            byte[] xmlValue = Base64.decode(digest.getValue());
            for (X509Certificate cert : certs) {
                try {
                    byte[] certValue = X509Util.getX509Digest((X509Certificate)cert, (String)digest.getAlgorithm());
                    if (certValue == null || !Arrays.equals(xmlValue, certValue)) continue;
                    return cert;
                }
                catch (SecurityException securityException) {
                }
            }
        }
        return null;
    }
}

