/*
 * Decompiled with CFR 0.152.
 */
package io.lettuce.core.failover.health;

import io.lettuce.core.ClientOptions;
import io.lettuce.core.RedisCredentials;
import io.lettuce.core.SslOptions;
import io.lettuce.core.failover.health.RedisRestException;
import io.lettuce.core.internal.Exceptions;
import io.lettuce.core.internal.LettuceAssert;
import io.lettuce.core.json.JsonArray;
import io.lettuce.core.json.JsonObject;
import io.lettuce.core.json.JsonParser;
import io.lettuce.core.json.JsonValue;
import io.lettuce.core.support.http.HttpClient;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;

class RedisRestClient {
    private static final InternalLogger log = InternalLoggerFactory.getInstance(RedisRestClient.class);
    private static final int DEFAULT_TIMEOUT_MS = 1000;
    public static final String V_1_BDBS_S_AVAILABILITY = "/v1/bdbs/%s/availability";
    public static final String V_1_BDBS = "/v1/bdbs";
    public static final String AVAILABILITY_EXTEND_CHECK = "extend_check";
    public static final String AVAILABILITY_LAG_TOLERANCE_MS = "availability_lag_tolerance_ms";
    private final URI endpoint;
    private final Supplier<RedisCredentials> credentialsSupplier;
    private final HttpClient.ConnectionConfig connectionConfig;
    private final HttpClient httpClient;
    private final AtomicReference<CompletableFuture<HttpClient.HttpConnection>> connectionFutureRef = new AtomicReference();

    RedisRestClient(URI restEndpoint, Supplier<RedisCredentials> credentialsSupplier, int timeoutMs, SslOptions sslOptions, HttpClient httpClient) {
        LettuceAssert.notNull((Object)restEndpoint, "Redis Enterprise REST API restEndpoint must not be null");
        LettuceAssert.notNull((Object)httpClient, "HttpClient must not be null");
        this.endpoint = restEndpoint;
        this.credentialsSupplier = credentialsSupplier;
        this.httpClient = httpClient;
        HttpClient.ConnectionConfig.Builder configBuilder = HttpClient.ConnectionConfig.builder().connectionTimeout(timeoutMs).readTimeout(timeoutMs);
        if (sslOptions != null) {
            configBuilder.sslOptions(sslOptions);
        }
        this.connectionConfig = configBuilder.build();
    }

    public List<BdbInfo> getBdbs() throws RedisRestException {
        HttpClient.Request request = HttpClient.Request.get(V_1_BDBS).queryParam("fields", "uid,endpoints").headers(this.createAuthHeaders()).build();
        HttpClient.Response response = this.executeWithRetry(request);
        if (response.getStatusCode() >= 200 && response.getStatusCode() < 300) {
            return this.readBdbs(response.getResponseBody(StandardCharsets.UTF_8));
        }
        throw new RedisRestException("Failed to get BDBs", response.getStatusCode(), response.getResponseBody(StandardCharsets.UTF_8));
    }

    public boolean checkBdbAvailability(Long uid, boolean lagAware) {
        return this.checkBdbAvailability(uid, lagAware, null);
    }

    public boolean checkBdbAvailability(Long uid, boolean extendedCheckEnabled, Long availabilityLagToleranceMs) {
        HttpClient.Request request;
        HttpClient.Response response;
        String availabilityPath = String.format(V_1_BDBS_S_AVAILABILITY, uid);
        HttpClient.Request.RequestBuilder requestBuilder = HttpClient.Request.get(availabilityPath).headers(this.createAuthHeaders());
        if (extendedCheckEnabled) {
            requestBuilder.queryParam(AVAILABILITY_EXTEND_CHECK, "lag");
            if (availabilityLagToleranceMs != null) {
                requestBuilder.queryParam(AVAILABILITY_LAG_TOLERANCE_MS, availabilityLagToleranceMs.toString());
            }
        }
        return (response = this.executeWithRetry(request = requestBuilder.build())).getStatusCode() >= 200 && response.getStatusCode() < 300;
    }

    private Map<String, String> createAuthHeaders() {
        if (this.credentialsSupplier == null) {
            return Collections.emptyMap();
        }
        RedisCredentials credentials = this.credentialsSupplier.get();
        if (credentials == null || credentials.getUsername() == null || credentials.getPassword() == null) {
            return Collections.emptyMap();
        }
        String username = credentials.getUsername();
        char[] password = credentials.getPassword();
        CharBuffer combined = CharBuffer.allocate(username.length() + 1 + password.length);
        combined.put(username).put(':').put(password).flip();
        ByteBuffer bytes = StandardCharsets.UTF_8.encode(combined);
        String encodedAuth = Base64.getEncoder().encodeToString(bytes.array());
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Basic " + encodedAuth);
        return headers;
    }

    private HttpClient.Response executeWithRetry(HttpClient.Request request) {
        try {
            return this.executeWithRetryAsync(request).join();
        }
        catch (Exception e) {
            throw RedisRestExceptions.bubble(e);
        }
    }

    private CompletableFuture<HttpClient.Response> executeWithRetryAsync(HttpClient.Request request) {
        return ((CompletableFuture)this.executeRequestAsync(request).handle((response, ex) -> {
            if (ex == null && !this.isRetryableStatus(response.getStatusCode())) {
                return CompletableFuture.completedFuture(response);
            }
            if (ex != null) {
                log.debug("Request failed, retrying after reconnection: {}", (Object)ex.getMessage());
            } else {
                log.debug("Received retryable status code {}, retrying after reconnection", (Object)response.getStatusCode());
            }
            return this.closeConnectionAsync().thenCompose(v -> this.executeRequestAsync(request));
        })).thenCompose(Function.identity());
    }

    private CompletableFuture<HttpClient.Response> executeRequestAsync(HttpClient.Request request) {
        return this.getConnectionAsync().thenCompose(conn -> conn.executeAsync(request));
    }

    private boolean isRetryableStatus(int statusCode) {
        return statusCode == 408 || statusCode == 429 || statusCode == 500 || statusCode == 502 || statusCode == 503 || statusCode == 504;
    }

    private CompletableFuture<HttpClient.HttpConnection> getConnectionAsync() {
        CompletableFuture<HttpClient.HttpConnection> existing;
        while ((existing = this.connectionFutureRef.get()) == null) {
            CompletableFuture<HttpClient.HttpConnection> placeholder = new CompletableFuture<HttpClient.HttpConnection>();
            if (!this.connectionFutureRef.compareAndSet(null, placeholder)) continue;
            this.httpClient.connectAsync(this.endpoint, this.connectionConfig).whenComplete((conn, ex) -> {
                if (ex != null) {
                    this.connectionFutureRef.compareAndSet(placeholder, null);
                    log.error("Failed to establish HTTP connection to {}", (Object)this.endpoint, ex);
                    placeholder.completeExceptionally((Throwable)ex);
                } else {
                    log.debug("Established HTTP connection to {}", (Object)this.endpoint);
                    placeholder.complete((HttpClient.HttpConnection)conn);
                }
            });
            return placeholder;
        }
        return existing;
    }

    private CompletableFuture<Void> closeConnectionAsync() {
        CompletableFuture existing = this.connectionFutureRef.getAndSet(null);
        if (existing == null) {
            return CompletableFuture.completedFuture(null);
        }
        return ((CompletableFuture)existing.handle((conn, ex) -> {
            if (conn != null) {
                return conn.closeAsync().exceptionally(closeEx -> {
                    log.warn("Error closing HTTP connection", closeEx);
                    return null;
                });
            }
            return CompletableFuture.completedFuture(null);
        })).thenCompose(Function.identity());
    }

    public void close() {
        this.closeConnectionAsync().join();
    }

    public CompletableFuture<Void> closeAsync() {
        return this.closeConnectionAsync();
    }

    private List<BdbInfo> readBdbs(String responseBody) {
        JsonParser jsonParser = ClientOptions.DEFAULT_JSON_PARSER.get();
        JsonArray bdbs = jsonParser.createJsonValue(responseBody).asJsonArray();
        ArrayList<BdbInfo> bdbInfoList = new ArrayList<BdbInfo>();
        for (JsonValue bdbElement : bdbs.asList()) {
            JsonObject bdb;
            if (!bdbElement.isJsonObject() || (bdb = bdbElement.asJsonObject()).get("uid") == null) continue;
            Number bdbIdValue = bdb.get("uid").asNumber();
            Long bdbId = bdbIdValue != null ? Long.valueOf(bdbIdValue.longValue()) : null;
            ArrayList<EndpointInfo> endpoints = new ArrayList<EndpointInfo>();
            JsonValue endpointsJson = bdb.get("endpoints");
            if (endpointsJson != null && endpointsJson.isJsonArray()) {
                JsonArray endpointsArray = endpointsJson.asJsonArray();
                for (JsonValue endpointElement : endpointsArray.asList()) {
                    if (!endpointElement.isJsonObject()) continue;
                    JsonObject endpoint = endpointElement.asJsonObject();
                    ArrayList<String> addrList = new ArrayList<String>();
                    JsonValue jsonAddr = endpoint.get("addr");
                    if (jsonAddr != null && jsonAddr.isJsonArray()) {
                        JsonArray addresses = jsonAddr.asJsonArray();
                        for (JsonValue addrElement : addresses.asList()) {
                            if (!addrElement.isString()) continue;
                            addrList.add(addrElement.asString());
                        }
                    }
                    String dnsName = endpoint.get("dns_name") != null ? endpoint.get("dns_name").asString() : null;
                    Integer port = endpoint.get("port") != null ? Integer.valueOf(endpoint.get("port").asNumber().intValue()) : null;
                    String endpointUid = endpoint.get("uid") != null ? endpoint.get("uid").asString() : null;
                    endpoints.add(new EndpointInfo(addrList, dnsName, port, endpointUid));
                }
            }
            bdbInfoList.add(new BdbInfo(bdbId, endpoints));
        }
        return bdbInfoList;
    }

    static final class RedisRestExceptions {
        private RedisRestExceptions() {
        }

        static RuntimeException bubble(Throwable t) {
            Throwable cause = Exceptions.unwrap(t);
            if (cause instanceof InterruptedException) {
                Thread.currentThread().interrupt();
                return new RedisRestException("REST Call was interrupted", cause);
            }
            if (cause instanceof IOException) {
                return new RedisRestException("REST call failed", cause);
            }
            if (cause instanceof RedisRestException) {
                return (RedisRestException)cause;
            }
            if (cause instanceof RuntimeException) {
                return (RuntimeException)cause;
            }
            return new RedisRestException("Unexpected REST failure", cause);
        }
    }

    static class EndpointInfo {
        private final List<String> addr;
        private final String dnsName;
        private final Integer port;
        private final String uid;

        EndpointInfo(List<String> addr, String dnsName, Integer port, String uid) {
            this.addr = addr;
            this.dnsName = dnsName;
            this.port = port;
            this.uid = uid;
        }

        List<String> getAddr() {
            return this.addr;
        }

        String getDnsName() {
            return this.dnsName;
        }

        Integer getPort() {
            return this.port;
        }

        String getUid() {
            return this.uid;
        }

        public String toString() {
            return "EndpointInfo{addr=" + this.addr + ", dnsName='" + this.dnsName + '\'' + ", port=" + this.port + ", uid='" + this.uid + '\'' + '}';
        }
    }

    static class BdbInfo {
        private final Long uid;
        private final List<EndpointInfo> endpoints;

        BdbInfo(Long uid, List<EndpointInfo> endpoints) {
            this.uid = uid;
            this.endpoints = endpoints;
        }

        Long getUid() {
            return this.uid;
        }

        List<EndpointInfo> getEndpoints() {
            return this.endpoints;
        }

        boolean matches(String host) {
            LettuceAssert.notNull((Object)host, "Host must not be null");
            for (EndpointInfo endpoint : this.endpoints) {
                if (host.equals(endpoint.getDnsName())) {
                    return true;
                }
                if (endpoint.getAddr() == null) continue;
                for (String addr : endpoint.getAddr()) {
                    if (!host.equals(addr)) continue;
                    return true;
                }
            }
            return false;
        }

        public String toString() {
            return "BdbInfo{uid='" + this.uid + '\'' + ", endpoints=" + this.endpoints + '}';
        }
    }
}

