/*
 * Decompiled with CFR 0.152.
 */
package io.lettuce.core.failover.health;

import io.lettuce.core.RedisURI;
import io.lettuce.core.failover.health.HealthCheck;
import io.lettuce.core.failover.health.HealthCheckStrategy;
import io.lettuce.core.failover.health.HealthStatus;
import io.lettuce.core.failover.health.HealthStatusChangeEvent;
import io.lettuce.core.failover.health.HealthStatusListener;
import io.lettuce.core.failover.health.ProbingPolicy;
import io.lettuce.core.internal.LettuceAssert;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HealthCheckImpl
implements HealthCheck {
    private static final Logger log = LoggerFactory.getLogger(HealthCheckImpl.class);
    private static final AtomicInteger workerCounter = new AtomicInteger(1);
    private static ExecutorService workers = Executors.newCachedThreadPool(r -> {
        Thread t = new Thread(r, "lettuce-healthcheck-worker-" + workerCounter.getAndIncrement());
        t.setDaemon(true);
        return t;
    });
    private final RedisURI endpoint;
    private final HealthCheckStrategy strategy;
    private final AtomicReference<HealthCheckResult> resultRef = new AtomicReference();
    private final List<HealthStatusListener> listeners = new CopyOnWriteArrayList<HealthStatusListener>();
    private final ScheduledExecutorService scheduler;
    private ScheduledFuture<?> scheduledTask;

    HealthCheckImpl(RedisURI endpoint, HealthCheckStrategy strategy) {
        LettuceAssert.isTrue(strategy.getNumProbes() > 0, "Number of HealthCheckStrategy probes must be greater than 0");
        this.endpoint = endpoint;
        this.strategy = strategy;
        this.resultRef.set(new HealthCheckResult(0L, HealthStatus.UNKNOWN));
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "lettuce-healthcheck-" + this.endpoint);
            t.setDaemon(true);
            return t;
        });
        log.debug("Created health check for {}", (Object)endpoint);
    }

    @Override
    public RedisURI getEndpoint() {
        return this.endpoint;
    }

    @Override
    public HealthStatus getStatus() {
        return this.resultRef.get().getStatus();
    }

    @Override
    public void start() {
        this.scheduledTask = this.scheduler.scheduleAtFixedRate(this::healthCheck, 0L, this.strategy.getInterval(), TimeUnit.MILLISECONDS);
    }

    @Override
    public void stop() {
        this.strategy.close();
        this.listeners.clear();
        if (this.scheduledTask != null) {
            this.scheduledTask.cancel(false);
        }
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(1L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    private HealthStatus doHealthCheck() {
        HealthStatus newStatus = this.strategy.doHealthCheck(this.endpoint);
        log.trace("Health check completed for {} with status {}", (Object)this.endpoint, (Object)newStatus);
        return newStatus;
    }

    private void healthCheck() {
        long me = System.currentTimeMillis();
        HealthStatus update = null;
        HealthProbeContext probeContext = new HealthProbeContext(this.strategy.getPolicy(), this.strategy.getNumProbes());
        while (!probeContext.isCompleted()) {
            Future<HealthStatus> future = workers.submit(this::doHealthCheck);
            try {
                update = future.get(this.strategy.getTimeout(), TimeUnit.MILLISECONDS);
                probeContext.record(update == HealthStatus.HEALTHY);
            }
            catch (ExecutionException | TimeoutException e) {
                future.cancel(true);
                if (log.isWarnEnabled()) {
                    log.warn(String.format("Health check timed out or failed for %s.", this.endpoint), (Throwable)e);
                }
                probeContext.record(false);
            }
            catch (InterruptedException e) {
                future.cancel(true);
                Thread.currentThread().interrupt();
                log.warn(String.format("Health check interrupted for %s.", this.endpoint), (Throwable)e);
                return;
            }
            if (probeContext.isCompleted()) continue;
            try {
                Thread.sleep(this.strategy.getDelayInBetweenProbes());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                log.warn(String.format("Health check interrupted while sleeping for %s.", this.endpoint), (Throwable)e);
                return;
            }
        }
        this.safeUpdate(me, probeContext.getResult());
    }

    void safeUpdate(long owner, HealthStatus status) {
        HealthCheckResult newResult = new HealthCheckResult(owner, status);
        AtomicBoolean wasUpdated = new AtomicBoolean(false);
        HealthCheckResult oldResult = this.resultRef.getAndUpdate(current -> {
            if (current.getTimestamp() < owner) {
                wasUpdated.set(true);
                return newResult;
            }
            wasUpdated.set(false);
            return current;
        });
        if (wasUpdated.get() && oldResult.getStatus() != status) {
            if (log.isInfoEnabled()) {
                log.info("Health status changed for {} from {} to {}", new Object[]{this.endpoint, oldResult.getStatus(), status});
            }
            this.notifyListeners(oldResult.getStatus(), status);
        }
    }

    private void notifyListeners(HealthStatus oldStatus, HealthStatus newStatus) {
        if (!this.listeners.isEmpty()) {
            HealthStatusChangeEvent event = new HealthStatusChangeEvent(this.endpoint, oldStatus, newStatus);
            for (HealthStatusListener listener : this.listeners) {
                try {
                    listener.onStatusChange(event);
                }
                catch (Exception e) {
                    log.error("Error notifying health status listener for endpoint {}", (Object)this.endpoint, (Object)e);
                }
            }
        }
    }

    @Override
    public long getMaxWaitFor() {
        return (long)(this.strategy.getTimeout() + this.strategy.getDelayInBetweenProbes()) * (long)this.strategy.getNumProbes();
    }

    @Override
    public void addListener(HealthStatusListener listener) {
        LettuceAssert.notNull((Object)listener, "HealthStatusListener must not be null");
        this.listeners.add(listener);
    }

    @Override
    public void removeListener(HealthStatusListener listener) {
        LettuceAssert.notNull((Object)listener, "HealthStatusListener must not be null");
        this.listeners.remove(listener);
    }

    private static class HealthCheckResult {
        private final long timestamp;
        private final HealthStatus status;

        public HealthCheckResult(long timestamp, HealthStatus status) {
            this.timestamp = timestamp;
            this.status = status;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public HealthStatus getStatus() {
            return this.status;
        }
    }

    static class HealthProbeContext
    implements ProbingPolicy.ProbeContext {
        private final ProbingPolicy policy;
        private int remainingProbes;
        private int successes;
        private int fails;
        private boolean isCompleted;
        private HealthStatus result;

        HealthProbeContext(ProbingPolicy policy, int maxProbes) {
            this.policy = policy;
            this.remainingProbes = maxProbes;
        }

        void record(boolean success) {
            if (success) {
                ++this.successes;
            } else {
                ++this.fails;
            }
            --this.remainingProbes;
            ProbingPolicy.Decision decision = this.policy.evaluate(this);
            if (decision == ProbingPolicy.Decision.SUCCESS) {
                this.setCompleted(HealthStatus.HEALTHY);
            } else if (decision == ProbingPolicy.Decision.FAIL) {
                this.setCompleted(HealthStatus.UNHEALTHY);
            }
        }

        @Override
        public int getRemainingProbes() {
            return this.remainingProbes;
        }

        @Override
        public int getSuccesses() {
            return this.successes;
        }

        @Override
        public int getFails() {
            return this.fails;
        }

        void setCompleted(HealthStatus status) {
            this.result = status;
            this.isCompleted = true;
        }

        boolean isCompleted() {
            return this.isCompleted;
        }

        HealthStatus getResult() {
            return this.result;
        }
    }
}

