/*
 * Copyright 2025, Redis Ltd. and Contributors
 * All rights reserved.
 *
 * Licensed under the MIT License.
 */

package io.lettuce.core.search.arguments;

import io.lettuce.core.protocol.CommandArgs;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import static io.lettuce.core.protocol.CommandKeyword.*;

/**
 * Field arguments for VECTOR fields in a RediSearch index.
 * <p>
 * Vector fields are floating-point vectors that are typically generated by external machine learning models. These vectors
 * represent unstructured data such as text, images, or other complex features. Redis allows you to search for similar vectors
 * using vector search algorithms like cosine similarity, Euclidean distance, and inner product.
 *
 * @param <K> Key type
 * @see <a href=
 *      "https://redis.io/docs/latest/develop/interact/search-and-query/basic-constructs/field-and-type-options/#vector-fields">Vector
 *      Fields</a>
 * @since 6.8
 * @author Tihomir Mateev
 */
@SuppressWarnings("OptionalUsedAsFieldOrParameterType")
public class VectorFieldArgs<K> extends FieldArgs<K> {

    /**
     * Vector similarity index algorithms.
     */
    public enum Algorithm {

        /**
         * Brute force algorithm.
         */
        FLAT,
        /**
         * Hierarchical, navigable, small world algorithm.
         */
        HNSW,
        /**
         * SVS-VAMANA algorithm provides high-performance approximate vector search optimized for specific use cases with
         * advanced compression and optimization features.
         *
         * <p>
         * Characteristics:
         * <ul>
         * <li>High-performance approximate search</li>
         * <li>Support for vector compression (LVQ, LeanVec)</li>
         * <li>Configurable graph construction and search parameters</li>
         * <li>Optimized for Intel platforms with fallback support</li>
         * </ul>
         *
         * <p>
         * Note: This algorithm may have specific requirements and limitations. Consult the Redis documentation for detailed
         * usage guidelines.
         *
         * @since Redis 8.2
         */
        SVS_VAMANA("SVS-VAMANA");

        private final String redisName;

        Algorithm() {
            this.redisName = name();
        }

        Algorithm(String redisName) {
            this.redisName = redisName;
        }

        @Override
        public String toString() {
            return redisName;
        }

    }

    /**
     * Vector data types.
     */
    public enum VectorType {
        /**
         * 16-bit brain floating point. Requires RediSearch v2.10 or later.
         */
        BFLOAT16,
        /**
         * 16-bit floating point. Requires RediSearch v2.10 or later.
         */
        FLOAT16,
        /**
         * 32-bit floating point.
         */
        FLOAT32,
        /**
         * 64-bit floating point.
         */
        FLOAT64
    }

    /**
     * Distance metrics for vector similarity.
     */
    public enum DistanceMetric {
        /**
         * Euclidean distance (L2 norm).
         */
        L2,
        /**
         * Cosine similarity.
         */
        COSINE,
        /**
         * Inner product.
         */
        IP
    }

    private Optional<Algorithm> algorithm = Optional.empty();

    private final Map<String, Object> attributes = new HashMap<>();

    /**
     * Create a new {@link VectorFieldArgs} using the builder pattern.
     * 
     * @param <K> Key type
     * @return a new {@link Builder}
     */
    public static <K> Builder<K> builder() {
        return new Builder<>();
    }

    @Override
    public String getFieldType() {
        return "VECTOR";
    }

    /**
     * Get the vector algorithm.
     *
     * @return the algorithm
     */
    public Optional<Algorithm> getAlgorithm() {
        return algorithm;
    }

    /**
     * Get the vector attributes.
     *
     * @return the attributes
     */
    public Map<String, Object> getAttributes() {
        return new HashMap<>(attributes);
    }

    @Override
    protected void buildTypeSpecificArgs(CommandArgs<K, ?> args) {
        algorithm.ifPresent(alg -> args.add(alg.toString()));

        if (!attributes.isEmpty()) {
            args.add(String.valueOf(attributes.size() * 2)); // count of attribute pairs
            attributes.forEach((key, value) -> {
                args.add(key);
                args.add(value.toString());
            });
        }
    }

    /**
     * Builder for {@link VectorFieldArgs}.
     * 
     * @param <K> Key type
     */
    public static class Builder<K> extends FieldArgs.Builder<K, VectorFieldArgs<K>, Builder<K>> {

        public Builder() {
            super(new VectorFieldArgs<>());
        }

        /**
         * Set the vector similarity index algorithm.
         * 
         * @param algorithm the algorithm
         * @return the instance of the {@link Builder} for the purpose of method chaining
         */
        public Builder<K> algorithm(Algorithm algorithm) {
            instance.algorithm = Optional.of(algorithm);
            return self();
        }

        /**
         * Use the FLAT (brute force) algorithm.
         * 
         * @return the instance of the {@link Builder} for the purpose of method chaining
         */
        public Builder<K> flat() {
            return algorithm(Algorithm.FLAT);
        }

        /**
         * Use the HNSW (hierarchical, navigable, small world) algorithm.
         *
         * @return the instance of the {@link Builder} for the purpose of method chaining
         */
        public Builder<K> hnsw() {
            return algorithm(Algorithm.HNSW);
        }

        /**
         * Use the SVS-VAMANA algorithm for high-performance approximate vector search.
         *
         * <p>
         * SVS-VAMANA provides advanced features including:
         * <ul>
         * <li>Vector compression support (LVQ, LeanVec)</li>
         * <li>Configurable graph construction parameters</li>
         * <li>Runtime search optimization</li>
         * <li>Intel platform optimizations</li>
         * </ul>
         *
         * @return the instance of the {@link Builder} for the purpose of method chaining
         * @since Redis 8.2
         */
        public Builder<K> svsVamana() {
            return algorithm(Algorithm.SVS_VAMANA);
        }

        /**
         * Set the vector data type.
         * 
         * @param type the vector data type
         * @return the instance of the {@link Builder} for the purpose of method chaining
         */
        public Builder<K> type(VectorType type) {
            instance.attributes.put(TYPE.toString(), type.toString());
            return self();
        }

        /**
         * Set the vector dimensionality.
         * 
         * @param dimensions the number of dimensions
         * @return the instance of the {@link Builder} for the purpose of method chaining
         */
        public Builder<K> dimensions(int dimensions) {
            instance.attributes.put(DIM.toString(), dimensions);
            return self();
        }

        /**
         * Set the distance metric.
         * 
         * @param metric the distance metric
         * @return the instance of the {@link Builder} for the purpose of method chaining
         */
        public Builder<K> distanceMetric(DistanceMetric metric) {
            instance.attributes.put(DISTANCE_METRIC.toString(), metric.toString());
            return self();
        }

        /**
         * Add a custom attribute.
         *
         * @param name the attribute name
         * @param value the attribute value
         * @return the instance of the {@link Builder} for the purpose of method chaining
         */
        public Builder<K> attribute(String name, Object value) {
            instance.attributes.put(name, value);
            return self();
        }

    }

}
