package io.intercom.android.nexus;

import androidx.annotation.Nullable;

import com.intercom.twig.BuildConfig;
import com.intercom.twig.Twig;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;
import java.util.Random;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import okio.ByteString;

class NexusSocket {
    private static final int OK_CLIENT_DISCONNECT = 4000;
    private static final int N_TIMEOUT_DISCONNECT = 4001;
    private static final int MAX_RECONNECT_TIME_SECONDS = 900;
    private static final String HEADER = "?X-Nexus-Version=android." + BuildConfig.VERSION_NAME;

    private final NexusListener listener;
    private final NexusTopicProvider topicProvider;
    private final ScheduledExecutorService timeoutExecutor;
    private final OkHttpClient client;
    private final Twig twig;
    private final String url;
    private final long connectionTimeoutSeconds;
    private final boolean shouldSendPresence;

    private WebSocket socket = CLOSED_SOCKET;

    // reconnectTimout will be null if there is no ongoing reconnection
    @Nullable private ScheduledFuture reconnectFuture;
    // timeoutFuture can be null if the connection hasn't been set up yet
    @Nullable private ScheduledFuture timeoutFuture;
    private final Runnable timeoutRunnable = new Runnable() {
        @Override public void run() {
            timedOut();
        }
    };

    private long lastReconnectAt = 0;
    private int reconnectAttempts = 0;

    NexusSocket(String url, int connectionTimeoutSeconds, boolean shouldSendPresence, Twig twig,
                ScheduledExecutorService timeoutExecutor, OkHttpClient client,
                NexusListener listener, NexusTopicProvider topicProvider) {
        this.url = url;
        this.connectionTimeoutSeconds = connectionTimeoutSeconds;
        this.shouldSendPresence = shouldSendPresence;
        this.twig = twig;
        this.listener = listener;
        this.topicProvider = topicProvider;
        this.client = client;
        this.timeoutExecutor = timeoutExecutor;
    }

    void connect() {
        twig.d("connecting to a socket...");

        Request request = new Request.Builder()
                .url(url + HEADER)
                .build();

        client.newWebSocket(request, webSocketListener);
        timeoutFuture = timeoutExecutor.schedule(timeoutRunnable, connectionTimeoutSeconds, TimeUnit.SECONDS);
    }

    void fire(final String data) {
        if (data.isEmpty()) {
            return;
        }
        try {
            twig.internal("firing: " + data);
            socket.send(data);
        } catch (IllegalStateException e) {
            twig.internal("Error when firing '" + data + "': " + e);
        }
    }

    void disconnect() {
        disconnect(OK_CLIENT_DISCONNECT, "Disconnect called by client");
    }

    boolean isConnected() {
        return socket != CLOSED_SOCKET;
    }

    private void disconnect(final int code, final String reason) {
        boolean closedSuccessfully = socket.close(code, reason);
        if (!closedSuccessfully) {
            twig.internal("Could not close socket while disconnecting, it may be already closed");
        }
    }

    private void timedOut() {
        if (socket == CLOSED_SOCKET) {
            scheduleReconnect();
        } else {
            disconnect(N_TIMEOUT_DISCONNECT, "Socket timed out");
        }
        listener.onConnectFailed();
    }

    private void resetTimeout() {
        if (timeoutFuture != null) {
            timeoutFuture.cancel(true);
        }
        timeoutFuture = timeoutExecutor.schedule(timeoutRunnable, connectionTimeoutSeconds, TimeUnit.SECONDS);
    }

    private void scheduleReconnect() {
        if (reconnectFuture != null) {
            // There's a reconnection attempt in progress
            return;
        }
        modifyReconnectAttempts();

        long delay = calculateReconnectTimerInSeconds(reconnectAttempts);
        twig.internal("Scheduling reconnect in: " + delay + " for attempt: " + reconnectAttempts);

        reconnectFuture = timeoutExecutor.schedule(new Runnable() {
            @Override public void run() {
                connect();
                reconnectFuture = null;
            }
        }, delay, TimeUnit.SECONDS);
    }

    private void modifyReconnectAttempts() {
        long millisSinceLastAttempt = System.currentTimeMillis() - lastReconnectAt;
        long maxReconnectTimeIncludingJitter = TimeUnit.SECONDS.toMillis(MAX_RECONNECT_TIME_SECONDS) * 2;
        if (millisSinceLastAttempt > maxReconnectTimeIncludingJitter) {
            twig.d("resetting reconnection attempts");
            reconnectAttempts = 1;
        } else {
            twig.d("incrementing reconnection attempts");
            reconnectAttempts++;
        }

        lastReconnectAt = System.currentTimeMillis();
    }

    private final WebSocketListener webSocketListener = new WebSocketListener() {

        @Override public void onOpen(WebSocket webSocket, Response response) {
            twig.internal("onOpen: " + response.message());
            socket = webSocket;
            resetTimeout();
            List<String> topics = topicProvider.getTopics();
            if (!topics.isEmpty()) {
                fire(NexusEvent.getSubscribeEvent(topics).toStringEncodedJsonObject());
            }
            if (shouldSendPresence) {
                fire(NexusEvent.getUserPresenceEvent().toStringEncodedJsonObject());
            }
            listener.onConnect();
        }

        @Override public void onMessage(WebSocket webSocket, String text) {
            resetTimeout();
            parseJsonString(text);
        }

        private void parseJsonString(String content) {
            // non-JSON messages are sometimes sent, we don't want to try and parse them
            // messages where content.equals(" ") or content.endsWith("|") are used in long-polling in the web messenger
            if (content.isEmpty() || content.equals(" ") || content.endsWith("|")) {
                return;
            }
            try {
                JSONObject json = new JSONObject(content);
                String eventName = json.optString("eventName");
                if (eventName.isEmpty() || eventName.equals("ACK")) {
                    //todo find out if we're meant to ignore these acks or what
                    twig.internal("onMessage ACK: " + content);
                } else {
                    twig.internal("onMessage TEXT: " + content);
                    listener.notifyEvent(new NexusEvent(json));
                }
            } catch (JSONException e) {
                twig.internal("onMessage: json parse exception for message: '" + content + " " + e);
            }
        }

        @Override public void onMessage(WebSocket webSocket, ByteString bytes) {
            twig.internal("Received bytes message " + bytes + ", resetting timeout");
            resetTimeout();
        }

        @Override public void onClosing(WebSocket webSocket, int code, String reason) {
            // accept close if requested by server
            twig.internal("Server requested close:  " + code + " - '" + reason + "'");
            webSocket.close(code, reason);
        }

        @Override public void onClosed(WebSocket webSocket, int code, String reason) {
            switch (code) {
                case OK_CLIENT_DISCONNECT:
                    shutdown();
                    break;
                default:
                    scheduleReconnect();
                    break;
            }
            twig.internal("onClose code: " + code + " reason: " + reason);
        }

        @Override public void onFailure(WebSocket webSocket, Throwable t, Response response) {
            if (shouldReconnectFromFailure(response)) {
                scheduleReconnect();
            } else {
                shutdown();
            }
            twig.internal("onFailure: " + t.getMessage());
            listener.onConnectFailed();
        }
    };

    private void shutdown() {
        socket = CLOSED_SOCKET;
        if (timeoutFuture != null) {
            timeoutFuture.cancel(true);
        }
        listener.onShutdown();
    }

    static long calculateReconnectTimerInSeconds(int reconnectAttempts) {
        int minimumBackoffSeconds = (int) Math.min(Math.pow(2, reconnectAttempts), MAX_RECONNECT_TIME_SECONDS);
        // Jitter is between 0 and minimumBackoffSeconds seconds inclusive
        int jitter = new Random().nextInt(minimumBackoffSeconds + 1);
        return (long) (minimumBackoffSeconds + jitter);
    }

    static boolean shouldReconnectFromFailure(@Nullable Response response) {
        if (response != null) {
            // This is an error from opening the connection, retry only if it is a 5xx error
            int statusCode = response.code();
            return statusCode >= 500 && statusCode <= 599;
        }
        return true;
    }

    private static final WebSocket CLOSED_SOCKET = new WebSocket() {
        @Override public Request request() {
            throw new NullPointerException("ClosedSocket has no request");
        }

        @Override public long queueSize() {
            return 0;
        }

        @Override public boolean send(String text) {
            return false;
        }

        @Override public boolean send(ByteString bytes) {
            return false;
        }

        @Override public boolean close(int code, String reason) {
            return false;
        }

        @Override public void cancel() {}
    };
}
