package io.intercom.android.sdk.push;

import android.annotation.TargetApi;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.os.Build;
import android.support.annotation.VisibleForTesting;

import com.intercom.twig.Twig;

import java.util.ArrayList;
import java.util.List;

import io.intercom.android.sdk.identity.AppConfig;
import io.intercom.android.sdk.logger.LumberMill;

import static java.util.Collections.synchronizedList;

public class SystemNotificationManager {
    private static final int NOTIFICATION_ID = 9999997;

    @VisibleForTesting final List<PushPayload> receivedPayloads = synchronizedList(new ArrayList<PushPayload>());

    private final Twig twig = LumberMill.getLogger();
    private final SystemNotificationFactory systemNotificationFactory;
    private final NotificationManager androidNotificationManager;

    static final String CHAT_REPLIES_CHANNEL = "intercom_chat_replies_channel";
    static final String NEW_CHATS_CHANNEL = "intercom_new_chats_channel";
    static final String ACTIONS_CHANNEL = "intercom_actions_channel";

    public SystemNotificationManager(NotificationManager androidNotificationManager) {
        this(androidNotificationManager, new SystemNotificationFactory());
    }

    @VisibleForTesting SystemNotificationManager(NotificationManager androidNotificationManager,
                                                 SystemNotificationFactory systemNotificationFactory) {
        this.androidNotificationManager = androidNotificationManager;
        this.systemNotificationFactory = systemNotificationFactory;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            setUpNotificationChannels();
        }
    }

    //TODO localize all these strings
    @TargetApi(Build.VERSION_CODES.O) private void setUpNotificationChannels() {
        NotificationChannel repliesChannel = new NotificationChannel(CHAT_REPLIES_CHANNEL,
                "Chat replies",
                NotificationManager.IMPORTANCE_HIGH);
        repliesChannel.setDescription("Reply notifications from chats in this app");
        androidNotificationManager.createNotificationChannel(repliesChannel);

        NotificationChannel newChatsChannel = new NotificationChannel(NEW_CHATS_CHANNEL,
                "New chats",
                NotificationManager.IMPORTANCE_HIGH);
        newChatsChannel.setDescription("New chat messages sent from this app");
        androidNotificationManager.createNotificationChannel(newChatsChannel);

        NotificationChannel actionsChannel = new NotificationChannel(ACTIONS_CHANNEL,
                "Actions",
                NotificationManager.IMPORTANCE_HIGH);
        actionsChannel.setDescription("Contain links to take an action in this app");
        androidNotificationManager.createNotificationChannel(actionsChannel);
    }

    public void clear() {
        if (!receivedPayloads.isEmpty()) {
            twig.i("Removing Intercom push notifications.");
        }

        androidNotificationManager.cancel(NOTIFICATION_ID);
        receivedPayloads.clear();
    }

    void createPushOnlyNotification(PushPayload payload, Context context) {
        Notification notification = systemNotificationFactory.createPushOnlyNotification(payload, context);
        String pushOnlyConversationId = payload.getPushOnlyConversationId();
        androidNotificationManager.notify(pushOnlyConversationId, pushOnlyConversationId.hashCode(), notification);
    }

    void createNotification(PushPayload payload, Context context, AppConfig appConfig) {
        // Hacky way to check for duplicate pushes.
        // This is caused by the device having multiple working tokens registered
        if (receivedPayloads.contains(payload)) {
            return;
        }

        receivedPayloads.add(payload);

        Notification notification = receivedPayloads.size() == 1
                ? systemNotificationFactory.createSingleNotification(payload, context, appConfig)
                : systemNotificationFactory.createGroupedNotification(receivedPayloads, context);

        androidNotificationManager.notify(NOTIFICATION_ID, notification);
    }
}
