package io.intercom.android.sdk;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.intercom.twig.Twig;

import java.util.HashMap;
import java.util.Map;

import io.intercom.android.sdk.logger.LumberMill;
import io.intercom.android.sdk.utilities.CustomAttributeValidator;

/**
 * <p>The Company object is used for adding companies to users in Intercom.</p>
 * <p>All of the default attributes you can modify are available as methods on the Company.Builder.</p>
 * <p>This is an example of how to create a Company object to update default attributes.</p>
 * <pre>
 * {@code
 * Company company = new Company.Builder()
 *                               .withCompanyId("12345")
 *                               .withName("TestCorp").build();
 * }
 * </pre>
 *
 * <p>You can also add custom attributes to your user.</p>
 *
 * <pre>
 * {@code
 * Company company = new Company.Builder()
 *                               .withCompanyId("12345")
 *                               .withName("TestCorp")
 *                               .withCustomAttribute("employee_count", 200);
 * }
 * </pre>
 *
 * <p>If you want to add multiple custom attributes at once you can pass a Map to the withCustomAttributes method.</p>
 */
public class Company {
    private static final Twig TWIG = LumberMill.getLogger();

    private static final String COMPANY_ID = "id";
    private static final String NAME = "name";
    private static final String CUSTOM_ATTRIBUTES = "custom_attributes";
    private static final String CREATED_AT = "created_at";
    private static final String MONTHLY_SPEND = "monthly_spend";
    private static final String PLAN = "plan";

    private final Map<String, Object> attributes;
    private final Map<String, Object> customAttributes;

    Company(Builder builder) {
        this.attributes = builder.attributes;
        this.customAttributes = builder.customAttributes;
    }

    Map<String, Object> getAttributes() {
        if (!customAttributes.isEmpty()) {
            attributes.put(CUSTOM_ATTRIBUTES, customAttributes);
        }
        return attributes;
    }

    public static final class Builder {
        final Map<String, Object> attributes = new HashMap<>();
        final Map<String, Object> customAttributes = new HashMap<>();

        /**
         * Build the company object to add a company to a user.
         *
         * @return the Company object
         */
        public Company build() {
            return new Company(this);
        }

        /**
         * Set the ID on the company
         *
         * @param companyId the company id you have defined for the company.
         * @return the Company.Builder object
         */
        public Builder withCompanyId(@Nullable String companyId) {
            attributes.put(COMPANY_ID, companyId);
            return this;
        }

        /**
         * Set the name of the company
         *
         * @param name the name of the company
         * @return the Company.Builder object
         */
        public Builder withName(@Nullable String name) {
            attributes.put(NAME, name);
            return this;
        }

        /**
         * Set the date the company was created at
         *
         * @param createdAt a Unix timestamp representing the time the company was created by you.
         * @return the Company.Builder object
         */
        public Builder withCreatedAt(@Nullable Long createdAt) {
            attributes.put(CREATED_AT, createdAt);
            return this;
        }

        /**
         * Set the monthly spend of this company
         *
         * @param monthlySpend how much revenue the company generates for your business.
         * @return the Company.Builder object
         */
        public Builder withMonthlySpend(@Nullable Integer monthlySpend) {
            attributes.put(MONTHLY_SPEND, monthlySpend);
            return this;
        }

        /**
         * Set the companies plan
         *
         * @param plan the name of the plan you have associated with the company.
         * @return the Company.Builder object
         */
        public Builder withPlan(@Nullable String plan) {
            attributes.put(PLAN, plan);
            return this;
        }

        /**
         * Set a custom attribute to the company
         *
         * <p>The key provided must not be null. This method accepts the following types: </p>
         * <p>String, Long, Float, Double, Boolean, Character, Byte, Short and Integer. </p>
         * <p>Any other types will not be added to the update.</p>
         *
         * @param key the name of the attribute
         * @param attribute the value of the attribute
         * @return the Company.Builder object
         */
        public Builder withCustomAttribute(@NonNull String key, @Nullable Object attribute) {
            if (key == null) {
                TWIG.w("The key you provided was null for the attribute " + attribute);
            } else if (CustomAttributeValidator.isValid(attribute)) {
                customAttributes.put(key, attribute);
            } else {
                TWIG.w("The custom company attribute " + key + " was of type "
                        + attribute.getClass().getSimpleName() + " We only accept the following types: "
                        + CustomAttributeValidator.getAcceptedTypes());
            }
            return this;
        }

        /**
         * Set several custom attributes on the company at once
         *
         * <p>This method accepts the following types: </p>
         * <p>String, Long, Float, Double, Boolean, Character, Byte, Short and Integer. </p>
         * <p>Any other types will not be added to the update.</p>
         *
         * @param attributes the map of attributes to update
         * @return the Company.Builder object
         */
        public Builder withCustomAttributes(@NonNull Map<String, ?> attributes) {
            if (attributes == null) {
                TWIG.w("The map of attributes you provided was null.");
            } else {
                for (Map.Entry<String, ?> attribute : attributes.entrySet()) {
                    if (CustomAttributeValidator.isValid(attribute.getValue())) {
                        customAttributes.put(attribute.getKey(), attribute.getValue());
                    } else {
                        TWIG.w("The custom company attribute " + attribute.getKey() + " was of type "
                                + attribute.getClass().getSimpleName() + " We only accept the following types: "
                                + CustomAttributeValidator.getAcceptedTypes());
                    }
                }
            }
            return this;
        }
    }
}
