package io.hengam.lib.datalytics.messages.downstream

import io.hengam.lib.MessageFields.MESSAGE_ID
import io.hengam.lib.messages.MessageType
import io.hengam.lib.messaging.DownstreamMessageParser
import io.hengam.lib.utils.Millis
import io.hengam.lib.utils.Time
import com.squareup.moshi.Json
import com.squareup.moshi.JsonClass
import java.util.*

@JsonClass(generateAdapter = true)
data class GeofenceMessage(
        @Json(name = MESSAGE_ID) val messageId: String,
        @Json(name = "id") val id: String,
        @Json(name = "lat") val lat: Double,
        @Json(name = "long") val long: Double,
        @Json(name = "radius") val radius: Float,
        @Json(name = "expiration_date") val expirationDate: Date? = null,

        /**
         * Determines whether the geofence should be triggered when the user enters the defined
         * region or exits. Should be one of [GEOFENCE_TRIGGER_ENTER] or [GEOFENCE_TRIGGER_EXIT].
         */
        @Json(name = "trigger") val trigger: Int = GEOFENCE_TRIGGER_ENTER,

        /**
         * Determines whether the geofence should trigger if the user is already in the trigger state
         * when the geofence is added. E.g., the [trigger] value is [GEOFENCE_TRIGGER_ENTER] and the
         * user is already in the region.
         *
         * The default value for will be `true` if [trigger] is [GEOFENCE_TRIGGER_ENTER] and `false`
         * if it is [GEOFENCE_TRIGGER_EXIT]
         */
        @Json(name = "trigger_on_init") val triggerOnInit: Boolean? = null,

        /**
         * The minimum amount of time the user should remain in the defined geofence region for it
         * to trigger. If a null or non-positive value is given then the geofence will be triggered
         * immediately (not considering any time lags generated by the system itself, e.g, for battery
         * optimizations).
         *
         * Only works when [trigger] is [GEOFENCE_TRIGGER_ENTER]
         */
        @Json(name = "dwell_time") @Millis val dwellTime: Time? = null,

        /**
         * This value translates to Android Geofence's `Notification Responsiveness` which is "the
         * best-effort notification responsiveness of the geofence". Setting a higher value means
         * that it is ok if the geofence triggers with a delay, but is recommended for saving power.
         */
        @Json(name = "responsiveness") @Millis val responsiveness: Time? = null,

        /**
         * The maximum number of times this geofence should trigger (e.g., the maximum number of times
         * you want a notification to be shown to the user when he/she enters a region repeatedly).
         *
         * The limit is defined for the geofence's `id`  and the geofence trigger counts are counted
         * globally during the lifetime of the user's installation. This means if multiple geofence
         * messages are sent with the same id they will share the trigger counts even if the previous
         * geofences with the same id have already been removed.
         */
        @Json(name = "limit") val limit: Int? = null,

        /**
         * The rate at which the geofence should be allowed to trigger. If the user enters (or exits)
         * the defined region in a period shorter than this time the Geofence will not trigger
         * multiple times.
         *
         * Similar to [limit], the [rateLimit] is defined for the geofence's `id` and the geofence
         * trigger times are stored globally during the lifetime of the user's installation.
         */
        @Json(name = "rate_limit") @Millis val rateLimit: Time? = null,

        /**
         * The message which will be handled by the [PostOffice] once the geofence has been triggered.
         * This will typically be a notification message, e.g., to show a notification once a user
         * enters a region, but it is perfectly valid to use any other type of message here as well.
         *
         * The contents should be in the _parcel_ format. The parcel may also contain multiple
         * messages, e.g., to show a notification and simultaneously perform another action.
         */
        @Json(name = "message") val message: Map<String, Any>
) {
    class Parser : DownstreamMessageParser<GeofenceMessage>(
            MessageType.Datalytics.Downstream.GEOFENCE_ADD,
            { GeofenceMessageJsonAdapter(it) }
    )

    companion object {
        const val GEOFENCE_TRIGGER_ENTER = 1
        const val GEOFENCE_TRIGGER_EXIT = 2
    }
}
