/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.webclient;

import io.helidon.common.configurable.LruCache;
import io.helidon.config.Config;
import io.netty.channel.ChannelHandler;
import io.netty.handler.proxy.HttpProxyHandler;
import io.netty.handler.proxy.Socks4ProxyHandler;
import io.netty.handler.proxy.Socks5ProxyHandler;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.SocketAddress;
import java.net.URI;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Proxy {
    private static final Logger LOGGER = Logger.getLogger(Proxy.class.getName());
    private static final Proxy NO_PROXY = new Proxy(Proxy.builder().type(ProxyType.NONE));
    private static final Pattern PORT_PATTERN = Pattern.compile(".*:(\\d+)");
    private static final Pattern IP_V4 = Pattern.compile("^(25[0-5]|2[0-4]\\d|[0-1]?\\d?\\d)(\\.(25[0-5]|2[0-4]\\d|[0-1]?\\d?\\d)){3}$");
    private static final Pattern IP_V6_IDENTIFIER = Pattern.compile("^\\[(?:[0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}]$");
    private static final Pattern IP_V6_HEX_IDENTIFIER = Pattern.compile("^\\[((?:[0-9A-Fa-f]{1,4}(?::[0-9A-Fa-f]{1,4})*)?)::((?:[0-9A-Fa-f]{1,4}(?::[0-9A-Fa-f]{1,4})*)?)]$");
    private static final Pattern IP_V6_HOST = Pattern.compile("^(?:[0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}$");
    private static final Pattern IP_V6_HEX_HOST = Pattern.compile("^((?:[0-9A-Fa-f]{1,4}(?::[0-9A-Fa-f]{1,4})*)?)::((?:[0-9A-Fa-f]{1,4}(?::[0-9A-Fa-f]{1,4})*)?)$");
    private static final LruCache<String, Boolean> IVP6_HOST_MATCH_RESULTS = LruCache.builder().capacity(100).build();
    private static final LruCache<String, Boolean> IVP6_IDENTIFIER_MATCH_RESULTS = LruCache.builder().capacity(100).build();
    private final ProxyType type;
    private final String host;
    private final int port;
    private final Function<URI, Boolean> noProxy;
    private final Optional<String> username;
    private final Optional<char[]> password;
    private final ProxySelector systemSelector;
    private final boolean useSystemSelector;

    private Proxy(Builder builder) {
        this.type = builder.type();
        this.systemSelector = builder.systemSelector();
        this.host = builder.host();
        this.useSystemSelector = null == this.host && null != this.systemSelector;
        this.port = builder.port();
        this.username = builder.username();
        this.password = builder.password();
        this.noProxy = this.useSystemSelector ? inetSocketAddress -> true : Proxy.prepareNoProxy(builder.noProxyHosts());
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Proxy noProxy() {
        return NO_PROXY;
    }

    public static Proxy create(Config config) {
        return Proxy.builder().config(config).build();
    }

    public static Proxy create() {
        return Proxy.builder().useSystemSelector(true).build();
    }

    Function<URI, Boolean> noProxyPredicate() {
        return this.noProxy;
    }

    static Function<URI, Boolean> prepareNoProxy(Set<String> noProxyHosts) {
        if (noProxyHosts.isEmpty()) {
            return address -> false;
        }
        boolean simple = true;
        for (String noProxyHost : noProxyHosts) {
            if (!noProxyHost.startsWith(".")) continue;
            simple = false;
            break;
        }
        if (simple) {
            return address -> noProxyHosts.contains(address.getHost()) || noProxyHosts.contains(address.getHost() + ":" + address.getPort());
        }
        LinkedList<BiFunction<String, Integer, Boolean>> hostMatchers = new LinkedList<BiFunction<String, Integer, Boolean>>();
        LinkedList<BiFunction<String, Integer, Boolean>> ipMatchers = new LinkedList<BiFunction<String, Integer, Boolean>>();
        Iterator<String> iterator = noProxyHosts.iterator();
        while (iterator.hasNext()) {
            String noProxyHost;
            String hostPart = noProxyHost = iterator.next();
            Integer portPart = null;
            Matcher portMatcher = PORT_PATTERN.matcher(noProxyHost);
            if (portMatcher.matches()) {
                portPart = Integer.parseInt(portMatcher.group(1));
                int index = noProxyHost.lastIndexOf(58);
                hostPart = noProxyHost.substring(0, index);
            }
            if (Proxy.isIpV4(hostPart)) {
                Proxy.exactMatch(ipMatchers, hostPart, portPart);
                continue;
            }
            if (Proxy.isIpV6Identifier(hostPart)) {
                if ("[::1]".equals(hostPart)) {
                    Proxy.exactMatch(ipMatchers, "0:0:0:0:0:0:0:1", portPart);
                }
                Proxy.exactMatch(ipMatchers, hostPart.substring(1, hostPart.length() - 1), portPart);
                continue;
            }
            if (hostPart.charAt(0) == '.') {
                Proxy.prefixedMatch(hostMatchers, hostPart, portPart);
                continue;
            }
            Proxy.exactMatch(hostMatchers, hostPart, portPart);
        }
        return address -> {
            String host = Proxy.resolveHost(address.getHost());
            int port = address.getPort();
            if (Proxy.isIpV4(host) || Proxy.isIpV6Host(host)) {
                for (BiFunction ipMatcher : ipMatchers) {
                    if (!((Boolean)ipMatcher.apply(host, port)).booleanValue()) continue;
                    LOGGER.finest(() -> "IP Address " + host + " bypasses proxy");
                    return true;
                }
                LOGGER.finest(() -> "IP Address " + host + " uses proxy");
            } else {
                for (BiFunction hostMatcher : hostMatchers) {
                    if (!((Boolean)hostMatcher.apply(host, port)).booleanValue()) continue;
                    LOGGER.finest(() -> "Host " + host + " bypasses proxy");
                    return true;
                }
                LOGGER.finest(() -> "Host " + host + " uses proxy");
            }
            return false;
        };
    }

    private static String resolveHost(String host) {
        if (host != null && Proxy.isIpV6Identifier(host)) {
            return host.substring(1, host.length() - 1);
        }
        return host;
    }

    private static void prefixedMatch(List<BiFunction<String, Integer, Boolean>> matchers, String hostPart, Integer portPart) {
        if (null == portPart) {
            matchers.add((host, port) -> Proxy.prefixHostMatch(hostPart, host));
        } else {
            matchers.add((host, port) -> portPart.equals(port) && Proxy.prefixHostMatch(hostPart, host));
        }
    }

    private static boolean prefixHostMatch(String hostPart, String host) {
        if (host.endsWith(hostPart)) {
            return true;
        }
        return host.equals(hostPart.substring(1));
    }

    private static void exactMatch(List<BiFunction<String, Integer, Boolean>> matchers, String hostPart, Integer portPart) {
        if (null == portPart) {
            matchers.add((host, port) -> hostPart.equals(host));
        } else {
            matchers.add((host, port) -> portPart.equals(port) && hostPart.equals(host));
        }
    }

    private static boolean isIpV4(String host) {
        return IP_V4.matcher(host).matches();
    }

    private static boolean isIpV6Identifier(String host) {
        return IVP6_IDENTIFIER_MATCH_RESULTS.computeValue((Object)host, () -> Proxy.isIpV6IdentifierRegExp(host)).orElse(false);
    }

    private static Optional<Boolean> isIpV6IdentifierRegExp(String host) {
        return Optional.of(IP_V6_IDENTIFIER.matcher(host).matches() || IP_V6_HEX_IDENTIFIER.matcher(host).matches());
    }

    private static boolean isIpV6Host(String host) {
        return IVP6_HOST_MATCH_RESULTS.computeValue((Object)host, () -> Proxy.isIpV6HostRegExp(host)).orElse(false);
    }

    private static Optional<Boolean> isIpV6HostRegExp(String host) {
        return Optional.of(IP_V6_HOST.matcher(host).matches() || IP_V6_HEX_HOST.matcher(host).matches());
    }

    public Optional<ChannelHandler> handler(URI address) {
        if (this.type == ProxyType.NONE) {
            return Optional.empty();
        }
        if (this.useSystemSelector) {
            return this.systemSelectorHandler(address);
        }
        if (this.noProxy.apply(address).booleanValue()) {
            return Optional.empty();
        }
        return Optional.of(this.handler());
    }

    private Optional<ChannelHandler> systemSelectorHandler(URI address) {
        List<java.net.Proxy> selected = this.systemSelector.select(URI.create("http://" + address.getHost() + ":" + address.getPort()));
        if (selected.isEmpty()) {
            return Optional.empty();
        }
        java.net.Proxy systemProxy = selected.iterator().next();
        switch (systemProxy.type()) {
            case DIRECT: {
                return Optional.empty();
            }
            case HTTP: {
                return Optional.of(this.httpProxy(systemProxy));
            }
            case SOCKS: {
                return Optional.of(this.socksProxy(systemProxy));
            }
        }
        throw new IllegalStateException("Unexpected proxy type: " + systemProxy.type());
    }

    private ChannelHandler handler() {
        switch (this.type) {
            case HTTP: {
                return this.httpProxy();
            }
            case SOCKS_4: {
                return this.socks4Proxy();
            }
            case SOCKS_5: {
                return this.socks5Proxy();
            }
        }
        throw new IllegalArgumentException("Unsupported proxy type: " + this.type);
    }

    private ChannelHandler socks5Proxy() {
        return (ChannelHandler)this.username.map(s -> new Socks5ProxyHandler((SocketAddress)this.address(), s, this.password.map(String::new).orElse(""))).orElseGet(() -> new Socks5ProxyHandler((SocketAddress)this.address()));
    }

    private ChannelHandler socks4Proxy() {
        return (ChannelHandler)this.username.map(s -> new Socks4ProxyHandler((SocketAddress)this.address(), s)).orElseGet(() -> new Socks4ProxyHandler((SocketAddress)this.address()));
    }

    private ChannelHandler httpProxy() {
        return (ChannelHandler)this.username.map(s -> new HttpProxyHandler((SocketAddress)this.address(), s, this.password.map(String::new).orElse(""))).orElseGet(() -> new HttpProxyHandler((SocketAddress)this.address()));
    }

    private ChannelHandler httpProxy(java.net.Proxy systemProxy) {
        return new HttpProxyHandler(systemProxy.address());
    }

    private ChannelHandler socksProxy(java.net.Proxy systemProxy) {
        return new Socks5ProxyHandler(systemProxy.address());
    }

    private InetSocketAddress address() {
        return new InetSocketAddress(this.host, this.port);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Proxy proxy = (Proxy)o;
        return this.port == proxy.port && this.useSystemSelector == proxy.useSystemSelector && this.type == proxy.type && Objects.equals(this.host, proxy.host) && Objects.equals(this.noProxy, proxy.noProxy) && Objects.equals(this.username, proxy.username) && Objects.equals(this.password, proxy.password) && Objects.equals(this.systemSelector, proxy.systemSelector);
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.type, this.host, this.port, this.noProxy, this.username, this.password, this.systemSelector, this.useSystemSelector});
    }

    public static enum ProxyType {
        NONE,
        SYSTEM,
        HTTP,
        SOCKS_4,
        SOCKS_5;

    }

    public static class Builder
    implements io.helidon.common.Builder<Proxy> {
        private final Set<String> noProxyHosts = new HashSet<String>();
        private ProxyType type;
        private String host;
        private int port = 80;
        private String username;
        private char[] password;
        private ProxySelector systemSelector;

        private Builder() {
        }

        public Proxy build() {
            if (null == this.host || this.host.isEmpty() && null == this.systemSelector) {
                return NO_PROXY;
            }
            return new Proxy(this);
        }

        public Builder config(Config config) {
            config.get("use-system-selector").asBoolean().ifPresent(this::useSystemSelector);
            if (this.type != ProxyType.SYSTEM) {
                config.get("type").asString().map(ProxyType::valueOf).ifPresent(this::type);
                config.get("host").asString().ifPresent(this::host);
                config.get("port").asInt().ifPresent(this::port);
                config.get("username").asString().ifPresent(this::username);
                config.get("password").asString().map(String::toCharArray).ifPresent(this::password);
                config.get("no-proxy").asList(String.class).ifPresent(hosts -> hosts.forEach(this::addNoProxy));
            }
            return this;
        }

        public Builder type(ProxyType type) {
            this.type = type;
            return this;
        }

        public Builder host(String host) {
            this.host = host;
            return this;
        }

        public Builder port(int port) {
            this.port = port;
            return this;
        }

        public Builder username(String username) {
            this.username = username;
            return this;
        }

        public Builder password(char[] password) {
            this.password = Arrays.copyOf(password, password.length);
            return this;
        }

        public Builder addNoProxy(String noProxyHost) {
            this.noProxyHosts.add(noProxyHost);
            return this;
        }

        public Builder useSystemSelector(boolean useIt) {
            if (useIt) {
                this.type = ProxyType.SYSTEM;
                this.systemSelector = ProxySelector.getDefault();
            } else {
                if (this.type == ProxyType.SYSTEM) {
                    this.type = ProxyType.NONE;
                }
                this.systemSelector = null;
            }
            return this;
        }

        ProxyType type() {
            return this.type;
        }

        String host() {
            return this.host;
        }

        int port() {
            return this.port;
        }

        Set<String> noProxyHosts() {
            return new HashSet<String>(this.noProxyHosts);
        }

        Optional<String> username() {
            return Optional.ofNullable(this.username);
        }

        Optional<char[]> password() {
            return Optional.ofNullable(this.password);
        }

        ProxySelector systemSelector() {
            return this.systemSelector;
        }
    }
}

