/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security;

import io.helidon.common.LazyValue;
import io.helidon.common.reactive.Single;
import io.helidon.config.Config;
import io.helidon.security.AuditEvent;
import io.helidon.security.NamedProvider;
import io.helidon.security.Security;
import io.helidon.security.SecurityContext;
import io.helidon.security.SecurityEnvironment;
import io.helidon.security.SecurityException;
import io.helidon.security.SecurityTime;
import io.helidon.security.SecurityUtil;
import io.helidon.security.internal.SecurityAuditEvent;
import io.helidon.security.spi.AuditProvider;
import io.helidon.security.spi.AuthenticationProvider;
import io.helidon.security.spi.AuthorizationProvider;
import io.helidon.security.spi.DigestProvider;
import io.helidon.security.spi.EncryptionProvider;
import io.helidon.security.spi.OutboundSecurityProvider;
import io.helidon.security.spi.ProviderSelectionPolicy;
import io.helidon.security.spi.SecurityProvider;
import io.helidon.security.spi.SubjectMappingProvider;
import io.helidon.tracing.Tracer;
import java.lang.annotation.Annotation;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.logging.Logger;

final class SecurityImpl
implements Security {
    static final Set<String> RESERVED_PROVIDER_KEYS = Set.of("name", "type", "class", "is-authentication-provider", "is-authorization-provider", "is-client-security-provider", "is-audit-provider");
    private static final Set<String> CONFIG_INTERNAL_PREFIXES = Set.of("provider-policy", "providers", "environment");
    private static final Logger LOGGER = Logger.getLogger(SecurityImpl.class.getName());
    private final Collection<Class<? extends Annotation>> annotations = new LinkedList<Class<? extends Annotation>>();
    private final List<Consumer<AuditProvider.TracedAuditEvent>> auditors = new LinkedList<Consumer<AuditProvider.TracedAuditEvent>>();
    private final Optional<SubjectMappingProvider> subjectMappingProvider;
    private final String instanceUuid;
    private final ProviderSelectionPolicy providerSelectionPolicy;
    private final LazyValue<Tracer> securityTracer;
    private final SecurityTime serverTime;
    private final Supplier<ExecutorService> executorService;
    private final Config securityConfig;
    private final boolean enabled;
    private final Map<String, Supplier<Single<Optional<String>>>> secrets;
    private final Map<String, EncryptionProvider.EncryptionSupport> encryptions;
    private final Map<String, DigestProvider.DigestSupport> digests;

    SecurityImpl(Security.Builder builder) {
        this.enabled = builder.enabled();
        this.instanceUuid = UUID.randomUUID().toString();
        this.serverTime = builder.serverTime();
        this.executorService = builder.executorService();
        this.annotations.addAll(SecurityUtil.getAnnotations(builder.allProviders()));
        this.securityTracer = LazyValue.create(() -> SecurityUtil.getTracer(builder.tracingEnabled(), builder.tracer()));
        this.subjectMappingProvider = Optional.ofNullable(builder.subjectMappingProvider());
        this.securityConfig = builder.config();
        if (!this.enabled) {
            this.audit(this.instanceUuid, SecurityAuditEvent.info("security.configure", "Security is disabled."));
        }
        final LinkedList<NamedProvider<AuthorizationProvider>> atzProviders = new LinkedList<NamedProvider<AuthorizationProvider>>();
        final LinkedList<NamedProvider<AuthenticationProvider>> atnProviders = new LinkedList<NamedProvider<AuthenticationProvider>>();
        final LinkedList<NamedProvider<OutboundSecurityProvider>> outboundProviders = new LinkedList<NamedProvider<OutboundSecurityProvider>>();
        atzProviders.addAll(builder.atzProviders());
        atnProviders.addAll(builder.atnProviders());
        outboundProviders.addAll(builder.outboundProviders());
        builder.auditProviders().forEach(auditProvider -> this.auditors.add(auditProvider.auditConsumer()));
        this.audit(this.instanceUuid, SecurityAuditEvent.info("security.configure", "Security initialized. Providers: audit: \"%s\"; authn: \"%s\"; authz: \"%s\"; identity propagation: \"%s\";").addParam(AuditEvent.AuditParam.plain("auditProviders", SecurityUtil.forAudit(builder.auditProviders()))).addParam(AuditEvent.AuditParam.plain("authenticationProvider", SecurityUtil.forAuditNamed(atnProviders))).addParam(AuditEvent.AuditParam.plain("authorizationProvider", SecurityUtil.forAuditNamed(atzProviders))).addParam(AuditEvent.AuditParam.plain("identityPropagationProvider", SecurityUtil.forAuditNamed(outboundProviders))));
        final NamedProvider<AuthenticationProvider> authnProvider = builder.authnProvider();
        final NamedProvider<AuthorizationProvider> authzProvider = builder.authzProvider();
        this.providerSelectionPolicy = builder.providerSelectionPolicy().apply(new ProviderSelectionPolicy.Providers(){

            @Override
            public <T extends SecurityProvider> List<NamedProvider<T>> getProviders(Class<T> providerType) {
                if (providerType.equals(AuthenticationProvider.class)) {
                    LinkedList result = new LinkedList();
                    result.add(authnProvider);
                    atnProviders.stream().filter(pr -> pr != authnProvider).forEach(atn -> result.add((NamedProvider)atn));
                    return result;
                }
                if (providerType.equals(AuthorizationProvider.class)) {
                    LinkedList result = new LinkedList();
                    result.add(authzProvider);
                    atzProviders.stream().filter(pr -> pr != authzProvider).forEach(atn -> result.add((NamedProvider)atn));
                    return result;
                }
                if (providerType.equals(OutboundSecurityProvider.class)) {
                    LinkedList result = new LinkedList();
                    outboundProviders.forEach(atn -> result.add((NamedProvider)atn));
                    return result;
                }
                throw new SecurityException("Security only supports AuthenticationProvider, AuthorizationProvider and OutboundSecurityProvider in provider selection policy, not " + providerType.getName());
            }
        });
        this.secrets = Map.copyOf(builder.secrets());
        this.encryptions = Map.copyOf(builder.encryptions());
        this.digests = Map.copyOf(builder.digests());
    }

    @Override
    public SecurityTime serverTime() {
        return this.serverTime;
    }

    @Override
    public SecurityContext.Builder contextBuilder(String id) {
        String newId = null == id || id.isEmpty() ? this.instanceUuid + ":?" : this.instanceUuid + ":" + id;
        return new SecurityContext.Builder(this).id(newId).executorService(this.executorService).tracingTracer((Tracer)this.securityTracer.get()).serverTime(this.serverTime);
    }

    @Override
    public SecurityContext createContext(String id) {
        return this.contextBuilder(id).build();
    }

    @Override
    public Tracer tracer() {
        return (Tracer)this.securityTracer.get();
    }

    @Override
    public Collection<Class<? extends Annotation>> customAnnotations() {
        return this.annotations;
    }

    @Override
    public Config configFor(String child) {
        String test = child.trim();
        if (test.isEmpty()) {
            throw new IllegalArgumentException("Root of security configuration is not available");
        }
        for (String prefix : CONFIG_INTERNAL_PREFIXES) {
            if (!child.equals(prefix) && !child.startsWith(prefix + ".")) continue;
            throw new IllegalArgumentException("Security configuration for " + prefix + " is not available");
        }
        return this.securityConfig.get(child);
    }

    @Override
    public Single<String> encrypt(String configurationName, byte[] bytesToEncrypt) {
        EncryptionProvider.EncryptionSupport encryption = this.encryptions.get(configurationName);
        if (encryption == null) {
            return Single.error((Throwable)new SecurityException("There is no configured encryption named " + configurationName));
        }
        return encryption.encrypt(bytesToEncrypt);
    }

    @Override
    public Single<byte[]> decrypt(String configurationName, String cipherText) {
        EncryptionProvider.EncryptionSupport encryption = this.encryptions.get(configurationName);
        if (encryption == null) {
            return Single.error((Throwable)new SecurityException("There is no configured encryption named " + configurationName));
        }
        return encryption.decrypt(cipherText);
    }

    @Override
    public Single<String> digest(String configurationName, byte[] bytesToDigest, boolean preHashed) {
        DigestProvider.DigestSupport digest = this.digests.get(configurationName);
        if (digest == null) {
            return Single.error((Throwable)new SecurityException("There is no configured digest named " + configurationName));
        }
        return digest.digest(bytesToDigest, preHashed);
    }

    @Override
    public Single<String> digest(String configurationName, byte[] bytesToDigest) {
        return this.digest(configurationName, bytesToDigest, false);
    }

    @Override
    public Single<Boolean> verifyDigest(String configurationName, byte[] bytesToDigest, String digest, boolean preHashed) {
        DigestProvider.DigestSupport digestSupport = this.digests.get(configurationName);
        if (digest == null) {
            return Single.error((Throwable)new SecurityException("There is no configured digest named " + configurationName));
        }
        return digestSupport.verify(bytesToDigest, preHashed, digest);
    }

    @Override
    public Single<Boolean> verifyDigest(String configurationName, byte[] bytesToDigest, String digest) {
        return this.verifyDigest(configurationName, bytesToDigest, digest, false);
    }

    @Override
    public Single<Optional<String>> secret(String configurationName) {
        Supplier<Single<Optional<String>>> singleSupplier = this.secrets.get(configurationName);
        if (singleSupplier == null) {
            return Single.error((Throwable)new SecurityException("Secret \"" + configurationName + "\" is not configured."));
        }
        return singleSupplier.get();
    }

    @Override
    public Single<String> secret(String configurationName, String defaultValue) {
        Supplier<Single<Optional<String>>> singleSupplier = this.secrets.get(configurationName);
        if (singleSupplier == null) {
            LOGGER.finest(() -> "There is no configured secret named " + configurationName + ", using default value");
            return Single.just((Object)defaultValue);
        }
        return singleSupplier.get().map(it -> it.orElse(defaultValue));
    }

    @Override
    public SecurityEnvironment.Builder environmentBuilder() {
        return SecurityEnvironment.builder(this.serverTime);
    }

    @Override
    public Optional<SubjectMappingProvider> subjectMapper() {
        return this.subjectMappingProvider;
    }

    @Override
    public boolean enabled() {
        return this.enabled;
    }

    @Override
    public void audit(String tracingId, AuditEvent event) {
        AuditProvider.AuditSource auditSource = AuditProvider.AuditSource.create();
        for (Consumer<AuditProvider.TracedAuditEvent> auditor : this.auditors) {
            auditor.accept(SecurityUtil.wrapEvent(tracingId, auditSource, event));
        }
    }

    @Override
    public ProviderSelectionPolicy providerSelectionPolicy() {
        return this.providerSelectionPolicy;
    }

    @Override
    public Supplier<ExecutorService> executorService() {
        return this.executorService;
    }

    @Override
    public Optional<? extends AuthenticationProvider> resolveAtnProvider(String providerName) {
        return this.resolveProvider(AuthenticationProvider.class, providerName);
    }

    @Override
    public Optional<AuthorizationProvider> resolveAtzProvider(String providerName) {
        return this.resolveProvider(AuthorizationProvider.class, providerName);
    }

    @Override
    public List<? extends OutboundSecurityProvider> resolveOutboundProvider(String providerName) {
        if (null != providerName) {
            return this.resolveProvider(OutboundSecurityProvider.class, providerName).map(List::of).orElse(List.of());
        }
        return this.providerSelectionPolicy.selectOutboundProviders();
    }

    private <T extends SecurityProvider> Optional<T> resolveProvider(Class<T> providerClass, String providerName) {
        if (null == providerName) {
            return this.providerSelectionPolicy.selectProvider(providerClass);
        }
        Optional<T> instance = this.providerSelectionPolicy.selectProvider(providerClass, providerName);
        if (instance.isPresent()) {
            return instance;
        }
        throw new SecurityException("Named " + providerClass.getSimpleName() + " expected for name \"" + providerName + "\" yet none is configured for such a name");
    }
}

