/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.microprofile.server;

import io.helidon.common.OptionalHelper;
import io.helidon.common.http.Http;
import io.helidon.config.Config;
import io.helidon.microprofile.config.MpConfig;
import io.helidon.microprofile.server.JaxRsApplication;
import io.helidon.microprofile.server.MpException;
import io.helidon.microprofile.server.Server;
import io.helidon.microprofile.server.spi.MpService;
import io.helidon.microprofile.server.spi.MpServiceContext;
import io.helidon.webserver.Handler;
import io.helidon.webserver.Routing;
import io.helidon.webserver.ServerConfiguration;
import io.helidon.webserver.Service;
import io.helidon.webserver.StaticContentSupport;
import io.helidon.webserver.WebServer;
import io.helidon.webserver.jersey.JerseySupport;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.Path;
import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.Priority;
import javax.enterprise.inject.se.SeContainer;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Application;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.ExceptionMapper;
import org.glassfish.jersey.server.ResourceConfig;

public class ServerImpl
implements Server {
    private static final Logger LOGGER = Logger.getLogger(ServerImpl.class.getName());
    private static final Logger JERSEY_LOGGER = Logger.getLogger(ServerImpl.class.getName() + ".jersey");
    private static final Logger STARTUP_LOGGER = Logger.getLogger("io.helidon.microprofile.startup.server");
    private static final int DEFAULT_PRIORITY = 100;
    private final SeContainer container;
    private final boolean containerCreated;
    private final String host;
    private final WebServer server;
    private final IdentityHashMap<Class<?>, Object> register = new IdentityHashMap();
    private int port = -1;

    ServerImpl(Server.Builder builder) {
        InetAddress listenHost;
        MpConfig mpConfig = (MpConfig)builder.getConfig();
        Config config = mpConfig.getConfig();
        this.container = builder.getCdiContainer();
        this.containerCreated = builder.getContainerCreated();
        if (null == builder.getHost()) {
            listenHost = InetAddress.getLoopbackAddress();
        } else {
            try {
                listenHost = InetAddress.getByName(builder.getHost());
            }
            catch (UnknownHostException e) {
                throw new MpException("Failed to create address for host: " + this.getHost(), e);
            }
        }
        this.host = listenHost.getHostName();
        Routing.Builder routingBuilder = Routing.builder();
        Config serverConfig = config.get("server");
        ServerConfiguration.Builder serverConfigBuilder = ServerConfiguration.builder((Config)serverConfig).port(builder.getPort()).bindAddress(listenHost);
        OptionalHelper.from(Optional.ofNullable(builder.basePath())).or(() -> config.get("server.base-path").value()).asOptional().ifPresent(basePath -> routingBuilder.any("/", new Handler[]{(req, res) -> {
            res.status((Http.ResponseStatus)Http.Status.MOVED_PERMANENTLY_301);
            res.headers().put("Location", new String[]{basePath});
            res.send();
        }}));
        STARTUP_LOGGER.finest("Builders ready");
        List<JaxRsApplication> applications = builder.getApplications();
        this.loadExtensions(builder, mpConfig, config, applications, routingBuilder, serverConfigBuilder);
        STARTUP_LOGGER.finest("Extensions loaded");
        applications.stream().map(JaxRsApplication::getConfig).forEach(resourceConfig -> resourceConfig.register((Object)new ExceptionMapper<Exception>(){

            public Response toResponse(Exception exception) {
                if (exception instanceof WebApplicationException) {
                    return ((WebApplicationException)exception).getResponse();
                }
                JERSEY_LOGGER.log(Level.WARNING, exception, () -> "Internal server error");
                return Response.serverError().build();
            }
        }));
        serverConfig.get("static.classpath").ifExists(cpConfig -> {
            Config context = cpConfig.get("context");
            StaticContentSupport.Builder cpBuilder = StaticContentSupport.builder((String)cpConfig.get("location").asString());
            cpBuilder.welcomeFileName(cpConfig.get("welcome").value().orElse("index.html"));
            StaticContentSupport staticContent = cpBuilder.build();
            if (context.exists()) {
                routingBuilder.register(context.asString(), new Service[]{staticContent});
            } else {
                routingBuilder.register(new Service[]{staticContent});
            }
        });
        STARTUP_LOGGER.finest("Static classpath");
        serverConfig.get("static.path").ifExists(pathConfig -> {
            Config context = pathConfig.get("context");
            StaticContentSupport.Builder pBuilder = StaticContentSupport.builder((Path)((Path)pathConfig.get("location").as(Path.class)));
            pathConfig.get("welcome").value().ifPresent(arg_0 -> ((StaticContentSupport.Builder)pBuilder).welcomeFileName(arg_0));
            StaticContentSupport staticContent = pBuilder.build();
            if (context.exists()) {
                routingBuilder.register(context.asString(), new Service[]{staticContent});
            } else {
                routingBuilder.register(new Service[]{staticContent});
            }
        });
        STARTUP_LOGGER.finest("Static path");
        applications.forEach(app -> {
            JerseySupport js = JerseySupport.builder((Application)app.getConfig()).executorService(app.getExecutorService().orElseGet(builder::getDefaultExecutorService)).build();
            if ("/".equals(app.getContextRoot())) {
                routingBuilder.register(new Service[]{js});
            } else {
                routingBuilder.register(app.getContextRoot(), new Service[]{js});
            }
        });
        STARTUP_LOGGER.finest("Registered jersey application(s)");
        this.server = routingBuilder.build().createServer(serverConfigBuilder.build());
        STARTUP_LOGGER.finest("Server created");
    }

    private static int findPriority(Class<?> aClass) {
        Priority priorityAnnot = aClass.getAnnotation(Priority.class);
        if (null != priorityAnnot) {
            return priorityAnnot.value();
        }
        return 100;
    }

    private void loadExtensions(Server.Builder builder, final MpConfig mpConfig, final Config config, final List<JaxRsApplication> apps, final Routing.Builder routingBuilder, final ServerConfiguration.Builder serverConfigBuilder) {
        LinkedList<MpService> extensions = new LinkedList<MpService>(builder.getExtensions());
        ServiceLoader.load(MpService.class).forEach(extensions::add);
        final LinkedList newApps = new LinkedList();
        MpServiceContext context = new MpServiceContext(){

            @Override
            public org.eclipse.microprofile.config.Config getConfig() {
                return mpConfig;
            }

            @Override
            public List<ResourceConfig> getApplications() {
                return apps.stream().map(JaxRsApplication::getConfig).collect(Collectors.toList());
            }

            @Override
            public void addApplication(Application application) {
                newApps.add(JaxRsApplication.create(application));
            }

            @Override
            public void addApplication(String contextRoot, Application application) {
                newApps.add(JaxRsApplication.builder().contextRoot(contextRoot).application(application).build());
            }

            @Override
            public Config getHelidonConfig() {
                return config;
            }

            @Override
            public SeContainer getCdiContainer() {
                return ServerImpl.this.container;
            }

            @Override
            public ServerConfiguration.Builder getServerConfigBuilder() {
                return serverConfigBuilder;
            }

            @Override
            public Routing.Builder getServerRoutingBuilder() {
                return routingBuilder;
            }

            @Override
            public <U> void register(Class<? extends U> key, U instance) {
                ServerImpl.this.register.put(key, instance);
            }
        };
        for (MpService extension : extensions) {
            extension.configure(context);
            apps.addAll(newApps);
            newApps.clear();
        }
    }

    @Override
    public SeContainer getContainer() {
        return this.container;
    }

    @Override
    public Server start() {
        STARTUP_LOGGER.entering(ServerImpl.class.getName(), "start");
        CountDownLatch cdl = new CountDownLatch(1);
        AtomicReference throwRef = new AtomicReference();
        long beforeT = System.nanoTime();
        this.server.start().whenComplete((webServer, throwable) -> {
            if (null != throwable) {
                STARTUP_LOGGER.log(Level.FINEST, "Startup failed", (Throwable)throwable);
                throwRef.set(throwable);
            } else {
                long t = TimeUnit.MILLISECONDS.convert(System.nanoTime() - beforeT, TimeUnit.NANOSECONDS);
                this.port = webServer.port();
                STARTUP_LOGGER.finest("Started up");
                if ("0.0.0.0".equals(this.host)) {
                    LOGGER.info(() -> "Server started on http://localhost:" + this.port + " (and all other host addresses) in " + t + " milliseconds.");
                } else {
                    LOGGER.info(() -> "Server started on http://" + this.host + ":" + this.port + " in " + t + " milliseconds.");
                }
            }
            cdl.countDown();
        });
        try {
            cdl.await();
            STARTUP_LOGGER.finest("Count down latch released");
        }
        catch (InterruptedException e) {
            throw new MpException("Interrupted while starting server", e);
        }
        if (throwRef.get() == null) {
            return this;
        }
        throw new MpException("Failed to start server", (Throwable)throwRef.get());
    }

    @Override
    public Server stop() {
        try {
            this.stopWebServer();
        }
        finally {
            if (this.containerCreated) {
                try {
                    this.container.close();
                }
                catch (IllegalStateException e) {
                    LOGGER.log(Level.SEVERE, "Container already closed", e);
                }
            }
        }
        return this;
    }

    private void stopWebServer() {
        CountDownLatch cdl = new CountDownLatch(1);
        AtomicReference throwRef = new AtomicReference();
        long beforeT = System.nanoTime();
        this.server.shutdown().whenComplete((webServer, throwable) -> {
            if (null != throwable) {
                throwRef.set(throwable);
            } else {
                long t = TimeUnit.MILLISECONDS.convert(System.nanoTime() - beforeT, TimeUnit.NANOSECONDS);
                LOGGER.info(() -> "Server stopped in " + t + " milliseconds.");
            }
            cdl.countDown();
        });
        try {
            cdl.await();
        }
        catch (InterruptedException e) {
            throw new MpException("Interrupted while shutting down server", e);
        }
        if (throwRef.get() != null) {
            throw new MpException("Failed to shut down server", (Throwable)throwRef.get());
        }
    }

    @Override
    public String getHost() {
        return this.host;
    }

    @Override
    public int getPort() {
        return this.port;
    }
}

