var HawtioOAuth;
(function (HawtioOAuth) {
    HawtioOAuth.pluginName = 'hawtio-oauth';
    HawtioOAuth.log = Logger.get(HawtioOAuth.pluginName);
    HawtioOAuth.oauthPlugins = [];
})(HawtioOAuth || (HawtioOAuth = {}));
/// <reference path="oauth.globals.ts"/>
var HawtioOAuth;
(function (HawtioOAuth) {
    var userProfile = null;
    function getUserProfile() {
        if (!userProfile) {
            HawtioOAuth.log.debug("Finding 'userProfile' from the active OAuth plugin");
            findUserProfile();
        }
        return userProfile;
    }
    HawtioOAuth.getUserProfile = getUserProfile;
    function findUserProfile() {
        var activePlugin = _.find(HawtioOAuth.oauthPlugins, function (plugin) {
            var profile = Core.pathGet(window, [plugin, 'userProfile']);
            HawtioOAuth.log.debug("Module:", plugin, "userProfile:", profile);
            return !_.isNil(profile);
        });
        userProfile = Core.pathGet(window, [activePlugin, 'userProfile']);
        HawtioOAuth.log.debug("Active OAuth plugin:", activePlugin);
    }
    function getOAuthToken() {
        var userProfile = getUserProfile();
        if (!userProfile) {
            return null;
        }
        return userProfile.token;
    }
    HawtioOAuth.getOAuthToken = getOAuthToken;
    function authenticatedHttpRequest(options) {
        return $.ajax(_.extend(options, {
            beforeSend: function (request) {
                var token = getOAuthToken();
                if (token) {
                    request.setRequestHeader('Authorization', 'Bearer ' + token);
                }
            }
        }));
    }
    HawtioOAuth.authenticatedHttpRequest = authenticatedHttpRequest;
})(HawtioOAuth || (HawtioOAuth = {}));
/// <reference path="../../oauth.globals.ts"/>
var GithubOAuth;
(function (GithubOAuth) {
    GithubOAuth.pluginName = 'hawtio-oauth-github';
    GithubOAuth.log = Logger.get(GithubOAuth.pluginName);
    GithubOAuth.templatePath = 'plugins/github/html';
    var LOCAL_STORAGE_KEY = 'GithubOAuthSettings';
    function emptyBeforeSend() {
        return true;
    }
    GithubOAuth.emptyBeforeSend = emptyBeforeSend;
    function getTokenCheckAuthURL(oauthSettings) {
        return UrlHelpers.join('https://api.github.com/applications', oauthSettings.clientId, 'tokens', oauthSettings.accessToken);
    }
    GithubOAuth.getTokenCheckAuthURL = getTokenCheckAuthURL;
    function getTokenCheckAuthHeader(oauthSettings) {
        return Core.getBasicAuthHeader(oauthSettings.clientId, oauthSettings.clientSecret);
    }
    GithubOAuth.getTokenCheckAuthHeader = getTokenCheckAuthHeader;
    function getAuthHeader(oauthSettings) {
        var token = oauthSettings.accessToken;
        if (!token) {
            return '';
        }
        return 'token ' + oauthSettings.accessToken;
    }
    GithubOAuth.getAuthHeader = getAuthHeader;
    function loadSettings() {
        var answer = {};
        if (LOCAL_STORAGE_KEY in localStorage) {
            var settings = localStorage[LOCAL_STORAGE_KEY];
            try {
                settings = angular.fromJson(settings);
                answer = settings;
            }
            catch (err) {
                localStorage.removeItem(LOCAL_STORAGE_KEY);
            }
        }
        return answer;
    }
    GithubOAuth.loadSettings = loadSettings;
    function storeSettings(settings, oauthSettings) {
        if (oauthSettings === void 0) { oauthSettings = undefined; }
        var toStore = {
            username: settings.username,
            avatarURL: settings.avatarURL,
            accessToken: settings.accessToken,
            name: settings.name
        };
        if (oauthSettings) {
            oauthSettings.username = toStore.username;
            oauthSettings.avatarURL = toStore.avatarURL;
            oauthSettings.accessToken = toStore.accessToken;
            oauthSettings.name = toStore.name;
        }
        ;
        localStorage[LOCAL_STORAGE_KEY] = angular.toJson(toStore);
    }
    GithubOAuth.storeSettings = storeSettings;
})(GithubOAuth || (GithubOAuth = {}));
/// <reference path="../../oauth.globals.ts"/>
/// <reference path="githubHelpers.ts"/>
var GithubOAuth;
(function (GithubOAuth) {
    GithubOAuth._module = angular.module(GithubOAuth.pluginName, []);
    var settings = {
        enabled: false,
        username: undefined,
        clientId: undefined,
        clientSecret: undefined,
        accessToken: undefined,
        avatarURL: undefined,
        name: undefined,
        scopes: ['user', 'repo', 'read:org'],
        authURL: 'https://github.com/login/oauth/authorize',
        tokenURL: 'https://api.github.com/authorizations',
        loginURL: 'https://api.github.com/user'
    };
    GithubOAuth._module.constant('githubOAuthSettings', settings);
    GithubOAuth._module.service('GithubOAuth', ['githubOAuthSettings', function (settings) {
            var self = {
                settings: settings,
                hasToken: function () {
                    return !Core.isBlank(self.settings.accessToken);
                },
                getToken: function () {
                    return self.settings.accessToken;
                },
                getHeader: function () {
                    return GithubOAuth.getAuthHeader(self.settings);
                },
                getPreferencesLink: function () { }
            };
            return self;
        }]);
    GithubOAuth._module.run(['preferencesRegistry', function (preferencesRegistry) {
            preferencesRegistry.addTab("Github", UrlHelpers.join(GithubOAuth.templatePath, "githubPreferences.html"), function () { return settings.enabled; });
            GithubOAuth.log.debug("loaded");
        }]);
    hawtioPluginLoader.addModule(GithubOAuth.pluginName);
    hawtioPluginLoader.registerPreBootstrapTask({
        name: 'GithubOAuthConfig',
        depends: ['HawtioOAuthConfig'],
        task: function (next) {
            if (window['HAWTIO_OAUTH_CONFIG']) {
                var clientId = settings.clientId = Core.pathGet(HAWTIO_OAUTH_CONFIG, ['github', 'clientId']);
                var clientSecret = settings.clientSecret = Core.pathGet(HAWTIO_OAUTH_CONFIG, ['github', 'clientSecret']);
                if (clientId && clientSecret) {
                    GithubOAuth.log.debug("enabled");
                    settings.enabled = true;
                }
            }
            next();
        }
    });
    hawtioPluginLoader.registerPreBootstrapTask({
        name: 'GithubOAuthSettings',
        depends: ['GithubOAuthConfig'],
        task: function (next) {
            if (settings.enabled) {
                _.assign(settings, GithubOAuth.loadSettings());
            }
            next();
        }
    });
    /*
    hawtioPluginLoader.registerPreBootstrapTask({
      name: 'GithubTokenReader',
      depends: ['GithubOAuthConfig'],
      task: (next) => {
        if (!settings.enabled) {
          next();
          return;
        }
        let uri = new URI();
        let search = uri.search(true);
        let accessCode = search['code'];
        if (accessCode) {
          log.debug("Found github access code");
          delete search['code'];
          $.ajax(settings.tokenURL, <any> {
            method: 'POST',
            data: {
              client_id: settings.clientId
            }
  
          });
          uri.search(search);
          window.location.href = uri.toString();
        } else {
          next();
        }
      }
    });
    */
})(GithubOAuth || (GithubOAuth = {}));
/// <reference path="../../oauth.globals.ts"/>
var GoogleOAuth;
(function (GoogleOAuth) {
    GoogleOAuth.pluginName = 'hawtio-oauth-google';
    GoogleOAuth.log = Logger.get(GoogleOAuth.pluginName);
    // Keep this unset unless we have a token
    GoogleOAuth.userProfile = null;
})(GoogleOAuth || (GoogleOAuth = {}));
/// <reference path="googleOAuthGlobals.ts"/>
var GoogleOAuth;
(function (GoogleOAuth) {
    var GOOGLE_TOKEN_STORAGE_KEY = 'googleAuthCreds';
    function authenticatedHttpRequest(options, userDetails) {
        return $.ajax(_.extend(options, {
            beforeSend: function (request) {
                if (userDetails.token) {
                    request.setRequestHeader('Authorization', 'Bearer ' + userDetails.token);
                }
            }
        }));
    }
    GoogleOAuth.authenticatedHttpRequest = authenticatedHttpRequest;
    function setupJQueryAjax(userDetails) {
        $.ajaxSetup({
            beforeSend: function (xhr) {
                var token = userDetails.token;
                if (token) {
                    xhr.setRequestHeader('Authorization', 'Bearer ' + token);
                }
            }
        });
    }
    GoogleOAuth.setupJQueryAjax = setupJQueryAjax;
    function doLogout(config, userDetails) {
        if (config === void 0) { config = window['GoogleOAuthConfig']; }
        if (userDetails === void 0) { userDetails = GoogleOAuth.userProfile; }
        console.debug("Logging out!");
        var token = getTokenStorage() || userDetails.token;
        var uri = new URI(window.location.href).removeQuery("code");
        var target = uri.toString();
        GoogleOAuth.log.debug("Now logging in with URI: " + target);
        clearTokenStorage();
        doLogin(GoogleOAuthConfig, {
            uri: target
        });
    }
    GoogleOAuth.doLogout = doLogout;
    function doLogin(config, options) {
        var clientId = config.clientId;
        var redirectURI = config.redirectURI;
        var scope = config.scope;
        var targetURI = config.authenticationURI;
        var uri = new URI(targetURI);
        uri.query({
            response_type: 'code',
            client_id: clientId,
            redirect_uri: redirectURI,
            scope: scope,
            approval_prompt: 'force'
        });
        var target = uri.toString();
        GoogleOAuth.log.debug("Redirecting to URI: ", target);
        window.location.href = target;
    }
    GoogleOAuth.doLogin = doLogin;
    function exchangeCodeForToken(config, code, options) {
        var clientId = config.clientId;
        var clientSecret = config.clientSecret;
        var redirectURI = config.redirectURI;
        var uri = new URI(config.tokenURI || 'https://www.googleapis.com/oauth2/v3/token');
        uri.query({
            code: code,
            client_id: clientId,
            client_secret: clientSecret,
            redirect_uri: redirectURI,
            grant_type: 'authorization_code'
        });
        var target = uri.toString();
        GoogleOAuth.log.debug("Redirecting to URI: ", target);
        return $.ajax({
            type: 'POST',
            url: target
        });
    }
    GoogleOAuth.exchangeCodeForToken = exchangeCodeForToken;
    function extractToken(query) {
        GoogleOAuth.log.debug("query: ", query);
        if (query.access_token && query.token_type === "Bearer") {
            GoogleOAuth.log.debug("Got token");
            var localStorage = Core.getLocalStorage();
            var creds = {
                type: query.token_type.toLowerCase(),
                token: query.access_token,
                expiry: query.expires_in
            };
            localStorage[GOOGLE_TOKEN_STORAGE_KEY] = angular.toJson(creds);
            delete query.token_type;
            delete query.access_token;
            delete query.expires_in;
            // SHOULD THIS BE CALLED?
            //var target = query.toString();
            //log.debug("redirecting to: ", target);
            //window.location.href = target;
            return creds;
        }
        else {
            GoogleOAuth.log.info("No token in URI");
            return undefined;
        }
    }
    GoogleOAuth.extractToken = extractToken;
    function clearTokenStorage() {
        var localStorage = Core.getLocalStorage();
        delete localStorage[GOOGLE_TOKEN_STORAGE_KEY];
    }
    GoogleOAuth.clearTokenStorage = clearTokenStorage;
    function getTokenStorage() {
        var localStorage = Core.getLocalStorage();
        var value = localStorage[GOOGLE_TOKEN_STORAGE_KEY];
        if (value) {
            try {
                return angular.fromJson(value);
            }
            catch (e) {
                GoogleOAuth.log.warn("Failed to parse token json: " + value + ". " + e);
            }
        }
        return null;
    }
    GoogleOAuth.getTokenStorage = getTokenStorage;
    function setTokenStorage(userDetails) {
        var localStorage = Core.getLocalStorage();
        localStorage[GOOGLE_TOKEN_STORAGE_KEY] = angular.toJson(userDetails);
    }
    GoogleOAuth.setTokenStorage = setTokenStorage;
    function checkToken(query) {
        var localStorage = Core.getLocalStorage();
        var answer = undefined;
        if (GOOGLE_TOKEN_STORAGE_KEY in localStorage) {
            try {
                answer = angular.fromJson(localStorage[GOOGLE_TOKEN_STORAGE_KEY]);
            }
            catch (e) {
                clearTokenStorage();
                // must be broken...
                GoogleOAuth.log.error("Error extracting googleAuthCreds value: ", e);
            }
        }
        if (!answer) {
            answer = extractToken(query);
        }
        GoogleOAuth.log.debug("Using creds: ", answer);
        return answer;
    }
    GoogleOAuth.checkToken = checkToken;
    function checkAuthorizationCode(uri) {
        return uri.query(true).code;
    }
    GoogleOAuth.checkAuthorizationCode = checkAuthorizationCode;
    function fetchUserInfo(http, successCallback, failureCallback) {
        http.get('https://www.googleapis.com/oauth2/v1/userinfo?alt=json&access_token=' + HawtioOAuth.getUserProfile().access_token).
            success(successCallback).error(failureCallback);
    }
    GoogleOAuth.fetchUserInfo = fetchUserInfo;
})(GoogleOAuth || (GoogleOAuth = {}));
/// <reference path="googleOAuthHelpers.ts"/>
var GoogleOAuth;
(function (GoogleOAuth) {
    HawtioOAuth.oauthPlugins.push('GoogleOAuth');
    GoogleOAuth._module = angular.module(GoogleOAuth.pluginName, []);
    hawtioPluginLoader.addModule(GoogleOAuth.pluginName);
    GoogleOAuth._module.config(['$provide', function ($provide) {
            $provide.decorator('userDetails', ['$delegate', function ($delegate) {
                    if (GoogleOAuth.userProfile) {
                        return _.merge($delegate, GoogleOAuth.userProfile, {
                            username: GoogleOAuth.userProfile.fullName,
                            logout: function () {
                                GoogleOAuth.doLogout(GoogleOAuthConfig, GoogleOAuth.userProfile);
                            }
                        });
                    }
                    return $delegate;
                }]);
        }]);
    GoogleOAuth._module.config(['$httpProvider', function ($httpProvider) {
            if (GoogleOAuth.userProfile && GoogleOAuth.userProfile.token) {
                $httpProvider.defaults.headers.common = {
                    'Authorization': 'Bearer ' + GoogleOAuth.userProfile.token
                };
            }
        }]);
    GoogleOAuth._module.run(['userDetails', function (userDetails) {
        }]);
    hawtioPluginLoader.registerPreBootstrapTask({
        name: 'GoogleOAuth',
        task: function (next) {
            if (!window['GoogleOAuthConfig']) {
                GoogleOAuth.log.debug("oauth disabled");
                next();
                return;
            }
            if (!GoogleOAuthConfig.clientId ||
                !GoogleOAuthConfig.redirectURI ||
                !GoogleOAuthConfig.scope ||
                !GoogleOAuthConfig.authenticationURI) {
                GoogleOAuth.log.warn("Invalid oauth config, disabled oauth", GoogleOAuthConfig);
                next();
                return;
            }
            GoogleOAuth.log.debug("config: ", GoogleOAuthConfig);
            var currentURI = new URI(window.location.href);
            try {
                var userDetails = GoogleOAuth.getTokenStorage();
                if (userDetails && userDetails.token) {
                    GoogleOAuth.userProfile = userDetails;
                    // setupJQueryAjax(userDetails);
                    next();
                    return;
                }
                else {
                    // old format, let's force an update by re-authenticating
                    GoogleOAuth.clearTokenStorage();
                }
            }
            catch (err) {
                // must be a bad stored token
                GoogleOAuth.clearTokenStorage();
            }
            var authorizationCode = GoogleOAuth.checkAuthorizationCode(currentURI);
            if (authorizationCode) {
                GoogleOAuth.log.info("found an authorization code so need to go back to google and get a token");
                GoogleOAuth.exchangeCodeForToken(GoogleOAuthConfig, authorizationCode, {
                    uri: currentURI.toString(),
                }).done(function (response) {
                    if (response && response.access_token) {
                        var tmp = {
                            token: response.access_token,
                            expiry: response.expires_in,
                            type: response.token_type
                        };
                        GoogleOAuth.userProfile = _.merge(tmp, response, { provider: GoogleOAuth.pluginName });
                        GoogleOAuth.setTokenStorage(GoogleOAuth.userProfile);
                        GoogleOAuth.setupJQueryAjax(GoogleOAuth.userProfile);
                        GoogleOAuth.log.info("Logged in with URL: " + window.location.href);
                        // lets remove the auth code
                        var uri = new URI(window.location.href).removeQuery("code");
                        var target = uri.toString();
                        GoogleOAuth.log.info("Now redirecting to: " + target);
                        window.location.href = target;
                    }
                    else {
                        GoogleOAuth.log.debug("No access token received!");
                        GoogleOAuth.clearTokenStorage();
                        GoogleOAuth.doLogin(GoogleOAuthConfig, {
                            uri: currentURI.toString()
                        });
                    }
                }).fail(function (jqHXR, textStatus, errorThrown) {
                    GoogleOAuth.log.error("Failed to fetch auth code, status: ", textStatus, " error: ", errorThrown);
                    GoogleOAuth.clearTokenStorage();
                    GoogleOAuth.doLogin(GoogleOAuthConfig, {
                        uri: currentURI.toString()
                    });
                });
            }
            else {
                GoogleOAuth.clearTokenStorage();
                GoogleOAuth.doLogin(GoogleOAuthConfig, {
                    uri: currentURI.toString()
                });
            }
        }
    });
})(GoogleOAuth || (GoogleOAuth = {}));
/// <reference path="../../oauth.globals.ts"/>
var OSOAuth;
(function (OSOAuth) {
    OSOAuth.pluginName = 'hawtio-oauth-os';
    OSOAuth.log = Logger.get(OSOAuth.pluginName);
    // Keep this unset unless we have a token
    OSOAuth.userProfile = null;
})(OSOAuth || (OSOAuth = {}));
/// <reference path="osOAuthGlobals.ts"/>
var OSOAuth;
(function (OSOAuth) {
    var OS_TOKEN_STORAGE_KEY = 'osAuthCreds';
    function currentTimeSeconds() {
        return Math.floor(new Date().getTime() / 1000);
    }
    OSOAuth.currentTimeSeconds = currentTimeSeconds;
    function authenticatedHttpRequest(options, userDetails) {
        return $.ajax(_.extend(options, {
            beforeSend: function (request) {
                if (userDetails.token) {
                    request.setRequestHeader('Authorization', 'Bearer ' + userDetails.token);
                }
            }
        }));
    }
    OSOAuth.authenticatedHttpRequest = authenticatedHttpRequest;
    function doLogout(config, userDetails) {
        if (config === void 0) { config = window['OSOAuthConfig']; }
        if (userDetails === void 0) { userDetails = OSOAuth.userProfile; }
        var currentURI = new URI(window.location.href);
        var uri = new URI(config.oauth_authorize_uri);
        uri.path("/apis/oauth.openshift.io/v1/oauthaccesstokens/" + userDetails.token);
        // The following request may return 403 when delegated authentication with a service account as OAuthClient is used
        // See https://github.com/openshift/origin/issues/7011
        authenticatedHttpRequest({
            type: 'DELETE',
            url: uri.toString()
        }, userDetails).always(function () {
            clearTokenStorage();
            doLogin(OSOAuthConfig, {
                uri: currentURI.toString()
            });
        });
    }
    OSOAuth.doLogout = doLogout;
    function doLogin(config, options) {
        var clientId = config.oauth_client_id;
        var targetURI = config.oauth_authorize_uri;
        var uri = new URI(targetURI);
        uri.query({
            client_id: clientId,
            response_type: 'token',
            state: options.uri,
            redirect_uri: options.uri,
            scope: config.scope
        });
        var target = uri.toString();
        OSOAuth.log.debug("Redirecting to URI: ", target);
        window.location.href = target;
    }
    OSOAuth.doLogin = doLogin;
    function extractToken(uri) {
        var query = uri.query(true);
        OSOAuth.log.debug("Query: ", query);
        var fragmentParams = new URI("?" + uri.fragment()).query(true);
        OSOAuth.log.debug("FragmentParams: ", fragmentParams);
        if (fragmentParams.access_token && (fragmentParams.token_type === "bearer") || fragmentParams.token_type === "Bearer") {
            OSOAuth.log.debug("Got token");
            var localStorage_1 = Core.getLocalStorage();
            var creds = {
                token_type: fragmentParams.token_type,
                access_token: fragmentParams.access_token,
                expires_in: fragmentParams.expires_in,
                obtainedAt: currentTimeSeconds()
            };
            localStorage_1[OS_TOKEN_STORAGE_KEY] = angular.toJson(creds);
            delete fragmentParams.token_type;
            delete fragmentParams.access_token;
            delete fragmentParams.expires_in;
            delete fragmentParams.scope;
            uri.fragment("").query(query);
            var target = uri.toString();
            OSOAuth.log.debug("redirecting to: ", target);
            window.location.href = target;
            return creds;
        }
        else {
            OSOAuth.log.debug("No token in URI");
            return undefined;
        }
    }
    OSOAuth.extractToken = extractToken;
    function clearTokenStorage() {
        var localStorage = Core.getLocalStorage();
        delete localStorage[OS_TOKEN_STORAGE_KEY];
    }
    OSOAuth.clearTokenStorage = clearTokenStorage;
    function checkToken(uri) {
        var localStorage = Core.getLocalStorage();
        var answer = undefined;
        if (OS_TOKEN_STORAGE_KEY in localStorage) {
            try {
                answer = angular.fromJson(localStorage[OS_TOKEN_STORAGE_KEY]);
            }
            catch (e) {
                clearTokenStorage();
                // must be broken...
                OSOAuth.log.debug("Error extracting osAuthCreds value: ", e);
            }
        }
        if (!answer) {
            answer = extractToken(uri);
        }
        OSOAuth.log.debug("Using creds: ", answer);
        return answer;
    }
    OSOAuth.checkToken = checkToken;
})(OSOAuth || (OSOAuth = {}));
/// <reference path="osOAuthHelpers.ts"/>
var OSOAuth;
(function (OSOAuth) {
    HawtioOAuth.oauthPlugins.push('OSOAuth');
    OSOAuth._module = angular.module(OSOAuth.pluginName, ['ngIdle']);
    OSOAuth._module.config(['$provide', function ($provide) {
            $provide.decorator('userDetails', ['$delegate', function ($delegate) {
                    if (OSOAuth.userProfile) {
                        var logout_1 = $delegate.logout;
                        // The decorated delegate should ideally not be mutated though AuthService declares getters and setters that are not easy to port over the decoratee.
                        return _.merge($delegate, OSOAuth.userProfile, {
                            logout: function () {
                                logout_1();
                                OSOAuth.doLogout(OSOAuthConfig, OSOAuth.userProfile);
                            }
                        });
                    }
                    return $delegate;
                }]);
        }]);
    OSOAuth._module.config(['$httpProvider', function ($httpProvider) {
            if (OSOAuth.userProfile && OSOAuth.userProfile.token) {
                $httpProvider.defaults.headers.common = {
                    'Authorization': 'Bearer ' + OSOAuth.userProfile.token
                };
            }
        }]);
    var keepaliveUri = undefined;
    var keepaliveInterval = undefined;
    OSOAuth._module.config(['KeepaliveProvider', function (KeepaliveProvider) {
            OSOAuth.log.debug("keepalive URI: ", keepaliveUri);
            OSOAuth.log.debug("keepalive interval: ", keepaliveInterval);
            if (keepaliveUri && keepaliveInterval) {
                KeepaliveProvider.http(keepaliveUri);
                KeepaliveProvider.interval(keepaliveInterval);
            }
        }]);
    OSOAuth._module.run(['userDetails', 'Keepalive', '$rootScope', function (userDetails, Keepalive, $rootScope) {
            if (OSOAuth.userProfile && OSOAuth.userProfile.token) {
                userDetails.login(OSOAuth.userProfile.metadata.name, null, OSOAuth.userProfile.token);
                OSOAuth.log.debug("Starting keepalive");
                $rootScope.$on('KeepaliveResponse', function ($event, data, status) {
                    OSOAuth.log.debug("keepaliveStatus: ", status);
                    OSOAuth.log.debug("keepalive response: ", data);
                    if (status === 401) {
                        userDetails.logout();
                    }
                });
                Keepalive.start();
            }
        }]);
    hawtioPluginLoader.registerPreBootstrapTask({
        name: 'OSOAuth',
        task: function (next) {
            var openshiftConfig = null;
            try {
                openshiftConfig = window['OPENSHIFT_CONFIG'];
            }
            catch (e) {
                // ignore
            }
            if (openshiftConfig) {
                var token_1 = openshiftConfig.token;
                if (token_1) {
                    OSOAuth.log.warn("Loading OAuth token from server. We should switch to using a real OAuth login!");
                    OSOAuth.userProfile = {
                        token: token_1
                    };
                    $.ajaxSetup({
                        beforeSend: function (xhr) {
                            xhr.setRequestHeader('Authorization', 'Bearer ' + token_1);
                        }
                    });
                    next();
                    return;
                }
            }
            if (!window['OSOAuthConfig']) {
                OSOAuth.log.debug("oauth disabled");
                next();
                return;
            }
            if (!OSOAuthConfig.oauth_client_id ||
                !OSOAuthConfig.oauth_authorize_uri) {
                OSOAuth.log.debug("Invalid oauth config, disabled oauth");
                next();
                return;
            }
            OSOAuth.log.debug("config: ", OSOAuthConfig);
            var currentURI = new URI(window.location.href);
            var fragmentParams = OSOAuth.checkToken(currentURI);
            if (fragmentParams) {
                var tmp_1 = {
                    token: fragmentParams.access_token,
                    expiry: fragmentParams.expires_in,
                    type: fragmentParams.token_type,
                    obtainedAt: fragmentParams.obtainedAt || 0
                };
                var uri = void 0;
                if (openshiftConfig && openshiftConfig.master_uri) {
                    uri = new URI(openshiftConfig.master_uri);
                    uri.segment('oapi/v1/users/~');
                }
                else {
                    uri = new URI(OSOAuthConfig.oauth_authorize_uri);
                    uri.path('/oapi/v1/users/~');
                }
                keepaliveUri = uri.toString();
                OSOAuth.userProfile = tmp_1;
                $.ajax({
                    type: 'GET',
                    url: keepaliveUri,
                    success: function (response) {
                        _.merge(OSOAuth.userProfile, tmp_1, response, { provider: OSOAuth.pluginName });
                        var obtainedAt = Core.parseIntValue(OSOAuth.userProfile.obtainedAt) || 0;
                        var expiry = Core.parseIntValue(OSOAuth.userProfile.expiry) || 0;
                        if (obtainedAt) {
                            var remainingTime = obtainedAt + expiry - OSOAuth.currentTimeSeconds();
                            if (remainingTime > 0) {
                                keepaliveInterval = Math.round(remainingTime / 4);
                            }
                        }
                        if (!keepaliveInterval) {
                            keepaliveInterval = 10;
                        }
                        OSOAuth.log.debug("userProfile: ", OSOAuth.userProfile);
                        $.ajaxSetup({
                            beforeSend: function (xhr) { return xhr.setRequestHeader('Authorization', 'Bearer ' + OSOAuth.userProfile.token); }
                        });
                        next();
                    },
                    error: function (jqXHR, textStatus, errorThrown) {
                        // The request may have been cancelled as the browser refresh request in
                        // extractToken may be triggered before getting the AJAX response.
                        // In that case, let's just skip the error and go through another refresh cycle.
                        // See http://stackoverflow.com/questions/2000609/jquery-ajax-status-code-0 for more details.
                        if (jqXHR.status > 0) {
                            OSOAuth.log.error('Failed to fetch user info, status: ', textStatus, ' error: ', errorThrown);
                            OSOAuth.clearTokenStorage();
                            OSOAuth.doLogin(OSOAuthConfig, { uri: currentURI.toString() });
                        }
                    },
                    beforeSend: function (request) { return request.setRequestHeader('Authorization', 'Bearer ' + OSOAuth.userProfile.token); }
                });
            }
            else {
                OSOAuth.clearTokenStorage();
                OSOAuth.doLogin(OSOAuthConfig, {
                    uri: currentURI.toString()
                });
            }
        }
    });
    hawtioPluginLoader.addModule(OSOAuth.pluginName);
})(OSOAuth || (OSOAuth = {}));
/// <reference path="../oauth.globals.ts"/>
var HawtioKeycloak;
(function (HawtioKeycloak) {
    HawtioKeycloak.pluginName = 'hawtio-oauth-keycloak';
    HawtioKeycloak.log = Logger.get(HawtioKeycloak.pluginName);
    /**
     * Variable for Keycloak config that users can initialise.
     */
    HawtioKeycloak.config = null;
    HawtioKeycloak.keycloak = null;
    /**
     * Used by HawtioOAuth, must have a 'token' field when set,
     * otherwise leave null
     */
    HawtioKeycloak.userProfile = null;
})(HawtioKeycloak || (HawtioKeycloak = {}));
/// <reference path="keycloak.globals.ts"/>
var HawtioKeycloak;
(function (HawtioKeycloak) {
    var AuthInterceptor = /** @class */ (function () {
        AuthInterceptor.$inject = ["$q", "userDetails", "keycloakService"];
        function AuthInterceptor($q, userDetails, keycloakService) {
            'ngInject';
            var _this = this;
            this.$q = $q;
            this.userDetails = userDetails;
            this.keycloakService = keycloakService;
            this.request = function (request) {
                // bypass for local templates
                if (request.url.indexOf('http') !== 0) {
                    return request;
                }
                var deferred = _this.$q.defer();
                _this.addBearer(request, deferred);
                return _this.$q.when(deferred.promise);
            };
            this.responseError = function (rejection) {
                if (rejection.status === 401 && _this.userDetails.loggedIn) {
                    _this.userDetails.logout();
                }
                return _this.$q.reject(rejection);
            };
        }
        AuthInterceptor.Factory = ["$q", "userDetails", "keycloakService", function ($q, userDetails, keycloakService) {
            'ngInject';
            return new AuthInterceptor($q, userDetails, keycloakService);
        }];
        AuthInterceptor.prototype.addBearer = function (request, deferred) {
            var _this = this;
            this.keycloakService.updateToken(function (token) {
                _this.userDetails.token = token;
                request.headers.Authorization = 'Bearer ' + token;
                deferred.notify();
                deferred.resolve(request);
            });
        };
        return AuthInterceptor;
    }());
    HawtioKeycloak.AuthInterceptor = AuthInterceptor;
})(HawtioKeycloak || (HawtioKeycloak = {}));
var HawtioKeycloak;
(function (HawtioKeycloak) {
    var TOKEN_UPDATE_INTERVAL = 5; // 5 sec.
    var KeycloakService = /** @class */ (function () {
        function KeycloakService(enabled, keycloak) {
            this.enabled = enabled;
            this.keycloak = keycloak;
        }
        KeycloakService.prototype.updateToken = function (onSuccess, onError) {
            var _this = this;
            this.keycloak.updateToken(TOKEN_UPDATE_INTERVAL)
                .success(function () {
                var token = _this.keycloak.token;
                onSuccess(token);
            })
                .error(function () {
                HawtioKeycloak.log.error("Couldn't update token");
                if (onError) {
                    onError();
                }
            });
        };
        KeycloakService.prototype.setupJQueryAjax = function (userDetails) {
            var _this = this;
            HawtioKeycloak.log.debug("Setting authorization header to token");
            $.ajaxSetup({
                beforeSend: function (xhr, settings) {
                    if (_this.keycloak.authenticated && !_this.keycloak.isTokenExpired(TOKEN_UPDATE_INTERVAL)) {
                        // hawtio uses BearerTokenLoginModule on server side
                        xhr.setRequestHeader('Authorization', Core.getBasicAuthHeader(HawtioKeycloak.keycloak.profile.username, HawtioKeycloak.keycloak.token));
                    }
                    else {
                        HawtioKeycloak.log.debug("Skipped request", settings.url, "for now.");
                        _this.updateToken(function (token) {
                            if (token) {
                                HawtioKeycloak.log.debug('Keycloak token refreshed. Set new value to userDetails');
                                userDetails.token = token;
                            }
                            HawtioKeycloak.log.debug("Re-sending request after successfully update keycloak token:", settings.url);
                            $.ajax(settings);
                        }, function () { return userDetails.logout(); });
                        return false;
                    }
                }
            });
        };
        return KeycloakService;
    }());
    HawtioKeycloak.KeycloakService = KeycloakService;
})(HawtioKeycloak || (HawtioKeycloak = {}));
/// <reference path="keycloak.globals.ts"/>
/// <reference path="keycloak.interceptor.ts"/>
/// <reference path="keycloak.service.ts"/>
var HawtioKeycloak;
(function (HawtioKeycloak) {
    applyAuthInterceptor.$inject = ["$provide", "$httpProvider"];
    loginUserDetails.$inject = ["userDetails", "keycloakService", "postLogoutTasks"];
    configureIdleTimeout.$inject = ["userDetails", "Idle", "$rootScope"];
    HawtioOAuth.oauthPlugins.push('HawtioKeycloak');
    angular
        .module(HawtioKeycloak.pluginName, ['ngIdle'])
        .config(applyAuthInterceptor)
        .factory('keycloakService', function () { return new HawtioKeycloak.KeycloakService(isKeycloakEnabled(), HawtioKeycloak.keycloak); })
        .run(loginUserDetails)
        .run(configureIdleTimeout);
    function isKeycloakEnabled() {
        if (HawtioKeycloak.keycloak && HawtioKeycloak.userProfile) {
            return true;
        }
        else {
            return false;
        }
    }
    function applyAuthInterceptor($provide, $httpProvider) {
        'ngInject';
        // only add the interceptor if we have keycloak otherwise
        // we'll get an undefined exception in the interceptor
        if (isKeycloakEnabled()) {
            HawtioKeycloak.log.debug("Applying AuthInterceptor to $http");
            $httpProvider.interceptors.push(HawtioKeycloak.AuthInterceptor.Factory);
        }
    }
    function loginUserDetails(userDetails, keycloakService, postLogoutTasks) {
        'ngInject';
        if (!isKeycloakEnabled()) {
            return;
        }
        userDetails.login(HawtioKeycloak.userProfile.username, null, HawtioKeycloak.userProfile.token);
        keycloakService.setupJQueryAjax(userDetails);
        HawtioKeycloak.log.debug("Register 'LogoutKeycloak' to postLogoutTasks");
        postLogoutTasks.addTask('LogoutKeycloak', function () {
            HawtioKeycloak.log.info("Log out Keycloak");
            HawtioKeycloak.keycloak.logout();
        });
    }
    function configureIdleTimeout(userDetails, Idle, $rootScope) {
        'ngInject';
        if (!isKeycloakEnabled()) {
            HawtioKeycloak.log.debug("Not enabling idle timeout");
            return;
        }
        HawtioKeycloak.log.debug("Enabling idle timeout");
        Idle.watch();
        $rootScope.$on('IdleTimeout', function () {
            HawtioKeycloak.log.debug("Idle timeout triggered");
            // let the end application handle this event
            // userDetails.logout();
        });
        $rootScope.$on('Keepalive', function () {
            if (HawtioKeycloak.keycloak) {
                HawtioKeycloak.keycloak.updateToken(5).success(function () {
                    userDetails.token = HawtioKeycloak.keycloak.token;
                });
            }
        });
    }
    hawtioPluginLoader
        .addModule(HawtioKeycloak.pluginName)
        .registerPreBootstrapTask({
        name: 'HawtioKeycloak',
        task: function (next) { return loadKeycloakJs(next); },
        depends: "KeycloakLoginBootstrap"
    });
    function loadKeycloakJs(callback) {
        if (!HawtioKeycloak.config) {
            HawtioKeycloak.log.debug("Keycloak disabled");
            callback();
            return;
        }
        var keycloakJsUri = new URI(HawtioKeycloak.config.url).segment('js/keycloak.js').toString();
        $.getScript(keycloakJsUri)
            .done(function (script, textStatus) { return initKeycloak(callback); })
            .fail(function (response) {
            HawtioKeycloak.log.warn("Error fetching keycloak adapter:", response);
            callback();
        });
    }
    function initKeycloak(callback) {
        HawtioKeycloak.keycloak = Keycloak(HawtioKeycloak.config);
        HawtioKeycloak.keycloak.init({ onLoad: 'login-required' })
            .success(function (authenticated) {
            HawtioKeycloak.log.debug("Authenticated:", authenticated);
            if (!authenticated) {
                HawtioKeycloak.keycloak.login({ redirectUri: window.location.href });
                return;
            }
            HawtioKeycloak.keycloak.loadUserProfile()
                .success(function (profile) {
                HawtioKeycloak.userProfile = profile;
                HawtioKeycloak.userProfile.token = HawtioKeycloak.keycloak.token;
                callback();
            })
                .error(function () {
                HawtioKeycloak.log.debug("Failed to load user profile");
                callback();
            });
        })
            .error(function (error) {
            HawtioKeycloak.log.warn("Failed to initialize Keycloak, token unavailable", error);
            callback();
        });
    }
})(HawtioKeycloak || (HawtioKeycloak = {}));
/// <reference path="oauth.helper.ts"/>
/// <reference path="github/ts/githubPlugin.ts"/>
/// <reference path="googleOAuth/ts/googleOAuthPlugin.ts"/>
/// <reference path="osOAuth/ts/osOAuthPlugin.ts"/>
/// <reference path="keycloak/keycloak.module.ts"/>
var HawtioOAuth;
(function (HawtioOAuth) {
    addProductInfo.$inject = ["aboutService"];
    var hawtioOAuthModule = angular
        .module(HawtioOAuth.pluginName, [
        'ngIdle',
        GithubOAuth.pluginName,
        GoogleOAuth.pluginName,
        HawtioKeycloak.pluginName,
        OSOAuth.pluginName
    ])
        .run(addProductInfo)
        .name;
    hawtioPluginLoader.addModule(HawtioOAuth.pluginName);
    function addProductInfo(aboutService) {
        'ngInject';
        aboutService.addProductInfo('Hawtio OAuth', '4.0.7');
    }
    /*
     * Fetch oauth config
     */
    hawtioPluginLoader.registerPreBootstrapTask({
        name: 'HawtioOAuthConfig',
        task: function (next) {
            $.getScript('oauth/config.js').always(next);
        }
    });
    /*
     * Global pre-bootstrap task that plugins can use to wait
     * until all oauth plugins have been processed
     *
     * OAuth plugins can add to this list via:
     *
     *   HawtioOAuth.oauthPlugins.push(<plugin name>);
     *
     * and then use a named task with the same name as <plugin name>
     */
    hawtioPluginLoader.registerPreBootstrapTask({
        name: 'HawtioOAuthBootstrap',
        depends: HawtioOAuth.oauthPlugins,
        task: function (next) {
            HawtioOAuth.getUserProfile();
            HawtioOAuth.log.info("All OAuth plugins have been executed:", HawtioOAuth.oauthPlugins);
            next();
        }
    });
})(HawtioOAuth || (HawtioOAuth = {}));
/// <reference path="githubPlugin.ts"/>
var GithubOAuth;
(function (GithubOAuth) {
    GithubOAuth._module.component('githubPreferences', {
        template: "\n    <div class=\"alert alert-success\" role=\"alert\" ng-if=\"model.accessToken && !model.trying\">\n      <p><img style=\"width: 64px; height: 64px;\" ng-src=\"{{model.avatarURL}}\">&nbsp;Logged in as <strong>{{model.name}}</strong>, <a href=\"\" ng-click=\"check()\">Check access</a><strong ng-if=\"model.data.app.name\">&nbsp;<i class=\"fa fa-check green\"></i></strong></p>\n      <p>Github access is enabled, <a href=\"\" ng-click=\"clearToken()\">disable access</a></p>\n    </div>\n    <div class=\"alert alert-warning\" role=\"alert\" ng-if=\"model.error\">\n      <button type=\"button\" class=\"close\" ng-click=\"model.error = false\" aria-label=\"Close\"><span aria-hidden=\"true\">&times;</span></button>\n      Error logging in: {{model.data.statusText}} - {{model.data.responseJSON.message}}\n    </div>\n    <div class=\"alert alert-info\" ng-show=\"model.trying\">\n      <div class=\"align-center\">\n        <div class=\"spinner spinner-lg\"></div>\n      </div>\n    </div>\n    <form ng-if=\"!model.accessToken\" class=\"form-horizontal\">\n      <p>Log into Github here to enable access to your Github organizations and repositories</p>\n      <div class=\"form-group\">\n        <label class=\"col-sm-2 control-label\" for=\"username\">User Name</label>\n        <div class=\"col-sm-10\">\n          <input class=\"form-control\" id=\"username\" type=\"text\" ng-model=\"model.username\">\n        </div>\n      </div>\n      <div class=\"form-group\">\n        <label class=\"col-sm-2 control-label\" for=\"password\">Password</label>\n        <div class=\"col-sm-10\">\n          <input class=\"form-control\" id=\"password\" type=\"password\" ng-model=\"model.password\">\n        </div>\n      </div>\n      <button class=\"btn btn-success pull-right\" ng-disabled=\"disabled()\" ng-click=\"login()\">Log In</button>\n    </form>\n    <!-- <pre>{{model | json}}</pre> -->\n    ",
        controllerAs: 'github',
        controller: ['$scope', 'githubOAuthSettings', function GithubPreferenceController($scope, githubOAuthSettings) {
                var model = $scope.model = {
                    trying: false,
                    error: false,
                    username: githubOAuthSettings.username,
                    avatarURL: githubOAuthSettings.avatarURL,
                    name: githubOAuthSettings.name,
                    password: undefined,
                    accessToken: githubOAuthSettings.accessToken,
                    data: undefined
                };
                var settings = $scope.settings = githubOAuthSettings;
                $scope.disabled = function () {
                    return Core.isBlank(model.username) || Core.isBlank(model.password);
                };
                var error = function (data) {
                    model.trying = false;
                    model.error = true;
                    model.data = data;
                    Core.$apply($scope);
                };
                $scope.clearToken = function () {
                    model.trying = true;
                    $.ajax(GithubOAuth.getTokenCheckAuthURL(githubOAuthSettings), {
                        method: 'DELETE',
                        headers: {
                            'Authorization': GithubOAuth.getTokenCheckAuthHeader(githubOAuthSettings)
                        },
                        success: function (data) {
                            model.trying = false;
                            model.data = data;
                            model.accessToken = undefined;
                            model.name = undefined;
                            model.avatarURL = undefined;
                            GithubOAuth.storeSettings(model, githubOAuthSettings);
                            Core.$apply($scope);
                        },
                        error: error,
                        beforeSend: GithubOAuth.emptyBeforeSend
                    });
                };
                $scope.check = function () {
                    model.trying = true;
                    $.ajax(GithubOAuth.getTokenCheckAuthURL(githubOAuthSettings), {
                        method: 'GET',
                        headers: {
                            'Authorization': GithubOAuth.getTokenCheckAuthHeader(githubOAuthSettings)
                        },
                        success: function (data) {
                            model.trying = false;
                            model.data = data;
                            Core.$apply($scope);
                        },
                        error: function (data) {
                            model.accessToken = undefined;
                            model.name = undefined;
                            model.avatarURL = undefined;
                            GithubOAuth.storeSettings(model, githubOAuthSettings);
                            error(data);
                        },
                        beforeSend: GithubOAuth.emptyBeforeSend
                    });
                };
                $scope.login = function () {
                    model.error = false;
                    model.trying = true;
                    var headers = {
                        'Authorization': Core.getBasicAuthHeader(model.username, model.password)
                    };
                    GithubOAuth.storeSettings(model, githubOAuthSettings);
                    $.ajax(settings.loginURL, {
                        method: 'GET',
                        headers: headers,
                        success: function (data) {
                            model.name = data.name;
                            model.avatarURL = data.avatar_url;
                            $.ajax(settings.tokenURL, {
                                method: 'POST',
                                contentType: 'application/json; charset=UTF-8',
                                mimeType: 'application/json',
                                dataType: 'json',
                                processData: false,
                                data: angular.toJson({
                                    client_id: githubOAuthSettings.clientId,
                                    client_secret: githubOAuthSettings.clientSecret,
                                    note: 'hawtio console access token',
                                    scopes: githubOAuthSettings.scopes
                                }),
                                headers: headers,
                                success: function (data) {
                                    model.trying = false;
                                    model.accessToken = data.token;
                                    delete model.password;
                                    GithubOAuth.storeSettings(model, githubOAuthSettings);
                                    Core.$apply($scope);
                                },
                                error: error,
                                beforeSend: GithubOAuth.emptyBeforeSend
                            });
                        },
                        error: error,
                        beforeSend: GithubOAuth.emptyBeforeSend
                    });
                };
            }]
    });
})(GithubOAuth || (GithubOAuth = {}));

angular.module('hawtio-oauth-templates', []).run(['$templateCache', function($templateCache) {$templateCache.put('plugins/github/html/githubPreferences.html','<github-preferences></github-preferences>\n');}]); hawtioPluginLoader.addModule("hawtio-oauth-templates");