/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.rest.api.service.impl;

import io.gravitee.repository.exceptions.TechnicalException;
import io.gravitee.repository.management.api.ApiKeyRepository;
import io.gravitee.repository.management.api.search.ApiKeyCriteria;
import io.gravitee.repository.management.model.ApiKey;
import io.gravitee.repository.management.model.Audit;
import io.gravitee.rest.api.model.ApiKeyEntity;
import io.gravitee.rest.api.model.ApplicationEntity;
import io.gravitee.rest.api.model.PrimaryOwnerEntity;
import io.gravitee.rest.api.model.SubscriptionEntity;
import io.gravitee.rest.api.model.SubscriptionStatus;
import io.gravitee.rest.api.model.key.ApiKeyQuery;
import io.gravitee.rest.api.model.v4.api.GenericApiModel;
import io.gravitee.rest.api.model.v4.plan.GenericPlanEntity;
import io.gravitee.rest.api.model.v4.plan.PlanSecurityType;
import io.gravitee.rest.api.service.ApiKeyGenerator;
import io.gravitee.rest.api.service.ApiKeyService;
import io.gravitee.rest.api.service.ApplicationService;
import io.gravitee.rest.api.service.AuditService;
import io.gravitee.rest.api.service.NotifierService;
import io.gravitee.rest.api.service.SubscriptionService;
import io.gravitee.rest.api.service.common.ExecutionContext;
import io.gravitee.rest.api.service.common.UuidString;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyActivatedException;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyExistingException;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyExpiredException;
import io.gravitee.rest.api.service.exceptions.ApiKeyNotFoundException;
import io.gravitee.rest.api.service.exceptions.InvalidApplicationApiKeyModeException;
import io.gravitee.rest.api.service.exceptions.SubscriptionClosedException;
import io.gravitee.rest.api.service.exceptions.SubscriptionNotActiveException;
import io.gravitee.rest.api.service.exceptions.TechnicalManagementException;
import io.gravitee.rest.api.service.impl.TransactionalService;
import io.gravitee.rest.api.service.notification.ApiHook;
import io.gravitee.rest.api.service.notification.NotificationParamsBuilder;
import io.gravitee.rest.api.service.v4.ApiTemplateService;
import io.gravitee.rest.api.service.v4.PlanSearchService;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

@Component
public class ApiKeyServiceImpl
extends TransactionalService
implements ApiKeyService {
    private final Logger LOGGER = LoggerFactory.getLogger(ApiKeyServiceImpl.class);
    @Lazy
    @Autowired
    private ApiKeyRepository apiKeyRepository;
    @Autowired
    private SubscriptionService subscriptionService;
    @Autowired
    private ApiKeyGenerator apiKeyGenerator;
    @Autowired
    private ApplicationService applicationService;
    @Autowired
    private PlanSearchService planSearchService;
    @Autowired
    private AuditService auditService;
    @Autowired
    private NotifierService notifierService;
    @Autowired
    private ApiTemplateService apiTemplateService;

    @Override
    public ApiKeyEntity generate(ExecutionContext executionContext, ApplicationEntity application, SubscriptionEntity subscription, String customApiKey) {
        if (!application.hasApiKeySharedMode()) {
            return this.generate(executionContext, subscription, customApiKey);
        }
        return this.findOrGenerate(executionContext, application, subscription, customApiKey);
    }

    @Override
    public ApiKeyEntity renew(ExecutionContext executionContext, SubscriptionEntity subscription) {
        return this.renew(executionContext, subscription, null);
    }

    @Override
    public ApiKeyEntity renew(ExecutionContext executionContext, SubscriptionEntity subscription, String customApiKey) {
        if (!subscription.getStatus().equals((Object)SubscriptionStatus.ACCEPTED) && !subscription.getStatus().equals((Object)SubscriptionStatus.PAUSED)) {
            throw new SubscriptionNotActiveException(subscription);
        }
        GenericPlanEntity genericPlanEntity = this.planSearchService.findById(executionContext, subscription.getPlan());
        PlanSecurityType planSecurityType = null;
        if (genericPlanEntity.getPlanSecurity() != null) {
            planSecurityType = PlanSecurityType.valueOfLabel((String)genericPlanEntity.getPlanSecurity().getType());
        }
        if (PlanSecurityType.API_KEY != planSecurityType) {
            throw new TechnicalManagementException("Invalid plan security.");
        }
        try {
            this.LOGGER.debug("Renew API Key for subscription {}", (Object)subscription.getId());
            ApiKey newApiKey = this.generateForSubscription(executionContext, subscription, customApiKey);
            newApiKey = this.apiKeyRepository.create(newApiKey);
            this.expireApiKeys(executionContext, this.apiKeyRepository.findBySubscription(subscription.getId()), newApiKey);
            ApiKeyEntity newApiKeyEntity = this.convert(executionContext, newApiKey);
            this.createAuditLog(executionContext, newApiKeyEntity, null, ApiKey.AuditEvent.APIKEY_RENEWED, newApiKey.getCreatedAt());
            this.triggerNotifierService(executionContext, ApiHook.APIKEY_RENEWED, newApiKey, newApiKeyEntity.getApplication(), newApiKeyEntity.getSubscriptions());
            return newApiKeyEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to renew an API Key for {}", (Object)subscription.getId(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to renew an API Key for %s", subscription.getId()), ex);
        }
    }

    @Override
    public ApiKeyEntity renew(ExecutionContext executionContext, ApplicationEntity application) {
        if (!application.hasApiKeySharedMode()) {
            throw new InvalidApplicationApiKeyModeException("Can't renew an API Key on application that doesn't use shared API Key mode");
        }
        try {
            this.LOGGER.debug("Renew API Key for application {}", (Object)application.getId());
            ApiKey newApiKey = this.generateForApplication(application);
            newApiKey = this.apiKeyRepository.create(newApiKey);
            List allApiKeys = this.apiKeyRepository.findByApplication(application.getId());
            this.expireApiKeys(executionContext, allApiKeys, newApiKey);
            this.addSharedSubscriptions(allApiKeys, newApiKey);
            ApiKeyEntity newApiKeyEntity = this.convert(executionContext, newApiKey);
            this.createAuditLog(executionContext, newApiKeyEntity, null, ApiKey.AuditEvent.APIKEY_RENEWED, newApiKey.getCreatedAt());
            this.triggerNotifierService(executionContext, ApiHook.APIKEY_RENEWED, newApiKey, newApiKeyEntity.getApplication(), newApiKeyEntity.getSubscriptions());
            return newApiKeyEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to renew an API Key for application {}", (Object)application.getId(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to renew an API Key for application %s", application.getId()), ex);
        }
    }

    private void expireApiKeys(ExecutionContext executionContext, Collection<ApiKey> apiKeys, ApiKey activeApiKey) throws TechnicalException {
        Instant expirationInst = activeApiKey.getCreatedAt().toInstant().plus(Duration.ofHours(2L));
        Date expirationDate = Date.from(expirationInst);
        for (ApiKey apiKey : apiKeys) {
            ApiKeyEntity apiKeyEntity = this.convert(executionContext, apiKey);
            if (apiKey.equals((Object)activeApiKey) || apiKeyEntity.isExpired()) continue;
            this.updateExpirationDate(executionContext, expirationDate, apiKeyEntity, apiKey);
        }
    }

    private void addSharedSubscriptions(Collection<ApiKey> apiKeys, ApiKey activeApiKey) throws TechnicalException {
        HashSet subscriptions = new HashSet();
        apiKeys.forEach(apiKey -> subscriptions.addAll(apiKey.getSubscriptions()));
        activeApiKey.setSubscriptions(new ArrayList(subscriptions));
        activeApiKey.setUpdatedAt(new Date());
        this.apiKeyRepository.update(activeApiKey);
    }

    private ApiKeyEntity findOrGenerate(ExecutionContext executionContext, ApplicationEntity application, SubscriptionEntity subscription, String customApiKey) {
        return this.findByApplication(executionContext, application.getId()).stream().map(apiKey -> this.addSubscription(executionContext, (ApiKeyEntity)apiKey, subscription)).max(Comparator.comparing(ApiKeyEntity::isRevoked, Comparator.reverseOrder()).thenComparing(ApiKeyEntity::getExpireAt, Comparator.nullsLast(Comparator.naturalOrder()))).orElseGet(() -> this.generate(executionContext, subscription, customApiKey));
    }

    private ApiKeyEntity addSubscription(ExecutionContext executionContext, ApiKeyEntity apiKeyEntity, SubscriptionEntity subscription) {
        try {
            return this.convert(executionContext, (ApiKey)this.apiKeyRepository.addSubscription(apiKeyEntity.getId(), subscription.getId()).orElseThrow(ApiKeyNotFoundException::new));
        }
        catch (TechnicalException e) {
            this.LOGGER.error("An error occurred while trying to add subscription to API Key", (Throwable)e);
            throw new TechnicalManagementException("An error occurred while trying to a add subscription to API Key");
        }
    }

    private ApiKeyEntity generate(ExecutionContext executionContext, SubscriptionEntity subscription, String customApiKey) {
        try {
            this.LOGGER.debug("Generate an API Key for subscription {}", (Object)subscription);
            ApiKey apiKey = this.generateForSubscription(executionContext, subscription, customApiKey);
            apiKey = this.apiKeyRepository.create(apiKey);
            ApiKeyEntity apiKeyEntity = this.convert(executionContext, apiKey);
            this.createAuditLog(executionContext, apiKeyEntity, null, ApiKey.AuditEvent.APIKEY_CREATED, apiKey.getCreatedAt());
            return apiKeyEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to generate an API Key for {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to generate an API Key for %s", subscription), ex);
        }
    }

    private ApiKey generateForSubscription(ExecutionContext executionContext, SubscriptionEntity subscription, String customApiKey) {
        if (StringUtils.isNotEmpty((CharSequence)customApiKey) && !this.canCreate(executionContext, customApiKey, subscription)) {
            throw new ApiKeyAlreadyExistingException();
        }
        Date now = new Date();
        if (subscription.getEndingAt() != null && subscription.getEndingAt().before(now)) {
            throw new SubscriptionClosedException(subscription.getId());
        }
        ApiKey apiKey = this.generateForApplication(subscription.getApplication(), customApiKey);
        apiKey.setSubscriptions(List.of(subscription.getId()));
        apiKey.setEnvironmentId(executionContext.getEnvironmentId());
        apiKey.setExpireAt(subscription.getEndingAt());
        return apiKey;
    }

    private ApiKey generateForApplication(ApplicationEntity application) {
        return this.generateForApplication(application.getId(), null);
    }

    private ApiKey generateForApplication(String application, String customApiKey) {
        ApiKey apiKey = new ApiKey();
        apiKey.setId(UuidString.generateRandom());
        apiKey.setApplication(application);
        apiKey.setEnvironmentId(apiKey.getEnvironmentId());
        apiKey.setCreatedAt(new Date());
        apiKey.setUpdatedAt(apiKey.getCreatedAt());
        apiKey.setKey(StringUtils.isNotEmpty((CharSequence)customApiKey) ? customApiKey : this.apiKeyGenerator.generate());
        return apiKey;
    }

    @Override
    public ApiKeyEntity reactivate(ExecutionContext executionContext, ApiKeyEntity apiKeyEntity) {
        try {
            ApiKey key = (ApiKey)this.apiKeyRepository.findById(apiKeyEntity.getId()).orElseThrow(ApiKeyNotFoundException::new);
            this.LOGGER.debug("Reactivate API Key id {}", (Object)apiKeyEntity.getId());
            if (!key.isRevoked() && !this.convert(executionContext, key).isExpired()) {
                throw new ApiKeyAlreadyActivatedException();
            }
            ApiKey previousApiKey = new ApiKey(key);
            key.setRevoked(false);
            key.setUpdatedAt(new Date());
            key.setRevokedAt(null);
            if (!apiKeyEntity.getApplication().hasApiKeySharedMode()) {
                SubscriptionEntity subscription = this.subscriptionService.findById((String)key.getSubscriptions().get(0));
                if (subscription.getStatus() != SubscriptionStatus.PAUSED && subscription.getStatus() != SubscriptionStatus.ACCEPTED) {
                    throw new SubscriptionNotActiveException(subscription);
                }
                key.setExpireAt(subscription.getEndingAt());
            }
            ApiKey updated = this.apiKeyRepository.update(key);
            ApiKeyEntity updatedEntity = this.convert(executionContext, updated);
            this.createAuditLog(executionContext, updatedEntity, previousApiKey, ApiKey.AuditEvent.APIKEY_REACTIVATED, key.getUpdatedAt());
            return updatedEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to reactivate an api key", (Throwable)ex);
            throw new TechnicalManagementException("An error occurs while trying to reactivate an API Key", ex);
        }
    }

    private void checkApiKeyExpired(ExecutionContext executionContext, ApiKey key) {
        if (key.isRevoked() || this.convert(executionContext, key).isExpired()) {
            throw new ApiKeyAlreadyExpiredException();
        }
    }

    @Override
    public ApiKeyEntity findById(ExecutionContext executionContext, String keyId) {
        try {
            return this.apiKeyRepository.findById(keyId).map(apiKey -> this.convert(executionContext, (ApiKey)apiKey)).orElseThrow(ApiKeyNotFoundException::new);
        }
        catch (TechnicalException e) {
            String message = String.format("An error occurs while trying to find a key with id %s", keyId);
            this.LOGGER.error(message, (Throwable)e);
            throw new TechnicalManagementException(message, e);
        }
    }

    @Override
    public List<ApiKeyEntity> findByKey(ExecutionContext executionContext, String apiKey) {
        try {
            this.LOGGER.debug("Find API Keys by key");
            return this.apiKeyRepository.findByKey(apiKey).stream().map(apiKey1 -> this.convert(executionContext, (ApiKey)apiKey1)).collect(Collectors.toList());
        }
        catch (TechnicalException e) {
            this.LOGGER.error("An error occurs while finding API Keys", (Throwable)e);
            throw new TechnicalManagementException("An error occurs while finding API Keys", e);
        }
    }

    @Override
    public List<ApiKeyEntity> findByKeyAndEnvironmentId(ExecutionContext executionContext, String apiKey) {
        try {
            this.LOGGER.debug("Find API Keys by key and environment id");
            return this.apiKeyRepository.findByKeyAndEnvironmentId(apiKey, executionContext.getEnvironmentId()).stream().map(apiKey1 -> this.convert(executionContext, (ApiKey)apiKey1)).collect(Collectors.toList());
        }
        catch (TechnicalException e) {
            this.LOGGER.error("An error occurs while finding API Keys", (Throwable)e);
            throw new TechnicalManagementException("An error occurs while finding API Keys", e);
        }
    }

    @Override
    public List<ApiKeyEntity> findBySubscription(ExecutionContext executionContext, String subscription) {
        try {
            this.LOGGER.debug("Find API Keys for subscription {}", (Object)subscription);
            SubscriptionEntity subscriptionEntity = this.subscriptionService.findById(subscription);
            Set keys = this.apiKeyRepository.findBySubscription(subscriptionEntity.getId());
            return keys.stream().map(apiKey -> this.convert(executionContext, (ApiKey)apiKey)).sorted((o1, o2) -> o2.getCreatedAt().compareTo(o1.getCreatedAt())).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while finding API Keys for subscription {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while finding API Keys for subscription %s", subscription), ex);
        }
    }

    @Override
    public ApiKeyEntity findByKeyAndApi(ExecutionContext executionContext, String apiKey, String apiId) {
        try {
            this.LOGGER.debug("Find an API Key by key for API {}", (Object)apiId);
            ApiKey key = (ApiKey)this.apiKeyRepository.findByKeyAndApi(apiKey, apiId).orElseThrow(ApiKeyNotFoundException::new);
            return this.convert(executionContext, key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to find an API Key for API {}", (Object)apiId, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to find an API Key for API %s", apiId), ex);
        }
    }

    @Override
    public List<ApiKeyEntity> findByApplication(ExecutionContext executionContext, String applicationId) {
        try {
            return this.apiKeyRepository.findByApplication(applicationId).stream().map(apiKey -> this.convert(executionContext, (ApiKey)apiKey)).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to find API Keys for application {}", (Object)applicationId, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to find API Keys for application %s", applicationId), ex);
        }
    }

    @Override
    public ApiKeyEntity update(ExecutionContext executionContext, ApiKeyEntity apiKeyEntity) {
        try {
            this.LOGGER.debug("Update API Key with id {}", (Object)apiKeyEntity.getId());
            ApiKey key = (ApiKey)this.apiKeyRepository.findById(apiKeyEntity.getId()).orElseThrow(ApiKeyNotFoundException::new);
            this.checkApiKeyExpired(executionContext, key);
            key.setEnvironmentId(apiKeyEntity.getEnvironmentId());
            key.setSubscriptions(apiKeyEntity.getSubscriptionIds());
            key.setPaused(apiKeyEntity.isPaused());
            this.updateExpirationDate(executionContext, apiKeyEntity.getExpireAt(), apiKeyEntity, key);
            return this.convert(executionContext, key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while updating an API Key with id {}", (Object)apiKeyEntity.getId(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while updating an API Key with id %s", apiKeyEntity.getId()), ex);
        }
    }

    @Override
    public ApiKeyEntity updateDaysToExpirationOnLastNotification(ExecutionContext executionContext, ApiKeyEntity apiKeyEntity, Integer value) {
        try {
            return this.apiKeyRepository.findById(apiKeyEntity.getId()).map(dbApiKey -> {
                dbApiKey.setDaysToExpirationOnLastNotification(value);
                try {
                    return this.apiKeyRepository.update(dbApiKey);
                }
                catch (TechnicalException ex) {
                    this.LOGGER.error("An error occurs while trying to update ApiKey with id {}", (Object)dbApiKey.getId(), (Object)ex);
                    throw new TechnicalManagementException(String.format("An error occurs while trying to update ApiKey with id %s", dbApiKey.getId()), ex);
                }
            }).map(apiKey -> this.convert(executionContext, (ApiKey)apiKey)).orElseThrow(ApiKeyNotFoundException::new);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to update apiKey", (Throwable)ex);
            throw new TechnicalManagementException("An error occurs while trying to update apiKey", ex);
        }
    }

    @Override
    public boolean canCreate(ExecutionContext executionContext, String apiKeyValue, SubscriptionEntity subscription) {
        return this.canCreate(executionContext, apiKeyValue, subscription.getApi(), subscription.getApplication());
    }

    @Override
    public boolean canCreate(ExecutionContext executionContext, String apiKey, String apiId, String applicationId) {
        this.LOGGER.debug("Check if an API Key can be created for api {} and application {}", (Object)apiId, (Object)applicationId);
        return this.findByKeyAndEnvironmentId(executionContext, apiKey).stream().noneMatch(existingKey -> this.isConflictingKey((ApiKeyEntity)existingKey, apiId, applicationId));
    }

    private boolean isConflictingKey(ApiKeyEntity existingKey, String apiId, String applicationId) {
        if (!existingKey.getApplication().getId().equals(applicationId)) {
            return true;
        }
        return existingKey.getSubscriptions().stream().map(SubscriptionEntity::getApi).anyMatch(apiId::equals);
    }

    @Override
    public Collection<ApiKeyEntity> search(ExecutionContext executionContext, ApiKeyQuery query) {
        try {
            this.LOGGER.debug("Search API Keys {}", (Object)query);
            ApiKeyCriteria apiKeyCriteria = this.toApiKeyCriteria(query);
            return this.apiKeyRepository.findByCriteria(apiKeyCriteria).stream().map(apiKey -> this.convert(executionContext, (ApiKey)apiKey)).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to search API Keys: {}", (Object)query, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to search API Keys: {}", query), ex);
        }
    }

    @Override
    public void delete(String apiKey) {
    }

    private void updateExpirationDate(ExecutionContext executionContext, Date expirationDate, ApiKeyEntity apiKeyEntity, ApiKey key) throws TechnicalException {
        SubscriptionEntity subscription;
        Date now = new Date();
        key.setUpdatedAt(now);
        if (key.isRevoked()) {
            this.apiKeyRepository.update(key);
            return;
        }
        if (expirationDate != null && expirationDate.before(now)) {
            expirationDate = now;
        }
        if (!(apiKeyEntity.getApplication() == null || apiKeyEntity.getApplication().hasApiKeySharedMode() || key.getSubscriptions().isEmpty() || (subscription = this.subscriptionService.findById((String)key.getSubscriptions().get(0))).getEndingAt() == null || expirationDate != null && subscription.getEndingAt().compareTo(expirationDate) >= 0)) {
            expirationDate = subscription.getEndingAt();
        }
        ApiKey oldkey = new ApiKey(key);
        key.setExpireAt(expirationDate);
        key.setDaysToExpirationOnLastNotification(null);
        this.apiKeyRepository.update(key);
        this.createAuditLog(executionContext, this.convert(executionContext, key), oldkey, ApiKey.AuditEvent.APIKEY_EXPIRED, key.getUpdatedAt());
        if (expirationDate != null) {
            NotificationParamsBuilder paramsBuilder = new NotificationParamsBuilder();
            if (key.getExpireAt() != null && now.before(key.getExpireAt())) {
                paramsBuilder.expirationDate(key.getExpireAt());
            }
            this.triggerNotifierService(executionContext, ApiHook.APIKEY_EXPIRED, key, paramsBuilder);
        }
    }

    private ApiKeyEntity convert(ExecutionContext executionContext, ApiKey apiKey) {
        ApiKeyEntity apiKeyEntity = new ApiKeyEntity();
        apiKeyEntity.setId(apiKey.getId());
        apiKeyEntity.setKey(apiKey.getKey());
        apiKeyEntity.setCreatedAt(apiKey.getCreatedAt());
        apiKeyEntity.setExpireAt(apiKey.getExpireAt());
        apiKeyEntity.setExpired(apiKey.getExpireAt() != null && new Date().after(apiKey.getExpireAt()));
        apiKeyEntity.setRevoked(apiKey.isRevoked());
        apiKeyEntity.setRevokedAt(apiKey.getRevokedAt());
        apiKeyEntity.setUpdatedAt(apiKey.getUpdatedAt());
        apiKeyEntity.setEnvironmentId(apiKey.getEnvironmentId());
        apiKeyEntity.setSubscriptions(this.subscriptionService.findByIdIn(apiKey.getSubscriptions()));
        apiKeyEntity.setApplication(this.applicationService.findById(executionContext, apiKey.getApplication()));
        apiKeyEntity.setDaysToExpirationOnLastNotification(apiKey.getDaysToExpirationOnLastNotification());
        return apiKeyEntity;
    }

    private ApiKeyCriteria toApiKeyCriteria(ApiKeyQuery query) {
        ApiKeyCriteria.ApiKeyCriteriaBuilder apiKeyCriteriaBuilder = ApiKeyCriteria.builder().includeRevoked(query.isIncludeRevoked()).includeFederated(query.isIncludeFederated()).from(query.getFrom()).to(query.getTo()).expireAfter(query.getExpireAfter()).expireBefore(query.getExpireBefore());
        if (query.getSubscriptions() != null) {
            apiKeyCriteriaBuilder.subscriptions(query.getSubscriptions());
        }
        if (query.getEnvironmentIds() != null) {
            apiKeyCriteriaBuilder.environments(query.getEnvironmentIds());
        }
        return apiKeyCriteriaBuilder.build();
    }

    private void createAuditLog(ExecutionContext executionContext, ApiKeyEntity key, ApiKey previousApiKey, ApiKey.AuditEvent event, Date eventDate) {
        key.getSubscriptions().forEach(subscription -> {
            LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
            properties.put(Audit.AuditProperties.API_KEY, key.getKey());
            properties.put(Audit.AuditProperties.API, subscription.getApi());
            properties.put(Audit.AuditProperties.APPLICATION, key.getApplication().getId());
            this.auditService.createApiAuditLog(executionContext, subscription.getApi(), properties, (Audit.AuditEvent)event, eventDate, previousApiKey, key);
        });
    }

    private void triggerNotifierService(ExecutionContext executionContext, ApiHook apiHook, ApiKey key, NotificationParamsBuilder paramsBuilder) {
        ApplicationEntity application = this.applicationService.findById(executionContext, key.getApplication());
        Set<SubscriptionEntity> subscriptions = this.subscriptionService.findByIdIn(key.getSubscriptions());
        this.triggerNotifierService(executionContext, apiHook, key, application, subscriptions, paramsBuilder);
    }

    private void triggerNotifierService(ExecutionContext executionContext, ApiHook apiHook, ApiKey key, ApplicationEntity application, Set<SubscriptionEntity> subscriptions) {
        this.triggerNotifierService(executionContext, apiHook, key, application, subscriptions, new NotificationParamsBuilder());
    }

    private void triggerNotifierService(ExecutionContext executionContext, ApiHook apiHook, ApiKey key, ApplicationEntity application, Set<SubscriptionEntity> subscriptions, NotificationParamsBuilder paramsBuilder) {
        subscriptions.forEach(subscription -> {
            GenericPlanEntity genericPlanEntity = this.planSearchService.findById(executionContext, subscription.getPlan());
            GenericApiModel genericApiModel = this.apiTemplateService.findByIdForTemplates(executionContext, subscription.getApi());
            PrimaryOwnerEntity owner = application.getPrimaryOwner();
            Map<String, Object> params = paramsBuilder.application(application).plan(genericPlanEntity).api(genericApiModel).owner(owner).apikey(key).build();
            this.notifierService.trigger(executionContext, apiHook, genericApiModel.getId(), params);
        });
    }
}

