/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.rest.api.service.impl;

import io.gravitee.repository.exceptions.TechnicalException;
import io.gravitee.repository.management.api.ApiKeyRepository;
import io.gravitee.repository.management.api.search.ApiKeyCriteria;
import io.gravitee.repository.management.model.ApiKey;
import io.gravitee.repository.management.model.Audit;
import io.gravitee.rest.api.model.ApiKeyEntity;
import io.gravitee.rest.api.model.ApiModelEntity;
import io.gravitee.rest.api.model.ApplicationEntity;
import io.gravitee.rest.api.model.PlanEntity;
import io.gravitee.rest.api.model.PrimaryOwnerEntity;
import io.gravitee.rest.api.model.SubscriptionEntity;
import io.gravitee.rest.api.model.SubscriptionStatus;
import io.gravitee.rest.api.model.key.ApiKeyQuery;
import io.gravitee.rest.api.service.ApiKeyGenerator;
import io.gravitee.rest.api.service.ApiKeyService;
import io.gravitee.rest.api.service.ApiService;
import io.gravitee.rest.api.service.ApplicationService;
import io.gravitee.rest.api.service.AuditService;
import io.gravitee.rest.api.service.NotifierService;
import io.gravitee.rest.api.service.PlanService;
import io.gravitee.rest.api.service.SubscriptionService;
import io.gravitee.rest.api.service.common.GraviteeContext;
import io.gravitee.rest.api.service.common.UuidString;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyActivatedException;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyExistingException;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyExpiredException;
import io.gravitee.rest.api.service.exceptions.ApiKeyNotFoundException;
import io.gravitee.rest.api.service.exceptions.InvalidApplicationApiKeyModeException;
import io.gravitee.rest.api.service.exceptions.SubscriptionClosedException;
import io.gravitee.rest.api.service.exceptions.SubscriptionNotActiveException;
import io.gravitee.rest.api.service.exceptions.TechnicalManagementException;
import io.gravitee.rest.api.service.impl.TransactionalService;
import io.gravitee.rest.api.service.notification.ApiHook;
import io.gravitee.rest.api.service.notification.NotificationParamsBuilder;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class ApiKeyServiceImpl
extends TransactionalService
implements ApiKeyService {
    private final Logger LOGGER = LoggerFactory.getLogger(ApiKeyServiceImpl.class);
    @Autowired
    private ApiKeyRepository apiKeyRepository;
    @Autowired
    private SubscriptionService subscriptionService;
    @Autowired
    private ApiKeyGenerator apiKeyGenerator;
    @Autowired
    private ApplicationService applicationService;
    @Autowired
    private ApiService apiService;
    @Autowired
    private PlanService planService;
    @Autowired
    private AuditService auditService;
    @Autowired
    private NotifierService notifierService;

    @Override
    public ApiKeyEntity generate(ApplicationEntity application, SubscriptionEntity subscription, String customApiKey) {
        if (!application.hasApiKeySharedMode()) {
            return this.generate(subscription, customApiKey);
        }
        return this.findOrGenerate(application, subscription, customApiKey);
    }

    @Override
    public ApiKeyEntity renew(SubscriptionEntity subscription) {
        return this.renew(subscription, null);
    }

    @Override
    public ApiKeyEntity renew(SubscriptionEntity subscription, String customApiKey) {
        try {
            this.LOGGER.debug("Renew API Key for subscription {}", (Object)subscription.getId());
            ApiKey newApiKey = this.generateForSubscription(subscription, customApiKey);
            newApiKey = this.apiKeyRepository.create(newApiKey);
            this.expireApiKeys(this.apiKeyRepository.findBySubscription(subscription.getId()), newApiKey);
            ApiKeyEntity newApiKeyEntity = this.convert(newApiKey);
            this.createAuditLog(newApiKeyEntity, null, ApiKey.AuditEvent.APIKEY_RENEWED, newApiKey.getCreatedAt());
            this.triggerNotifierService(ApiHook.APIKEY_RENEWED, newApiKey, newApiKeyEntity.getApplication(), newApiKeyEntity.getSubscriptions());
            return newApiKeyEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to renew an API Key for {}", (Object)subscription.getId(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to renew an API Key for %s", subscription.getId()), ex);
        }
    }

    @Override
    public ApiKeyEntity renew(ApplicationEntity application) {
        if (!application.hasApiKeySharedMode()) {
            throw new InvalidApplicationApiKeyModeException("Can't renew an API key on application that doesn't use shared API key mode");
        }
        try {
            this.LOGGER.debug("Renew API Key for application {}", (Object)application.getId());
            ApiKey newApiKey = this.generateForApplication(application.getId());
            newApiKey = this.apiKeyRepository.create(newApiKey);
            List allApiKeys = this.apiKeyRepository.findByApplication(application.getId());
            this.expireApiKeys(allApiKeys, newApiKey);
            this.addSharedSubscriptions(allApiKeys, newApiKey);
            ApiKeyEntity newApiKeyEntity = this.convert(newApiKey);
            this.createAuditLog(newApiKeyEntity, null, ApiKey.AuditEvent.APIKEY_RENEWED, newApiKey.getCreatedAt());
            this.triggerNotifierService(ApiHook.APIKEY_RENEWED, newApiKey, newApiKeyEntity.getApplication(), newApiKeyEntity.getSubscriptions());
            return newApiKeyEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to renew an API Key for application {}", (Object)application.getId(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to renew an API Key for application %s", application.getId()), ex);
        }
    }

    private void expireApiKeys(Collection<ApiKey> apiKeys, ApiKey activeApiKey) throws TechnicalException {
        Instant expirationInst = activeApiKey.getCreatedAt().toInstant().plus(Duration.ofHours(2L));
        Date expirationDate = Date.from(expirationInst);
        for (ApiKey apiKey : apiKeys) {
            ApiKeyEntity apiKeyEntity = this.convert(apiKey);
            if (apiKey.equals((Object)activeApiKey) || apiKeyEntity.isExpired()) continue;
            this.setExpiration(expirationDate, apiKeyEntity, apiKey);
        }
    }

    private void addSharedSubscriptions(Collection<ApiKey> apiKeys, ApiKey activeApiKey) throws TechnicalException {
        HashSet subscriptions = new HashSet();
        apiKeys.forEach(apiKey -> subscriptions.addAll(apiKey.getSubscriptions()));
        activeApiKey.setSubscriptions(new ArrayList(subscriptions));
        activeApiKey.setUpdatedAt(new Date());
        this.apiKeyRepository.update(activeApiKey);
    }

    private ApiKeyEntity findOrGenerate(ApplicationEntity application, SubscriptionEntity subscription, String customApiKey) {
        return this.findByApplication(application.getId()).stream().peek(apiKey -> this.addSubscription((ApiKeyEntity)apiKey, subscription)).max(Comparator.comparing(ApiKeyEntity::isRevoked, Comparator.reverseOrder()).thenComparing(ApiKeyEntity::getExpireAt, Comparator.nullsLast(Comparator.naturalOrder()))).orElseGet(() -> this.generate(subscription, customApiKey));
    }

    private void addSubscription(ApiKeyEntity apiKeyEntity, SubscriptionEntity subscription) {
        try {
            ApiKey apiKey = (ApiKey)this.apiKeyRepository.findById(apiKeyEntity.getId()).orElseThrow(ApiKeyNotFoundException::new);
            ArrayList<String> subscriptions = new ArrayList<String>(apiKey.getSubscriptions());
            subscriptions.add(subscription.getId());
            apiKey.setSubscriptions(subscriptions);
            apiKey.setUpdatedAt(new Date());
            this.apiKeyRepository.update(apiKey);
        }
        catch (TechnicalException e) {
            this.LOGGER.error("An error occurred while trying to add subscription to API Key", (Throwable)e);
            throw new TechnicalManagementException("An error occurred while trying to a add subscription to API Key");
        }
    }

    private ApiKeyEntity generate(SubscriptionEntity subscription, String customApiKey) {
        try {
            this.LOGGER.debug("Generate an API Key for subscription {}", (Object)subscription);
            ApiKey apiKey = this.generateForSubscription(subscription, customApiKey);
            apiKey = this.apiKeyRepository.create(apiKey);
            ApiKeyEntity apiKeyEntity = this.convert(apiKey);
            this.createAuditLog(apiKeyEntity, null, ApiKey.AuditEvent.APIKEY_CREATED, apiKey.getCreatedAt());
            return apiKeyEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to generate an API Key for {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to generate an API Key for %s", subscription), ex);
        }
    }

    private ApiKey generateForSubscription(SubscriptionEntity subscription, String customApiKey) {
        if (StringUtils.isNotEmpty((CharSequence)customApiKey) && !this.canCreate(customApiKey, subscription)) {
            throw new ApiKeyAlreadyExistingException();
        }
        Date now = new Date();
        if (subscription.getEndingAt() != null && subscription.getEndingAt().before(now)) {
            throw new SubscriptionClosedException(subscription.getId());
        }
        ApiKey apiKey = this.generateForApplication(subscription.getApplication(), customApiKey);
        apiKey.setSubscriptions(List.of(subscription.getId()));
        apiKey.setExpireAt(subscription.getEndingAt());
        return apiKey;
    }

    private ApiKey generateForApplication(String application) {
        return this.generateForApplication(application, null);
    }

    private ApiKey generateForApplication(String application, String customApiKey) {
        ApiKey apiKey = new ApiKey();
        apiKey.setId(UuidString.generateRandom());
        apiKey.setApplication(application);
        apiKey.setCreatedAt(new Date());
        apiKey.setUpdatedAt(apiKey.getCreatedAt());
        apiKey.setKey(StringUtils.isNotEmpty((CharSequence)customApiKey) ? customApiKey : this.apiKeyGenerator.generate());
        return apiKey;
    }

    @Override
    public void revoke(String keyId, boolean notify) {
        try {
            ApiKey key = (ApiKey)this.apiKeyRepository.findById(keyId).orElseThrow(ApiKeyNotFoundException::new);
            this.revoke(key, notify);
        }
        catch (TechnicalException e) {
            String message = String.format("An error occurs while trying to revoke a key with id %s", keyId);
            this.LOGGER.error(message, (Throwable)e);
            throw new TechnicalManagementException(message, e);
        }
    }

    @Override
    public void revoke(ApiKeyEntity apiKeyEntity, boolean notify) {
        this.revoke(apiKeyEntity.getId(), notify);
    }

    private void revoke(ApiKey key, boolean notify) throws TechnicalException {
        this.LOGGER.debug("Revoke API Key with id {}", (Object)key.getId());
        this.checkApiKeyExpired(key);
        ApiKey previousApiKey = new ApiKey(key);
        key.setRevoked(true);
        key.setUpdatedAt(new Date());
        key.setRevokedAt(key.getUpdatedAt());
        this.apiKeyRepository.update(key);
        this.createAuditLog(this.convert(previousApiKey), previousApiKey, ApiKey.AuditEvent.APIKEY_REVOKED, key.getUpdatedAt());
        if (notify) {
            this.triggerNotifierService(ApiHook.APIKEY_REVOKED, key, new NotificationParamsBuilder());
        }
    }

    @Override
    public ApiKeyEntity reactivate(ApiKeyEntity apiKeyEntity) {
        try {
            ApiKey key = (ApiKey)this.apiKeyRepository.findById(apiKeyEntity.getId()).orElseThrow(ApiKeyNotFoundException::new);
            this.LOGGER.debug("Reactivate API Key id {}", (Object)apiKeyEntity.getId());
            if (!key.isRevoked() && !this.convert(key).isExpired()) {
                throw new ApiKeyAlreadyActivatedException();
            }
            ApiKey previousApiKey = new ApiKey(key);
            key.setRevoked(false);
            key.setUpdatedAt(new Date());
            key.setRevokedAt(null);
            if (!apiKeyEntity.getApplication().hasApiKeySharedMode()) {
                SubscriptionEntity subscription = this.subscriptionService.findById((String)key.getSubscriptions().get(0));
                if (subscription.getStatus() != SubscriptionStatus.PAUSED && subscription.getStatus() != SubscriptionStatus.ACCEPTED) {
                    throw new SubscriptionNotActiveException(subscription);
                }
                key.setExpireAt(subscription.getEndingAt());
            }
            ApiKey updated = this.apiKeyRepository.update(key);
            ApiKeyEntity updatedEntity = this.convert(updated);
            this.createAuditLog(updatedEntity, previousApiKey, ApiKey.AuditEvent.APIKEY_REACTIVATED, key.getUpdatedAt());
            return updatedEntity;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to reactivate an api key", (Throwable)ex);
            throw new TechnicalManagementException("An error occurs while trying to reactivate an api key", ex);
        }
    }

    private void checkApiKeyExpired(ApiKey key) {
        if (key.isRevoked() || this.convert(key).isExpired()) {
            throw new ApiKeyAlreadyExpiredException();
        }
    }

    @Override
    public ApiKeyEntity findById(String keyId) {
        try {
            return this.apiKeyRepository.findById(keyId).map(this::convert).orElseThrow(ApiKeyNotFoundException::new);
        }
        catch (TechnicalException e) {
            String message = String.format("An error occurs while trying to find a key with id %s", keyId);
            this.LOGGER.error(message, (Throwable)e);
            throw new TechnicalManagementException(message, e);
        }
    }

    @Override
    public List<ApiKeyEntity> findByKey(String apiKey) {
        try {
            this.LOGGER.debug("Find API Keys by key");
            return this.apiKeyRepository.findByKey(apiKey).stream().map(this::convert).collect(Collectors.toList());
        }
        catch (TechnicalException e) {
            this.LOGGER.error("An error occurs while finding API keys", (Throwable)e);
            throw new TechnicalManagementException("An error occurs while finding API keys", e);
        }
    }

    @Override
    public List<ApiKeyEntity> findBySubscription(String subscription) {
        try {
            this.LOGGER.debug("Find API Keys for subscription {}", (Object)subscription);
            SubscriptionEntity subscriptionEntity = this.subscriptionService.findById(subscription);
            Set keys = this.apiKeyRepository.findBySubscription(subscriptionEntity.getId());
            return keys.stream().map(this::convert).sorted((o1, o2) -> o2.getCreatedAt().compareTo(o1.getCreatedAt())).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while finding API keys for subscription {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while finding API keys for subscription %s", subscription), ex);
        }
    }

    @Override
    public ApiKeyEntity findByKeyAndApi(String apiKey, String apiId) {
        try {
            this.LOGGER.debug("Find an API Key by key for API {}", (Object)apiId);
            ApiKey key = (ApiKey)this.apiKeyRepository.findByKeyAndApi(apiKey, apiId).orElseThrow(ApiKeyNotFoundException::new);
            return this.convert(key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to find an API Key for API {}", (Object)apiId, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to find an API Key for API %s", apiId), ex);
        }
    }

    @Override
    public List<ApiKeyEntity> findByApplication(String applicationId) {
        try {
            return this.apiKeyRepository.findByApplication(applicationId).stream().map(this::convert).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to find API Keys for application {}", (Object)applicationId, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to find API Keys for application %s", applicationId), ex);
        }
    }

    @Override
    public ApiKeyEntity update(ApiKeyEntity apiKeyEntity) {
        try {
            this.LOGGER.debug("Update API Key with id {}", (Object)apiKeyEntity.getId());
            ApiKey key = (ApiKey)this.apiKeyRepository.findById(apiKeyEntity.getId()).orElseThrow(ApiKeyNotFoundException::new);
            this.checkApiKeyExpired(key);
            key.setSubscriptions(apiKeyEntity.getSubscriptionIds());
            key.setPaused(apiKeyEntity.isPaused());
            if (apiKeyEntity.getExpireAt() != null) {
                this.setExpiration(apiKeyEntity.getExpireAt(), apiKeyEntity, key);
            } else {
                key.setUpdatedAt(new Date());
                this.apiKeyRepository.update(key);
            }
            return this.convert(key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while updating an API Key with id {}", (Object)apiKeyEntity.getId(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while updating an API Key with id %s", apiKeyEntity.getId()), ex);
        }
    }

    @Override
    public ApiKeyEntity updateDaysToExpirationOnLastNotification(ApiKeyEntity apiKeyEntity, Integer value) {
        try {
            return this.apiKeyRepository.findById(apiKeyEntity.getId()).map(dbApiKey -> {
                dbApiKey.setDaysToExpirationOnLastNotification(value);
                try {
                    return this.apiKeyRepository.update(dbApiKey);
                }
                catch (TechnicalException ex) {
                    this.LOGGER.error("An error occurs while trying to update ApiKey with id {}", (Object)dbApiKey.getId(), (Object)ex);
                    throw new TechnicalManagementException(String.format("An error occurs while trying to update ApiKey with id %s", dbApiKey.getId()), ex);
                }
            }).map(this::convert).orElseThrow(ApiKeyNotFoundException::new);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to update apiKey", (Throwable)ex);
            throw new TechnicalManagementException("An error occurs while trying to update apiKey", ex);
        }
    }

    @Override
    public boolean canCreate(String apiKeyValue, SubscriptionEntity subscription) {
        return this.canCreate(apiKeyValue, subscription.getApi(), subscription.getApplication());
    }

    @Override
    public boolean canCreate(String apiKey, String apiId, String applicationId) {
        this.LOGGER.debug("Check if an API Key can be created for api {} and application {}", (Object)apiId, (Object)applicationId);
        return this.findByKey(apiKey).stream().noneMatch(existingKey -> this.isConflictingKey((ApiKeyEntity)existingKey, apiId, applicationId));
    }

    private boolean isConflictingKey(ApiKeyEntity existingKey, String apiId, String applicationId) {
        if (!existingKey.getApplication().getId().equals(applicationId)) {
            return true;
        }
        return existingKey.getSubscriptions().stream().map(SubscriptionEntity::getApi).anyMatch(apiId::equals);
    }

    @Override
    public Collection<ApiKeyEntity> search(ApiKeyQuery query) {
        try {
            this.LOGGER.debug("Search api keys {}", (Object)query);
            ApiKeyCriteria.Builder builder = this.toApiKeyCriteriaBuilder(query);
            return this.apiKeyRepository.findByCriteria(builder.build()).stream().map(this::convert).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to search api keys: {}", (Object)query, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to search api keys: {}", query), ex);
        }
    }

    @Override
    public void delete(String apiKey) {
    }

    private void setExpiration(Date expirationDate, ApiKeyEntity apiKeyEntity, ApiKey key) throws TechnicalException {
        Date now = new Date();
        if (now.after(expirationDate)) {
            expirationDate = now;
        }
        key.setUpdatedAt(now);
        if (!key.isRevoked()) {
            SubscriptionEntity subscription;
            if (!(apiKeyEntity.getApplication() == null || apiKeyEntity.getApplication().hasApiKeySharedMode() || key.getSubscriptions().isEmpty() || (subscription = this.subscriptionService.findById((String)key.getSubscriptions().get(0))).getEndingAt() == null || expirationDate != null && subscription.getEndingAt().compareTo(expirationDate) >= 0)) {
                expirationDate = subscription.getEndingAt();
            }
            ApiKey oldkey = new ApiKey(key);
            key.setExpireAt(expirationDate);
            key.setDaysToExpirationOnLastNotification(null);
            this.apiKeyRepository.update(key);
            NotificationParamsBuilder paramsBuilder = new NotificationParamsBuilder();
            if (key.getExpireAt() != null && now.before(key.getExpireAt())) {
                paramsBuilder.expirationDate(key.getExpireAt());
            }
            this.triggerNotifierService(ApiHook.APIKEY_EXPIRED, key, paramsBuilder);
            this.createAuditLog(this.convert(key), oldkey, ApiKey.AuditEvent.APIKEY_EXPIRED, key.getUpdatedAt());
        } else {
            this.apiKeyRepository.update(key);
        }
    }

    private ApiKeyEntity convert(ApiKey apiKey) {
        ApiKeyEntity apiKeyEntity = new ApiKeyEntity();
        apiKeyEntity.setId(apiKey.getId());
        apiKeyEntity.setKey(apiKey.getKey());
        apiKeyEntity.setCreatedAt(apiKey.getCreatedAt());
        apiKeyEntity.setExpireAt(apiKey.getExpireAt());
        apiKeyEntity.setExpired(apiKey.getExpireAt() != null && new Date().after(apiKey.getExpireAt()));
        apiKeyEntity.setRevoked(apiKey.isRevoked());
        apiKeyEntity.setRevokedAt(apiKey.getRevokedAt());
        apiKeyEntity.setUpdatedAt(apiKey.getUpdatedAt());
        apiKeyEntity.setSubscriptions(this.subscriptionService.findByIdIn(apiKey.getSubscriptions()));
        apiKeyEntity.setApplication(this.applicationService.findById(GraviteeContext.getCurrentEnvironment(), apiKey.getApplication()));
        apiKeyEntity.setDaysToExpirationOnLastNotification(apiKey.getDaysToExpirationOnLastNotification());
        return apiKeyEntity;
    }

    private ApiKeyCriteria.Builder toApiKeyCriteriaBuilder(ApiKeyQuery query) {
        return new ApiKeyCriteria.Builder().includeRevoked(query.isIncludeRevoked()).plans(query.getPlans()).from(query.getFrom()).to(query.getTo()).expireAfter(query.getExpireAfter()).expireBefore(query.getExpireBefore());
    }

    private void createAuditLog(ApiKeyEntity key, ApiKey previousApiKey, ApiKey.AuditEvent event, Date eventDate) {
        key.getSubscriptions().forEach(subscription -> {
            LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
            properties.put(Audit.AuditProperties.API_KEY, key.getKey());
            properties.put(Audit.AuditProperties.API, subscription.getApi());
            properties.put(Audit.AuditProperties.APPLICATION, key.getApplication().getId());
            this.auditService.createApiAuditLog(subscription.getApi(), properties, (Audit.AuditEvent)event, eventDate, previousApiKey, key);
        });
    }

    private void triggerNotifierService(ApiHook apiHook, ApiKey key, NotificationParamsBuilder paramsBuilder) {
        ApplicationEntity application = this.applicationService.findById(GraviteeContext.getCurrentEnvironment(), key.getApplication());
        Set<SubscriptionEntity> subscriptions = this.subscriptionService.findByIdIn(key.getSubscriptions());
        this.triggerNotifierService(apiHook, key, application, subscriptions, paramsBuilder);
    }

    private void triggerNotifierService(ApiHook apiHook, ApiKey key, ApplicationEntity application, Set<SubscriptionEntity> subscriptions) {
        this.triggerNotifierService(apiHook, key, application, subscriptions, new NotificationParamsBuilder());
    }

    private void triggerNotifierService(ApiHook apiHook, ApiKey key, ApplicationEntity application, Set<SubscriptionEntity> subscriptions, NotificationParamsBuilder paramsBuilder) {
        subscriptions.forEach(subscription -> {
            PlanEntity plan = this.planService.findById(subscription.getPlan());
            ApiModelEntity api = this.apiService.findByIdForTemplates(subscription.getApi());
            PrimaryOwnerEntity owner = application.getPrimaryOwner();
            Map<String, Object> params = paramsBuilder.application(application).plan(plan).api(api).owner(owner).apikey(key).build();
            this.notifierService.trigger(apiHook, api.getId(), params);
        });
    }
}

