/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.rest.api.service.impl;

import io.gravitee.repository.exceptions.TechnicalException;
import io.gravitee.repository.management.api.ApiKeyRepository;
import io.gravitee.repository.management.api.search.ApiKeyCriteria;
import io.gravitee.repository.management.model.ApiKey;
import io.gravitee.repository.management.model.Audit;
import io.gravitee.rest.api.model.ApiKeyEntity;
import io.gravitee.rest.api.model.ApiModelEntity;
import io.gravitee.rest.api.model.ApplicationEntity;
import io.gravitee.rest.api.model.PlanEntity;
import io.gravitee.rest.api.model.PrimaryOwnerEntity;
import io.gravitee.rest.api.model.SubscriptionEntity;
import io.gravitee.rest.api.model.SubscriptionStatus;
import io.gravitee.rest.api.model.key.ApiKeyQuery;
import io.gravitee.rest.api.service.ApiKeyGenerator;
import io.gravitee.rest.api.service.ApiKeyService;
import io.gravitee.rest.api.service.ApiService;
import io.gravitee.rest.api.service.ApplicationService;
import io.gravitee.rest.api.service.AuditService;
import io.gravitee.rest.api.service.NotifierService;
import io.gravitee.rest.api.service.PlanService;
import io.gravitee.rest.api.service.SubscriptionService;
import io.gravitee.rest.api.service.common.GraviteeContext;
import io.gravitee.rest.api.service.common.UuidString;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyActivatedException;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyExistingException;
import io.gravitee.rest.api.service.exceptions.ApiKeyAlreadyExpiredException;
import io.gravitee.rest.api.service.exceptions.ApiKeyNotFoundException;
import io.gravitee.rest.api.service.exceptions.SubscriptionClosedException;
import io.gravitee.rest.api.service.exceptions.SubscriptionNotActiveException;
import io.gravitee.rest.api.service.exceptions.TechnicalManagementException;
import io.gravitee.rest.api.service.impl.TransactionalService;
import io.gravitee.rest.api.service.notification.ApiHook;
import io.gravitee.rest.api.service.notification.NotificationParamsBuilder;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class ApiKeyServiceImpl
extends TransactionalService
implements ApiKeyService {
    private final Logger LOGGER = LoggerFactory.getLogger(ApiKeyServiceImpl.class);
    @Autowired
    private ApiKeyRepository apiKeyRepository;
    @Autowired
    private SubscriptionService subscriptionService;
    @Autowired
    private ApiKeyGenerator apiKeyGenerator;
    @Autowired
    private ApplicationService applicationService;
    @Autowired
    private ApiService apiService;
    @Autowired
    private PlanService planService;
    @Autowired
    private AuditService auditService;
    @Autowired
    private NotifierService notifierService;

    @Override
    public ApiKeyEntity generate(String subscription) {
        return this.generate(subscription, null);
    }

    @Override
    public ApiKeyEntity generate(String subscription, String customApiKey) {
        try {
            this.LOGGER.debug("Generate an API Key for subscription {}", (Object)subscription);
            ApiKey apiKey = this.generateForSubscription(subscription, customApiKey);
            apiKey = this.apiKeyRepository.create(apiKey);
            this.createAuditLog(apiKey, null, ApiKey.AuditEvent.APIKEY_CREATED, apiKey.getCreatedAt());
            return ApiKeyServiceImpl.convert(apiKey);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to generate an API Key for {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to generate an API Key for %s", subscription), ex);
        }
    }

    @Override
    public ApiKeyEntity renew(String subscription) {
        return this.renew(subscription, null);
    }

    @Override
    public ApiKeyEntity renew(String subscription, String customApiKey) {
        try {
            this.LOGGER.debug("Renew API Key for subscription {}", (Object)subscription);
            ApiKey newApiKey = this.generateForSubscription(subscription, customApiKey);
            newApiKey = this.apiKeyRepository.create(newApiKey);
            Instant expirationInst = newApiKey.getCreatedAt().toInstant().plus(Duration.ofHours(2L));
            Date expirationDate = Date.from(expirationInst);
            Set oldKeys = this.apiKeyRepository.findBySubscription(subscription);
            for (ApiKey oldKey : oldKeys) {
                if (oldKey.equals((Object)newApiKey) || ApiKeyServiceImpl.convert(oldKey).isExpired()) continue;
                this.setExpiration(expirationDate, oldKey);
            }
            this.createAuditLog(newApiKey, null, ApiKey.AuditEvent.APIKEY_RENEWED, newApiKey.getCreatedAt());
            this.triggerNotifierService(ApiHook.APIKEY_RENEWED, newApiKey);
            return ApiKeyServiceImpl.convert(newApiKey);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to renew an API Key for {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to renew an API Key for %s", subscription), ex);
        }
    }

    private ApiKey generateForSubscription(String subscription) {
        return this.generateForSubscription(subscription, null);
    }

    private ApiKey generateForSubscription(String subscription, String customApiKey) {
        SubscriptionEntity subscriptionEntity = this.subscriptionService.findById(subscription);
        if (customApiKey != null && !this.canCreate(customApiKey, subscriptionEntity.getApi(), subscriptionEntity.getApplication())) {
            throw new ApiKeyAlreadyExistingException();
        }
        Date now = new Date();
        if (subscriptionEntity.getEndingAt() != null && subscriptionEntity.getEndingAt().before(now)) {
            throw new SubscriptionClosedException(subscription);
        }
        ApiKey apiKey = new ApiKey();
        apiKey.setId(UuidString.generateRandom());
        apiKey.setSubscription(subscription);
        apiKey.setApplication(subscriptionEntity.getApplication());
        apiKey.setPlan(subscriptionEntity.getPlan());
        apiKey.setCreatedAt(new Date());
        apiKey.setUpdatedAt(apiKey.getCreatedAt());
        apiKey.setKey(customApiKey != null ? customApiKey : this.apiKeyGenerator.generate());
        apiKey.setApi(subscriptionEntity.getApi());
        apiKey.setExpireAt(subscriptionEntity.getEndingAt());
        return apiKey;
    }

    @Override
    public void revoke(String keyId, boolean notify) {
        try {
            ApiKey key = (ApiKey)this.apiKeyRepository.findById(keyId).orElseThrow(() -> new ApiKeyNotFoundException());
            this.revoke(key, notify);
        }
        catch (TechnicalException e) {
            String message = String.format("An error occurs while trying to revoke a key with id %s", keyId);
            this.LOGGER.error(message, (Throwable)e);
            throw new TechnicalManagementException(message, e);
        }
    }

    @Override
    public void revoke(ApiKeyEntity apiKeyEntity, boolean notify) {
        this.revoke(apiKeyEntity.getKey(), apiKeyEntity.getApi(), notify);
    }

    @Override
    public void revoke(String apiKey, String apiId, boolean notify) {
        try {
            ApiKey key = (ApiKey)this.apiKeyRepository.findByKeyAndApi(apiKey, apiId).orElseThrow(() -> new ApiKeyNotFoundException());
            this.revoke(key, notify);
        }
        catch (TechnicalException e) {
            this.LOGGER.error("An error occurs while trying to revoke an api key", (Throwable)e);
            throw new TechnicalManagementException("An error occurs while trying to revoke an api key", e);
        }
    }

    private void revoke(ApiKey key, boolean notify) throws TechnicalException {
        this.LOGGER.debug("Revoke API Key with id {}", (Object)key.getId());
        this.checkApiKeyExpired(key);
        ApiKey previousApiKey = new ApiKey(key);
        key.setRevoked(true);
        key.setUpdatedAt(new Date());
        key.setRevokedAt(key.getUpdatedAt());
        this.apiKeyRepository.update(key);
        this.createAuditLog(key, previousApiKey, ApiKey.AuditEvent.APIKEY_REVOKED, key.getUpdatedAt());
        if (notify) {
            this.triggerNotifierService(ApiHook.APIKEY_REVOKED, key);
        }
    }

    @Override
    public ApiKeyEntity reactivate(ApiKeyEntity apiKeyEntity) {
        return this.reactivate(apiKeyEntity.getKey(), apiKeyEntity.getApi());
    }

    @Override
    public ApiKeyEntity reactivate(String apiKey, String apiId) {
        try {
            ApiKey key = (ApiKey)this.apiKeyRepository.findByKeyAndApi(apiKey, apiId).orElseThrow(() -> new ApiKeyNotFoundException());
            this.LOGGER.debug("Reactivate API Key id {} for API {}", (Object)key.getId(), (Object)apiId);
            if (!key.isRevoked() && !ApiKeyServiceImpl.convert(key).isExpired()) {
                throw new ApiKeyAlreadyActivatedException();
            }
            SubscriptionEntity subscription = this.subscriptionService.findById(key.getSubscription());
            if (subscription.getStatus() != SubscriptionStatus.PAUSED && subscription.getStatus() != SubscriptionStatus.ACCEPTED) {
                throw new SubscriptionNotActiveException(subscription);
            }
            ApiKey previousApiKey = new ApiKey(key);
            key.setRevoked(false);
            key.setUpdatedAt(new Date());
            key.setRevokedAt(null);
            key.setExpireAt(subscription.getEndingAt());
            ApiKey updated = this.apiKeyRepository.update(key);
            this.createAuditLog(key, previousApiKey, ApiKey.AuditEvent.APIKEY_REACTIVATED, key.getUpdatedAt());
            return ApiKeyServiceImpl.convert(updated);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to reactivate an api key", (Throwable)ex);
            throw new TechnicalManagementException("An error occurs while trying to reactivate an api key", ex);
        }
    }

    private void checkApiKeyExpired(ApiKey key) {
        if (key.isRevoked() || ApiKeyServiceImpl.convert(key).isExpired()) {
            throw new ApiKeyAlreadyExpiredException();
        }
    }

    @Override
    public ApiKeyEntity findById(String keyId) {
        try {
            return this.apiKeyRepository.findById(keyId).map(ApiKeyServiceImpl::convert).orElseThrow(() -> new ApiKeyNotFoundException());
        }
        catch (TechnicalException e) {
            String message = String.format("An error occurs while trying to find a key with id %s", keyId);
            this.LOGGER.error(message, (Throwable)e);
            throw new TechnicalManagementException(message, e);
        }
    }

    @Override
    public List<ApiKeyEntity> findByKey(String apiKey) {
        try {
            this.LOGGER.debug("Find API Keys by key");
            return this.apiKeyRepository.findByKey(apiKey).stream().map(ApiKeyServiceImpl::convert).collect(Collectors.toList());
        }
        catch (TechnicalException e) {
            this.LOGGER.error("An error occurs while finding API keys", (Throwable)e);
            throw new TechnicalManagementException("An error occurs while finding API keys", e);
        }
    }

    @Override
    public List<ApiKeyEntity> findBySubscription(String subscription) {
        try {
            this.LOGGER.debug("Find API Keys for subscription {}", (Object)subscription);
            SubscriptionEntity subscriptionEntity = this.subscriptionService.findById(subscription);
            Set keys = this.apiKeyRepository.findBySubscription(subscriptionEntity.getId());
            return keys.stream().map(ApiKeyServiceImpl::convert).sorted((o1, o2) -> o2.getCreatedAt().compareTo(o1.getCreatedAt())).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while finding API keys for subscription {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while finding API keys for subscription %s", subscription), ex);
        }
    }

    @Override
    public ApiKeyEntity findByKeyAndApi(String apiKey, String apiId) {
        try {
            this.LOGGER.debug("Find an API Key by key for API {}", (Object)apiId);
            ApiKey key = (ApiKey)this.apiKeyRepository.findByKeyAndApi(apiKey, apiId).orElseThrow(() -> new ApiKeyNotFoundException());
            return ApiKeyServiceImpl.convert(key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to find an API Key for API {}", (Object)apiId, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to find an API Key for API %s", apiId), ex);
        }
    }

    @Override
    public ApiKeyEntity update(ApiKeyEntity apiKeyEntity) {
        try {
            this.LOGGER.debug("Update API Key with id {}", (Object)apiKeyEntity.getId());
            ApiKey key = (ApiKey)this.apiKeyRepository.findByKeyAndApi(apiKeyEntity.getKey(), apiKeyEntity.getApi()).orElseThrow(() -> new ApiKeyNotFoundException());
            this.checkApiKeyExpired(key);
            key.setPaused(apiKeyEntity.isPaused());
            key.setPlan(apiKeyEntity.getPlan());
            if (apiKeyEntity.getExpireAt() != null) {
                this.setExpiration(apiKeyEntity.getExpireAt(), key);
            } else {
                key.setUpdatedAt(new Date());
                this.apiKeyRepository.update(key);
            }
            return ApiKeyServiceImpl.convert(key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while updating an API Key with id {}", (Object)apiKeyEntity.getId(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while updating an API Key with id %s", apiKeyEntity.getId()), ex);
        }
    }

    @Override
    public ApiKeyEntity updateDaysToExpirationOnLastNotification(ApiKeyEntity apiKeyEntity, Integer value) {
        return this.updateDaysToExpirationOnLastNotification(apiKeyEntity.getKey(), apiKeyEntity.getApi(), value);
    }

    @Override
    public ApiKeyEntity updateDaysToExpirationOnLastNotification(String apiKey, String apiId, Integer value) {
        try {
            return this.apiKeyRepository.findByKeyAndApi(apiKey, apiId).map(dbApiKey -> {
                dbApiKey.setDaysToExpirationOnLastNotification(value);
                try {
                    return this.apiKeyRepository.update(dbApiKey);
                }
                catch (TechnicalException ex) {
                    this.LOGGER.error("An error occurs while trying to update ApiKey with id {}", (Object)dbApiKey.getId(), (Object)ex);
                    throw new TechnicalManagementException(String.format("An error occurs while trying to update ApiKey with id %s", dbApiKey.getId()), ex);
                }
            }).map(ApiKeyServiceImpl::convert).orElseThrow(ApiKeyNotFoundException::new);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to update apiKey", (Throwable)ex);
            throw new TechnicalManagementException("An error occurs while trying to update apiKey", ex);
        }
    }

    @Override
    public boolean canCreate(String apiKey, String apiId, String applicationId) {
        this.LOGGER.debug("Check if an API Key can be created for api {} and application {}", (Object)apiId, (Object)applicationId);
        try {
            return this.apiKeyRepository.findByKey(apiKey).stream().noneMatch(existingKey -> !existingKey.getApplication().equals(applicationId) || existingKey.getApplication().equals(applicationId) && existingKey.getApi().equals(apiId));
        }
        catch (TechnicalException ex) {
            String message = String.format("An error occurs while checking if API Key can be created for api %s and application %s", apiId, applicationId);
            this.LOGGER.error(message, (Throwable)ex);
            throw new TechnicalManagementException(message, ex);
        }
    }

    @Override
    public Collection<ApiKeyEntity> search(ApiKeyQuery query) {
        try {
            this.LOGGER.debug("Search api keys {}", (Object)query);
            ApiKeyCriteria.Builder builder = this.toApiKeyCriteriaBuilder(query);
            return this.apiKeyRepository.findByCriteria(builder.build()).stream().map(ApiKeyServiceImpl::convert).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to search api keys: {}", (Object)query, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to search api keys: {}", query), ex);
        }
    }

    @Override
    public void delete(String apiKey) {
    }

    private void setExpiration(Date expirationDate, ApiKey key) throws TechnicalException {
        Date now = new Date();
        if (now.after(expirationDate)) {
            expirationDate = now;
        }
        key.setUpdatedAt(now);
        if (!key.isRevoked()) {
            SubscriptionEntity subscription = this.subscriptionService.findById(key.getSubscription());
            if (subscription.getEndingAt() != null && (expirationDate == null || subscription.getEndingAt().compareTo(expirationDate) < 0)) {
                expirationDate = subscription.getEndingAt();
            }
            ApiKey oldkey = new ApiKey(key);
            key.setExpireAt(expirationDate);
            key.setDaysToExpirationOnLastNotification(null);
            this.apiKeyRepository.update(key);
            NotificationParamsBuilder paramsBuilder = new NotificationParamsBuilder();
            if (key.getExpireAt() != null && now.before(key.getExpireAt())) {
                paramsBuilder.expirationDate(key.getExpireAt());
            }
            this.triggerNotifierService(ApiHook.APIKEY_EXPIRED, key, paramsBuilder);
            this.createAuditLog(key, oldkey, ApiKey.AuditEvent.APIKEY_EXPIRED, key.getUpdatedAt());
        } else {
            this.apiKeyRepository.update(key);
        }
    }

    private static ApiKeyEntity convert(ApiKey apiKey) {
        ApiKeyEntity apiKeyEntity = new ApiKeyEntity();
        apiKeyEntity.setId(apiKey.getId());
        apiKeyEntity.setKey(apiKey.getKey());
        apiKeyEntity.setCreatedAt(apiKey.getCreatedAt());
        apiKeyEntity.setExpireAt(apiKey.getExpireAt());
        apiKeyEntity.setExpired(apiKey.getExpireAt() != null && new Date().after(apiKey.getExpireAt()));
        apiKeyEntity.setRevoked(apiKey.isRevoked());
        apiKeyEntity.setRevokedAt(apiKey.getRevokedAt());
        apiKeyEntity.setUpdatedAt(apiKey.getUpdatedAt());
        apiKeyEntity.setSubscription(apiKey.getSubscription());
        apiKeyEntity.setApplication(apiKey.getApplication());
        apiKeyEntity.setPlan(apiKey.getPlan());
        apiKeyEntity.setDaysToExpirationOnLastNotification(apiKey.getDaysToExpirationOnLastNotification());
        apiKeyEntity.setApi(apiKey.getApi());
        return apiKeyEntity;
    }

    private ApiKeyCriteria.Builder toApiKeyCriteriaBuilder(ApiKeyQuery query) {
        return new ApiKeyCriteria.Builder().includeRevoked(query.isIncludeRevoked()).plans(query.getPlans()).from(query.getFrom()).to(query.getTo()).expireAfter(query.getExpireAfter()).expireBefore(query.getExpireBefore());
    }

    private void createAuditLog(ApiKey key, ApiKey previousApiKey, ApiKey.AuditEvent event, Date eventDate) {
        LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
        properties.put(Audit.AuditProperties.API_KEY, key.getKey());
        properties.put(Audit.AuditProperties.API, key.getApi());
        properties.put(Audit.AuditProperties.APPLICATION, key.getApplication());
        this.auditService.createApiAuditLog(key.getApi(), properties, (Audit.AuditEvent)event, eventDate, previousApiKey, key);
    }

    private void triggerNotifierService(ApiHook apiHook, ApiKey key) {
        this.triggerNotifierService(apiHook, key, new NotificationParamsBuilder());
    }

    private void triggerNotifierService(ApiHook apiHook, ApiKey key, NotificationParamsBuilder paramsBuilder) {
        PlanEntity plan = this.planService.findById(key.getPlan());
        ApplicationEntity application = this.applicationService.findById(GraviteeContext.getCurrentEnvironment(), key.getApplication());
        ApiModelEntity api = this.apiService.findByIdForTemplates(key.getApi());
        PrimaryOwnerEntity owner = application.getPrimaryOwner();
        Map<String, Object> params = paramsBuilder.application(application).plan(plan).api(api).owner(owner).apikey(key).build();
        this.notifierService.trigger(apiHook, api.getId(), params);
    }
}

