/*
 * Copyright (C) 2015 Square, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.github.xyzxqs.libs.xtask;


/**
 * The result of executing an Worker.
 */
public final class Result<T> {
    public static <T> Result<T> error(Throwable error) {
        if (error == null) throw new NullPointerException("error == null");
        return new Result<>(null, error);
    }

    public static <T> Result<T> result(T response) {
        if (response == null) throw new NullPointerException("result == null");
        return new Result<>(response, null);
    }

    private final T result;
    private final Throwable error;

    private Result(T result, Throwable error) {
        this.result = result;
        this.error = error;
    }

    /**
     * The result received from executing an HTTP request. Only present when {@link #isError()} is
     * false, null otherwise.
     */
    public T result() {
        return result;
    }

    /**
     * The error experienced while attempting to execute an HTTP request. Only present when {@link
     * #isError()} is true, null otherwise.
     */
    public Throwable error() {
        return error;
    }

    /**
     * {@code true} if the request resulted in an error. See {@link #error()} for the cause.
     */
    public boolean isError() {
        return error != null;
    }

    @Override
    public String toString() {
        return isError() ? error.toString() : result.toString();
    }
}
