/*
 * Copyright 2017 xyzxqs (xyzxqs@gmail.com)
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.xyzxqs.libs.xtask;

import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.concurrent.Executor;

import io.github.xyzxqs.libs.xtask.func.Action;
import io.github.xyzxqs.libs.xtask.func.BiConsumer;
import io.github.xyzxqs.libs.xtask.func.BiFunction;
import io.github.xyzxqs.libs.xtask.func.Consumer;
import io.github.xyzxqs.libs.xtask.func.Function;
import io.github.xyzxqs.libs.xtask.func.Supplier;

/**
 * @author xyzxqs (xyzxqs@gmail.com)
 */

public class Xtask<T> implements Cancellable {

    private Worker worker;
    private Result<T> taskResult;
    private boolean canceled;
    private final ArrayList<Consumer<Result<T>>> resultConsumers;
    private final ArrayList<Action> cancelActions;


    private Xtask() {
        resultConsumers = new ArrayList<>();
        cancelActions = new ArrayList<>();
        canceled = false;
    }

    public static <T> Xtask<T> supplyAsync(final @NonNull Supplier<T> supplier) {
        return supplyAsync(supplier, WorkerFactory.instance().getDefaultExecutor());
    }

    public static <T> Xtask<T> supplyAsync(final @NonNull Supplier<T> supplier,
                                           @NonNull Executor exec) {
        final Xtask<T> task = new Xtask<>();
        task.worker = WorkerFactory.instance().<T>createWorker()
                .supplyAsync(new Supplier<T>() {
                    @Override
                    public T get() throws Exception {
                        return supplier.get();
                    }
                })
                .onResult(new Consumer<Result<T>>() {
                    @Override
                    public void accept(Result<T> result) throws Exception {
                        task.result(result);
                    }
                });/*no ifCanceled(...), Xtask<T> handle this itself*/
        task.execute(exec);
        return task;
    }

    private void execute() {
        execute(WorkerFactory.instance().getDefaultExecutor());
    }

    private void execute(Executor exec) {
        canceled = false;
        worker.execute(exec);
    }

    private void result(@NonNull Result<T> result) {
        if (!canceled) {
            taskResult = result;
            for (Consumer<Result<T>> consumer : resultConsumers) {
                try {
                    consumer.accept(result);
                } catch (Exception e) {
                    //ignore
                }
            }
        }
    }

    @Override
    public void cancel() {
        canceled = true;
        if (worker != null) {
            worker.cancel();
        }
        for (Action action : cancelActions) {
            try {
                action.run();
            } catch (Exception e) {
                //ignore
            }
        }
    }

    public Xtask<T> onResult(@NonNull Consumer<Result<T>> consumer) {
        Objects.checkNotNull(consumer, "consumer == null");
        resultConsumers.add(consumer);
        if (taskResult != null) {
            try {
                consumer.accept(taskResult);
            } catch (Exception e) {
                //ignore
            }
        }
        return this;
    }

    public Xtask<T> ifCanceled(@NonNull Action action) {
        Objects.checkNotNull(action, "action == null");
        cancelActions.add(action);
        if (canceled) {
            try {
                action.run();
            } catch (Exception e) {
                //ignore
            }
        }
        return this;
    }

    public <U> Xtask<U> thenApplyAsync(final @NonNull Function<Result<T>, ? extends U> fn) {
        return thenApplyAsync(fn, WorkerFactory.instance().getDefaultExecutor());
    }

    public <U> Xtask<U> thenApplyAsync(final @NonNull Function<Result<T>, ? extends U> fn,
                                       final @NonNull Executor exec) {
        final Xtask<U> xtask = new Xtask<>();
        onResult(new Consumer<Result<T>>() {
            @Override
            public void accept(final Result<T> result) throws Exception {
                xtask.worker = WorkerFactory.instance().<U>createWorker()
                        .supplyAsync(new Supplier<U>() {
                            @Override
                            public U get() throws Exception {
                                return fn.apply(result);
                            }
                        })
                        .onResult(new Consumer<Result<U>>() {
                            @Override
                            public void accept(Result<U> uResult) throws Exception {
                                xtask.result(uResult);
                            }
                        });
                xtask.execute(exec);
            }
        }).ifCanceled(new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        });
        return xtask;
    }

    public Xtask<Void> thenAcceptAsync(final @NonNull Consumer<Result<T>> action) {
        return thenAcceptAsync(action, WorkerFactory.instance().getDefaultExecutor());
    }

    public Xtask<Void> thenAcceptAsync(final @NonNull Consumer<Result<T>> action,
                                       final @NonNull Executor exec) {
        final Xtask<Void> xtask = new Xtask<>();
        onResult(new Consumer<Result<T>>() {
            @Override
            public void accept(final Result<T> result) throws Exception {
                xtask.worker = WorkerFactory.instance().<Void>createWorker()
                        .supplyAsync(new Supplier<Void>() {
                            @Override
                            public Void get() throws Exception {
                                action.accept(result);
                                return null;
                            }
                        })
                        .onResult(new Consumer<Result<Void>>() {
                            @Override
                            public void accept(Result<Void> uResult) throws Exception {
                                xtask.result(uResult);
                            }
                        });
                xtask.execute(exec);
            }
        }).ifCanceled(new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        });
        return xtask;
    }

    public Xtask<Void> thenRunAsync(final @NonNull Action action) {
        return thenRunAsync(action, WorkerFactory.instance().getDefaultExecutor());
    }

    public Xtask<Void> thenRunAsync(final @NonNull Action action, final @NonNull Executor exec) {
        final Xtask<Void> xtask = new Xtask<>();
        onResult(new Consumer<Result<T>>() {
            @Override
            public void accept(final Result<T> result) throws Exception {
                xtask.worker = WorkerFactory.instance().<Void>createWorker()
                        .supplyAsync(new Supplier<Void>() {
                            @Override
                            public Void get() throws Exception {
                                action.run();
                                return null;
                            }
                        })
                        .onResult(new Consumer<Result<Void>>() {
                            @Override
                            public void accept(Result<Void> uResult) throws Exception {
                                xtask.result(uResult);
                            }
                        });
                xtask.execute(exec);
            }
        }).ifCanceled(new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        });
        return xtask;
    }

    public <U, V> Xtask<V> thenCombineAsync(@NonNull Xtask<U> other,
                                            final @NonNull BiFunction<Result<T>, Result<U>, ? extends V> fn) {
        return thenCombineAsync(other, fn, WorkerFactory.instance().getDefaultExecutor());
    }

    public <U, V> Xtask<V> thenCombineAsync(@NonNull Xtask<U> other,
                                            final @NonNull BiFunction<Result<T>, Result<U>, ? extends V> fn,
                                            final @NonNull Executor exec) {
        final Xtask<V> xtask = new Xtask<>();
        new TaskBothOperator<>(this, other, new TaskBothOperator.Callback<T, U>() {
            @Override
            public void onResult(final Result<T> result1, final Result<U> result2) {
                xtask.worker = WorkerFactory.instance().<V>createWorker()
                        .supplyAsync(new Supplier<V>() {
                            @Override
                            public V get() throws Exception {
                                return fn.apply(result1, result2);
                            }
                        }).onResult(new Consumer<Result<V>>() {
                            @Override
                            public void accept(Result<V> vResult) throws Exception {
                                xtask.result(vResult);
                            }
                        });
                xtask.execute(exec);
            }
        });

        Action cancelAction = new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        };
        ifCanceled(cancelAction);
        other.ifCanceled(cancelAction);

        return xtask;
    }

    public <U> Xtask<Void> thenAcceptBothAsync(@NonNull Xtask<U> other,
                                               final @NonNull BiConsumer<Result<T>, Result<U>> action) {
        return thenAcceptBothAsync(other, action, WorkerFactory.instance().getDefaultExecutor());
    }

    public <U> Xtask<Void> thenAcceptBothAsync(@NonNull Xtask<U> other,
                                               final @NonNull BiConsumer<Result<T>, Result<U>> action,
                                               final @NonNull Executor exec) {
        final Xtask<Void> xtask = new Xtask<>();
        new TaskBothOperator<>(this, other, new TaskBothOperator.Callback<T, U>() {
            @Override
            public void onResult(final Result<T> result1, final Result<U> result2) {
                xtask.worker = WorkerFactory.instance().<Void>createWorker()
                        .supplyAsync(new Supplier<Void>() {
                            @Override
                            public Void get() throws Exception {
                                action.accept(result1, result2);
                                return null;
                            }
                        })
                        .onResult(new Consumer<Result<Void>>() {
                            @Override
                            public void accept(Result<Void> voidResult) throws Exception {
                                xtask.result(voidResult);
                            }
                        });
                xtask.execute(exec);
            }
        });

        Action cancelAction = new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        };
        ifCanceled(cancelAction);
        other.ifCanceled(cancelAction);
        return xtask;
    }

    public <U> Xtask<Void> runAfterBothAsync(@NonNull Xtask<U> other, final @NonNull Action action) {
        return runAfterBothAsync(other, action, WorkerFactory.instance().getDefaultExecutor());
    }

    public <U> Xtask<Void> runAfterBothAsync(@NonNull Xtask<U> other,
                                             final @NonNull Action action,
                                             final @NonNull Executor exec) {
        final Xtask<Void> xtask = new Xtask<>();
        new TaskBothOperator<>(this, other, new TaskBothOperator.Callback<T, U>() {
            @Override
            public void onResult(Result<T> result1, Result<U> result2) {
                xtask.worker = WorkerFactory.instance().<Void>createWorker()
                        .supplyAsync(new Supplier<Void>() {
                            @Override
                            public Void get() throws Exception {
                                action.run();
                                return null;
                            }
                        })
                        .onResult(new Consumer<Result<Void>>() {
                            @Override
                            public void accept(Result<Void> voidResult) throws Exception {
                                xtask.result(voidResult);
                            }
                        });
                xtask.execute(exec);

            }
        });
        Action cancelAction = new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        };
        ifCanceled(cancelAction);
        other.ifCanceled(cancelAction);
        return xtask;
    }

    public <U> Xtask<U> applyToEitherAsync(@NonNull Xtask<T> other,
                                           final @NonNull Function<Result<T>, U> fn) {
        return applyToEitherAsync(other, fn, WorkerFactory.instance().getDefaultExecutor());
    }

    public <U> Xtask<U> applyToEitherAsync(@NonNull Xtask<T> other,
                                           final @NonNull Function<Result<T>, U> fn,
                                           final @NonNull Executor exec) {
        final Xtask<U> xtask = new Xtask<>();
        new TaskEitherOperator<>(this, other, new TaskEitherOperator.Callback<T>() {
            @Override
            public void onResult(final Result<T> result) {
                xtask.worker = WorkerFactory.instance().<U>createWorker()
                        .supplyAsync(new Supplier<U>() {
                            @Override
                            public U get() throws Exception {
                                return fn.apply(result);
                            }
                        })
                        .onResult(new Consumer<Result<U>>() {
                            @Override
                            public void accept(Result<U> uResult) throws Exception {
                                xtask.result(uResult);
                            }
                        });
                xtask.execute(exec);
            }
        });

        ifCanceled(new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        });

        /*no need to check other.ifCanceled*/
        return xtask;
    }

    public Xtask<Void> acceptEitherAsync(@NonNull Xtask<T> other, final @NonNull Consumer<Result<T>> action) {
        return acceptEitherAsync(other, action, WorkerFactory.instance().getDefaultExecutor());
    }

    public Xtask<Void> acceptEitherAsync(@NonNull Xtask<T> other,
                                         final @NonNull Consumer<Result<T>> action,
                                         final @NonNull Executor exec) {
        final Xtask<Void> xtask = new Xtask<>();
        new TaskEitherOperator<>(this, other, new TaskEitherOperator.Callback<T>() {
            @Override
            public void onResult(final Result<T> result) {
                xtask.worker = WorkerFactory.instance().<Void>createWorker()
                        .supplyAsync(new Supplier<Void>() {
                            @Override
                            public Void get() throws Exception {
                                action.accept(result);
                                return null;
                            }
                        })
                        .onResult(new Consumer<Result<Void>>() {
                            @Override
                            public void accept(Result<Void> voidResult) throws Exception {
                                xtask.result(voidResult);
                            }
                        });
                xtask.execute(exec);
            }
        });
        ifCanceled(new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        });

        /*no need to check other.ifCanceled*/
        return xtask;
    }

    public <U> Xtask<Void> runAfterEitherAsync(@NonNull Xtask<U> other, final @NonNull Action action) {
        return runAfterEitherAsync(other, action, WorkerFactory.instance().getDefaultExecutor());
    }

    public <U> Xtask<Void> runAfterEitherAsync(@NonNull Xtask<U> other,
                                               final @NonNull Action action,
                                               final @NonNull Executor exec) {
        final Xtask<Void> xtask = new Xtask<>();

        new TaskEitherOperator2<>(this, other, new TaskEitherOperator2.Callback() {
            @Override
            public void onResult() {
                xtask.worker = WorkerFactory.instance().<Void>createWorker()
                        .supplyAsync(new Supplier<Void>() {
                            @Override
                            public Void get() throws Exception {
                                action.run();
                                return null;
                            }
                        })
                        .onResult(new Consumer<Result<Void>>() {
                            @Override
                            public void accept(Result<Void> voidResult) throws Exception {
                                xtask.result(voidResult);
                            }
                        });
                xtask.execute(exec);
            }
        });

        ifCanceled(new Action() {
            @Override
            public void run() throws Exception {
                xtask.cancel();
            }
        });

        /*no need to check other.ifCanceled*/
        return xtask;
    }

    private static class TaskBothOperator<T1, T2> {
        public interface Callback<T1, T2> {
            void onResult(Result<T1> result1, Result<T2> result2);
        }

        private Result<T1> result1;
        private Result<T2> result2;

        private TaskBothOperator(Xtask<T1> t1Xtask, Xtask<T2> t2Xtask, final Callback<T1, T2> callback) {
            t1Xtask.onResult(new Consumer<Result<T1>>() {
                @Override
                public void accept(Result<T1> t1Result) throws Exception {
                    synchronized (TaskBothOperator.this) {
                        if (result2 != null) {
                            callback.onResult(t1Result, result2);
                        } else {
                            result1 = t1Result;
                        }
                    }
                }
            });

            t2Xtask.onResult(new Consumer<Result<T2>>() {
                @Override
                public void accept(Result<T2> t2Result) throws Exception {
                    synchronized (TaskBothOperator.this) {
                        if (result1 != null) {
                            callback.onResult(result1, t2Result);
                        } else {
                            result2 = t2Result;
                        }
                    }
                }
            });
        }
    }

    private static class TaskEitherOperator<T> {
        public interface Callback<T> {
            void onResult(Result<T> result);
        }

        private boolean finish = false;

        TaskEitherOperator(Xtask<T> xtask1, Xtask<T> xtask2, final Callback<T> callback) {
            xtask1.onResult(new Consumer<Result<T>>() {
                @Override
                public void accept(Result<T> result) throws Exception {
                    synchronized (TaskEitherOperator.this) {
                        if (!finish) {
                            callback.onResult(result);
                            finish = true;
                        }
                    }
                }
            });

            xtask2.onResult(new Consumer<Result<T>>() {
                @Override
                public void accept(Result<T> result) throws Exception {
                    synchronized (TaskEitherOperator.this) {
                        if (!finish) {
                            callback.onResult(result);
                            finish = true;
                        }
                    }
                }
            });
        }
    }

    private static class TaskEitherOperator2<T, U> {
        public interface Callback {
            void onResult();
        }

        private boolean finish = false;

        TaskEitherOperator2(Xtask<T> xtask1, Xtask<U> xtask2, final Callback callback) {
            xtask1.onResult(new Consumer<Result<T>>() {
                @Override
                public void accept(Result<T> result) throws Exception {
                    synchronized (TaskEitherOperator2.this) {
                        if (!finish) {
                            callback.onResult();
                            finish = true;
                        }
                    }
                }
            });

            xtask2.onResult(new Consumer<Result<U>>() {
                @Override
                public void accept(Result<U> result) throws Exception {
                    synchronized (TaskEitherOperator2.this) {
                        if (!finish) {
                            callback.onResult();
                            finish = true;
                        }
                    }
                }
            });
        }
    }
}
