package io.github.xyzxqs.libs.xtask.retrofit2;

import java.lang.annotation.Annotation;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

import io.github.xyzxqs.libs.xtask.func.Action;
import io.github.xyzxqs.libs.xtask.func.Supplier;
import io.github.xyzxqs.libs.xtask.Xtask;
import retrofit2.Call;
import retrofit2.CallAdapter;
import retrofit2.HttpException;
import retrofit2.Response;
import retrofit2.Retrofit;

/**
 * A {@linkplain CallAdapter.Factory call adapter} which creates Java 8 futures.
 * <p>
 * Adding this class to {@link Retrofit} allows you to return {@link io.github.xyzxqs.libs.xtask.Xtask} from
 * service methods.
 * <pre><code>
 * interface MyService {
 *   &#64;GET("user/me")
 *   Xtask&lt;User&gt; getUser()
 * }
 * </code></pre>
 * There are two configurations supported for the {@code CompletableFuture} type parameter:
 * <ul>
 * <li>Direct body (e.g., {@code CompletableFuture<User>}) returns the deserialized body for 2XX
 * responses, sets {@link retrofit2.HttpException HttpException} errors for non-2XX responses, and
 * sets {@link java.io.IOException} for network errors.</li>
 * <li>Response wrapped body (e.g., {@code CompletableFuture<Response<User>>}) returns a
 * {@link retrofit2.Response} object for all HTTP responses and sets {@link java.io.IOException} for network
 * errors</li>
 * </ul>
 *
 * @author xyzxqs (xyzxqs@gmail.com)
 */
public class XtaskCallAdapterFactory extends CallAdapter.Factory {

    public static XtaskCallAdapterFactory create() {
        return new XtaskCallAdapterFactory();
    }

    private XtaskCallAdapterFactory() {

    }

    @Override
    public CallAdapter<?, ?> get(Type returnType, Annotation[] annotations, Retrofit retrofit) {
        if (getRawType(returnType) != Xtask.class) {
            return null;
        }
        if (!(returnType instanceof ParameterizedType)) {
            throw new IllegalStateException("Xtask return type must be parameterized"
                    + " as Xtask<Foo> or Xtask<? extends Foo>");
        }
        Type innerType = getParameterUpperBound(0, (ParameterizedType) returnType);

        if (getRawType(innerType) != Response.class) {
            // Generic type is not Response<T>. Use it for body-only adapter.
            return new BodyCallAdapter<>(innerType);
        }

        // Generic type is Response<T>. Extract T and createWorker the Response version of the adapter.
        if (!(innerType instanceof ParameterizedType)) {
            throw new IllegalStateException("Response must be parameterized"
                    + " as Response<Foo> or Response<? extends Foo>");
        }
        Type responseType = getParameterUpperBound(0, (ParameterizedType) innerType);
        return new ResponseCallAdapter<>(responseType);
    }

    private static final class BodyCallAdapter<R> implements CallAdapter<R, Xtask<R>> {
        private final Type responseType;

        BodyCallAdapter(Type responseType) {
            this.responseType = responseType;
        }

        @Override
        public Type responseType() {
            return responseType;
        }

        @Override
        public Xtask<R> adapt(final Call<R> call) {
            return Xtask.supplyAsync(new Supplier<R>() {
                @Override
                public R get() throws Exception {
                    Response<R> response = call.execute();
                    if (response.isSuccessful()) {
                        return response.body();
                    } else {
                        throw new HttpException(response);
                    }
                }
            }).ifCanceled(new Action() {
                @Override
                public void run() throws Exception {
                    call.cancel();
                }
            });
        }
    }

    private static final class ResponseCallAdapter<R>
            implements CallAdapter<R, Xtask<Response<R>>> {
        private final Type responseType;

        ResponseCallAdapter(Type responseType) {
            this.responseType = responseType;
        }

        @Override
        public Type responseType() {
            return responseType;
        }

        @Override
        public Xtask<Response<R>> adapt(final Call<R> call) {
            return Xtask.supplyAsync(new Supplier<Response<R>>() {
                @Override
                public Response<R> get() throws Exception {
                    return call.execute();
                }
            }).ifCanceled(new Action() {
                @Override
                public void run() throws Exception {
                    call.cancel();
                }
            });
        }
    }
}
