package io.github.threetenjaxb.core;

import java.time.Year;
import jakarta.xml.bind.annotation.adapters.XmlAdapter;

/**
 * {@code XmlAdapter} mapping JSR-310 {@code Year} to ISO proleptic year number
 * <p>
 * Year number interpretation details:
 * <ul>
 * <li>{@link java.time.Year#of(int)}</li>
 * <li>{@link java.time.Year#getValue()}</li>
 * </ul>
 * <p>
 * Be aware that using this adapter will yield {@code null} when unmarshalling
 * {@code xsd:gYear} types. Use {@link YearAsTextXmlAdapter} instead.
 *
 * @see jakarta.xml.bind.annotation.adapters.XmlAdapter
 * @see java.time.Year
 */
public class YearXmlAdapter extends XmlAdapter<Integer, Year> {
    @Override
    public Year unmarshal(Integer isoYearInt) {
        return isoYearInt != null ? Year.of(isoYearInt) : null;
    }

    @Override
    public Integer marshal(Year year) {
        return year != null ? year.getValue() : null;
    }
}
