/*
 * Decompiled with CFR 0.152.
 */
package jadx.core.utils.files;

import jadx.core.utils.ListUtils;
import jadx.core.utils.exceptions.JadxRuntimeException;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final Logger LOG = LoggerFactory.getLogger(FileUtils.class);
    public static final int READ_BUFFER_SIZE = 8192;
    private static final int MAX_FILENAME_LENGTH = 128;
    public static final String JADX_TMP_INSTANCE_PREFIX = "jadx-instance-";
    public static final String JADX_TMP_PREFIX = "jadx-tmp-";
    private static Path tempRootDir = FileUtils.createTempRootDir();
    private static final Object MKDIR_SYNC = new Object();
    private static final byte[] HEX_ARRAY = "0123456789abcdef".getBytes(StandardCharsets.US_ASCII);
    private static final byte[] ZIP_FILE_MAGIC = new byte[]{80, 75, 3, 4};

    private FileUtils() {
    }

    public static synchronized Path updateTempRootDir(Path newTempRootDir) {
        try {
            Path dir;
            FileUtils.makeDirs(newTempRootDir);
            tempRootDir = dir = Files.createTempDirectory(newTempRootDir, JADX_TMP_INSTANCE_PREFIX, new FileAttribute[0]);
            dir.toFile().deleteOnExit();
            return dir;
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to update temp root directory", e);
        }
    }

    private static Path createTempRootDir() {
        try {
            Path dir = Files.createTempDirectory(JADX_TMP_INSTANCE_PREFIX, new FileAttribute[0]);
            dir.toFile().deleteOnExit();
            return dir;
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to create temp root directory", e);
        }
    }

    public static List<Path> expandDirs(List<Path> paths) {
        ArrayList<Path> files = new ArrayList<Path>(paths.size());
        for (Path path : paths) {
            if (Files.isDirectory(path, new LinkOption[0])) {
                FileUtils.expandDir(path, files);
                continue;
            }
            files.add(path);
        }
        return files;
    }

    private static void expandDir(Path dir, List<Path> files) {
        try (Stream<Path> walk = Files.walk(dir, FileVisitOption.FOLLOW_LINKS);){
            walk.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(files::add);
        }
        catch (Exception e) {
            LOG.error("Failed to list files in directory: {}", (Object)dir, (Object)e);
        }
    }

    public static void addFileToJar(JarOutputStream jar, File source, String entryName) throws IOException {
        try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(source));){
            JarEntry entry = new JarEntry(entryName);
            entry.setTime(source.lastModified());
            jar.putNextEntry(entry);
            FileUtils.copyStream(in, jar);
            jar.closeEntry();
        }
    }

    public static void makeDirsForFile(Path path) {
        if (path != null) {
            FileUtils.makeDirs(path.toAbsolutePath().getParent().toFile());
        }
    }

    public static void makeDirsForFile(File file) {
        if (file != null) {
            FileUtils.makeDirs(file.getParentFile());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void makeDirs(@Nullable File dir) {
        if (dir != null) {
            Object object = MKDIR_SYNC;
            synchronized (object) {
                if (!dir.mkdirs() && !dir.isDirectory()) {
                    throw new JadxRuntimeException("Can't create directory " + dir);
                }
            }
        }
    }

    public static void makeDirs(@Nullable Path dir) {
        if (dir != null) {
            FileUtils.makeDirs(dir.toFile());
        }
    }

    public static void deleteFileIfExists(Path filePath) throws IOException {
        Files.deleteIfExists(filePath);
    }

    public static boolean deleteDir(File dir) {
        FileUtils.deleteDir(dir.toPath());
        return true;
    }

    public static void deleteDirIfExists(Path dir) {
        if (Files.exists(dir, new LinkOption[0])) {
            try {
                FileUtils.deleteDir(dir);
            }
            catch (Exception e) {
                LOG.error("Failed to delete dir: {}", (Object)dir.toAbsolutePath(), (Object)e);
            }
        }
    }

    private static void deleteDir(Path dir) {
        FileUtils.deleteDir(dir, false);
    }

    private static void deleteDir(Path dir, boolean keepRootDir) {
        try {
            final ArrayList files = new ArrayList();
            final ArrayList directories = new ArrayList();
            Files.walkFileTree(dir, Collections.emptySet(), Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                @NotNull
                public FileVisitResult visitFile(@NotNull Path file, @NotNull BasicFileAttributes attrs) {
                    files.add(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                @NotNull
                public FileVisitResult postVisitDirectory(@NotNull Path directory, IOException exc) {
                    directories.add(directory);
                    return FileVisitResult.CONTINUE;
                }
            });
            if (!files.isEmpty()) {
                files.parallelStream().forEach(path -> {
                    try {
                        Files.delete(path);
                    }
                    catch (Exception e) {
                        LOG.warn("Failed to delete file {}", (Object)path.toAbsolutePath(), (Object)e);
                    }
                });
            }
            if (keepRootDir) {
                ListUtils.removeLast(directories);
            }
            for (Path directory : directories) {
                try {
                    Files.delete(directory);
                }
                catch (IOException e) {
                    LOG.warn("Failed to delete directory {}", (Object)directory.toAbsolutePath(), (Object)e);
                }
            }
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to delete directory " + dir, e);
        }
    }

    public static void clearTempRootDir() {
        if (Files.isDirectory(tempRootDir, new LinkOption[0])) {
            FileUtils.clearDir(tempRootDir);
        }
    }

    public static void clearDir(Path clearDir) {
        try {
            FileUtils.deleteDir(clearDir, true);
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to clear directory " + clearDir, e);
        }
    }

    @Deprecated
    public static Path createTempDir(String prefix) {
        try {
            Path dir = Files.createTempDirectory(tempRootDir, prefix, new FileAttribute[0]);
            dir.toFile().deleteOnExit();
            return dir;
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to create temp directory with suffix: " + prefix, e);
        }
    }

    @Deprecated
    public static Path createTempFile(String suffix) {
        try {
            Path path = Files.createTempFile(tempRootDir, JADX_TMP_PREFIX, suffix, new FileAttribute[0]);
            path.toFile().deleteOnExit();
            return path;
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to create temp file with suffix: " + suffix, e);
        }
    }

    @Deprecated
    public static Path createTempFileNoDelete(String suffix) {
        try {
            return Files.createTempFile(Files.createTempDirectory("jadx-persist", new FileAttribute[0]), "jadx-", suffix, new FileAttribute[0]);
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to create temp file with suffix: " + suffix, e);
        }
    }

    @Deprecated
    public static Path createTempFileNonPrefixed(String fileName) {
        try {
            Path path = Files.createFile(tempRootDir.resolve(fileName), new FileAttribute[0]);
            path.toFile().deleteOnExit();
            return path;
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to create non-prefixed temp file: " + fileName, e);
        }
    }

    public static void copyStream(InputStream input, OutputStream output) throws IOException {
        int count;
        byte[] buffer = new byte[8192];
        while ((count = input.read(buffer)) != -1) {
            output.write(buffer, 0, count);
        }
    }

    public static byte[] streamToByteArray(InputStream input) throws IOException {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream();){
            FileUtils.copyStream(input, out);
            byte[] byArray = out.toByteArray();
            return byArray;
        }
    }

    public static void close(Closeable c) {
        if (c == null) {
            return;
        }
        try {
            c.close();
        }
        catch (IOException e) {
            LOG.error("Close exception for {}", (Object)c, (Object)e);
        }
    }

    public static void writeFile(Path file, String data) throws IOException {
        FileUtils.makeDirsForFile(file);
        Files.writeString(file, (CharSequence)data, StandardCharsets.UTF_8, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    }

    public static void writeFile(Path file, byte[] data) throws IOException {
        FileUtils.makeDirsForFile(file);
        Files.write(file, data, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    }

    public static void writeFile(Path file, InputStream is) throws IOException {
        FileUtils.makeDirsForFile(file);
        Files.copy(is, file, StandardCopyOption.REPLACE_EXISTING);
    }

    public static String readFile(Path textFile) throws IOException {
        return Files.readString(textFile);
    }

    public static boolean renameFile(Path sourcePath, Path targetPath) {
        try {
            Files.move(sourcePath, targetPath, StandardCopyOption.REPLACE_EXISTING);
            return true;
        }
        catch (NoSuchFileException e) {
            LOG.error("File to rename not found {}", (Object)sourcePath, (Object)e);
        }
        catch (FileAlreadyExistsException e) {
            LOG.error("File with that name already exists {}", (Object)targetPath, (Object)e);
        }
        catch (IOException e) {
            LOG.error("Error renaming file {}", (Object)e.getMessage(), (Object)e);
        }
        return false;
    }

    @NotNull
    public static File prepareFile(File file) {
        File saveFile = FileUtils.cutFileName(file);
        FileUtils.makeDirsForFile(saveFile);
        return saveFile;
    }

    private static File cutFileName(File file) {
        Object name = file.getName();
        if (((String)name).length() <= 128) {
            return file;
        }
        int dotIndex = ((String)name).indexOf(46);
        int cutAt = 128 - ((String)name).length() + dotIndex - 1;
        name = cutAt <= 0 ? ((String)name).substring(0, 127) : ((String)name).substring(0, cutAt) + ((String)name).substring(dotIndex);
        return new File(file.getParentFile(), (String)name);
    }

    public static String bytesToHex(byte[] bytes) {
        if (bytes == null || bytes.length == 0) {
            return "";
        }
        byte[] hexChars = new byte[bytes.length * 2];
        for (int j = 0; j < bytes.length; ++j) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0xF];
        }
        return new String(hexChars, StandardCharsets.UTF_8);
    }

    public static String byteToHex(int value) {
        int v = value & 0xFF;
        byte[] hexChars = new byte[]{HEX_ARRAY[v >>> 4], HEX_ARRAY[v & 0xF]};
        return new String(hexChars, StandardCharsets.US_ASCII);
    }

    public static String intToHex(int value) {
        byte[] hexChars = new byte[8];
        int v = value;
        for (int i = 7; i >= 0; --i) {
            hexChars[i] = HEX_ARRAY[v & 0xF];
            v >>>= 4;
        }
        return new String(hexChars, StandardCharsets.US_ASCII);
    }

    public static boolean isZipFile(File file) {
        boolean bl;
        FileInputStream is = new FileInputStream(file);
        try {
            int len = ZIP_FILE_MAGIC.length;
            byte[] headers = new byte[len];
            int read = ((InputStream)is).read(headers);
            bl = read == len && Arrays.equals(headers, ZIP_FILE_MAGIC);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)is).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                LOG.error("Failed to read zip file: {}", (Object)file.getAbsolutePath(), (Object)e);
                return false;
            }
        }
        ((InputStream)is).close();
        return bl;
    }

    public static String getPathBaseName(Path file) {
        String fileName = file.getFileName().toString();
        int extEndIndex = fileName.lastIndexOf(46);
        if (extEndIndex == -1) {
            return fileName;
        }
        return fileName.substring(0, extEndIndex);
    }

    public static File toFile(String path) {
        if (path == null) {
            return null;
        }
        return new File(path);
    }

    public static List<Path> toPaths(List<File> files) {
        return files.stream().map(File::toPath).collect(Collectors.toList());
    }

    public static List<Path> toPaths(File[] files) {
        return Stream.of(files).map(File::toPath).collect(Collectors.toList());
    }

    public static List<Path> toPathsWithTrim(File[] files) {
        return Stream.of(files).map(FileUtils::toPathWithTrim).collect(Collectors.toList());
    }

    public static Path toPathWithTrim(File file) {
        return FileUtils.toPathWithTrim(file.getPath());
    }

    public static Path toPathWithTrim(String file) {
        return Path.of(file.trim(), new String[0]);
    }

    public static List<Path> fileNamesToPaths(List<String> fileNames) {
        return fileNames.stream().map(x$0 -> Paths.get(x$0, new String[0])).collect(Collectors.toList());
    }

    public static List<File> toFiles(List<Path> paths) {
        return paths.stream().map(Path::toFile).collect(Collectors.toList());
    }

    public static String md5Sum(String str) {
        return FileUtils.md5Sum(str.getBytes(StandardCharsets.UTF_8));
    }

    public static String md5Sum(byte[] data) {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(data);
            return FileUtils.bytesToHex(md.digest());
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to build hash", e);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static String buildInputsHash(List<Path> inputPaths) {
        try (ByteArrayOutputStream bout = new ByteArrayOutputStream();){
            String string;
            try (DataOutputStream data = new DataOutputStream(bout);){
                List<Path> inputFiles = FileUtils.expandDirs(inputPaths);
                Collections.sort(inputFiles);
                data.write(inputPaths.size());
                data.write(inputFiles.size());
                for (Path inputFile : inputFiles) {
                    FileTime modifiedTime = Files.getLastModifiedTime(inputFile, new LinkOption[0]);
                    data.writeLong(modifiedTime.toMillis());
                }
                string = FileUtils.md5Sum(bout.toByteArray());
            }
            return string;
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to build hash for inputs", e);
        }
    }
}

