package io.github.rmuhamedgaliev.yams.server;

import com.google.inject.Guice;
import com.google.inject.Injector;
import com.google.inject.Module;
import com.thetransactioncompany.cors.CORSFilter;
import io.github.rmuhamedgaliev.yams.config.ConfigModule;
import io.github.rmuhamedgaliev.yams.config.IReadConfigFile;
import io.github.rmuhamedgaliev.yams.config.impl.ReadConfigFile;
import io.github.rmuhamedgaliev.yams.server.filters.gzip.GZIPFilter;
import org.eclipse.jetty.server.*;
import org.eclipse.jetty.servlet.FilterHolder;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.jboss.resteasy.plugins.guice.GuiceResteasyBootstrapServletContextListener;
import org.jboss.resteasy.plugins.server.servlet.HttpServletDispatcher;

import javax.servlet.DispatcherType;
import java.util.EnumSet;

/**
 * @author <a href="mailto:rinat.muhamedgaliev@gmail.com">Rinat Muhamedgaliev</a>
 * Main class for extend your app. This class contain Jetty starter with Guice and RestEasy configs
 */
public class Application {
    private final static Server server = new Server();
    private static Injector globalInjector;
    private static Injector internalInjector = Guice.createInjector(new ConfigModule());
    private static IReadConfigFile config = (ReadConfigFile) internalInjector.getInstance(IReadConfigFile.class);

    /**
     * Constructor for start your app with Guice module
     * @param module of Guice module extended of {@link io.github.rmuhamedgaliev.yams.server.ServerModule}
     */
    public Application(Module module) {
        checkConfigs(module);
        globalInjector = Guice.createInjector(module);
    }

    /**
     * Check incoming module extended of {@link io.github.rmuhamedgaliev.yams.server.ServerModule}
     * @param module @param module of Guice module extended of {@link io.github.rmuhamedgaliev.yams.server.ServerModule}
     */
    private void checkConfigs(Module module) {
        if (!module.getClass().getSuperclass().equals(ServerModule.class)) {
            System.exit(0);
        }
    }

    /**
     * Start app for listen HTTP port
     * @param join only for testing this param
     * @throws Exception any exception in start server
     */
    public void run(boolean join) throws Exception {
        globalInjector.getAllBindings();
        globalInjector.createChildInjector().getAllBindings();
        addConnector(config.getConfig().getHTTP().getPort(), config.getConfig().getHTTP().getHost());
        disableSendServerVersion();
        addRESTHandler();
        startServer(join);
    }

    /**
     * Production method for start app
     * @throws Exception any exception in start server
     */
    public void run() throws Exception {
        globalInjector.getAllBindings();
        globalInjector.createChildInjector().getAllBindings();
        addConnector(config.getConfig().getHTTP().getPort(), config.getConfig().getHTTP().getHost());
        disableSendServerVersion();
        addRESTHandler();
        startServer(true);
    }

    /**
     * Stop app with stop Jetty listen port
     * @throws Exception any exception while stop server
     */
    public void stop() throws Exception {
        server.stop();
    }

    /**
     * HTTP connector for set config server
     * @param PORT int based value for set port of Jetty server
     * @param host name of where server based this param can be null
     */
    private static void addConnector(int PORT, String host) {
        ServerConnector connector = new ServerConnector(server);
        if (host != null) {
            connector.setHost(host);
        }
        connector.setPort(PORT);
        server.addConnector(connector);
        disableSendServerVersion();
    }

    /**
     * Disable send version of Jetty server
     */
    private static void disableSendServerVersion() {
        for (Connector y : server.getConnectors()) {
            for (ConnectionFactory x : y.getConnectionFactories()) {
                if (x instanceof HttpConnectionFactory) {
                    ((HttpConnectionFactory) x).getHttpConfiguration().setSendServerVersion(false);
                }
            }
        }
    }

    /**
     * Set REST based handlers with <b>/rs/</b> prefix
     */
    private static void addRESTHandler() {
        ServletContextHandler sch = new ServletContextHandler(server, "/rs/*", true, false);
        sch.addEventListener(globalInjector.getInstance(GuiceResteasyBootstrapServletContextListener.class));
        ServletHolder sh = new ServletHolder(HttpServletDispatcher.class);
        sch.addServlet(sh, "/*");
        ServletHandler handler = new ServletHandler();

        FilterHolder fh = handler.addFilterWithMapping(CORSFilter.class, "/*", EnumSet.of(DispatcherType.REQUEST));

        fh.setInitParameter("cors.supportedMethods", "GET, HEAD, PUT, POST, DELETE, OPTIONS");
        fh.setInitParameter("cors.allowOrigin", "*");
        fh.setInitParameter("cors.supportedHeaders", "*");

        sch.addFilter(GZIPFilter.class, "/*", EnumSet.of(DispatcherType.REQUEST));
        sch.addFilter(fh, "/*", EnumSet.of(DispatcherType.REQUEST));
        server.setHandler(sch);
    }

    /**
     * Run server
     * @param join only for testing this param
     * @throws Exception any exception while server starting
     */
    private static void startServer(boolean join) throws Exception {
        server.start();
        if (join) {
            server.join();
        }
    }
}
