/*
 * Decompiled with CFR 0.152.
 */
package org.ton.java.cell;

import java.io.Serializable;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.ton.java.bitstring.BitString;
import org.ton.java.cell.BocFlags;
import org.ton.java.cell.CellBuilder;
import org.ton.java.cell.CellType;
import org.ton.java.cell.IdxItem;
import org.ton.java.cell.LevelMask;
import org.ton.java.cell.UnsignedByteReader;
import org.ton.java.utils.Utils;

public class Cell
implements Serializable {
    private static final Logger log = LoggerFactory.getLogger(Cell.class);
    BitString bits;
    List<Cell> refs;
    private CellType type;
    public int index;
    public boolean exotic;
    public LevelMask levelMask;
    private byte[] hashes;
    private int[] depthLevels;

    public BitString getBits() {
        return this.bits;
    }

    public boolean isExotic() {
        return this.exotic;
    }

    public List<Cell> getRefs() {
        return new ArrayList<Cell>(this.refs);
    }

    public byte[] getHashes() {
        return this.hashes;
    }

    public int[] getDepthLevels() {
        return this.depthLevels;
    }

    public int hashCode() {
        return new BigInteger(this.getHash()).intValue();
    }

    public boolean equals(Object o) {
        if (o instanceof Cell) {
            return Arrays.equals(this.getHash(), ((Cell)o).getHash());
        }
        return false;
    }

    public Cell() {
        this.bits = new BitString();
        this.refs = new ArrayList<Cell>(4);
        this.exotic = false;
        this.type = CellType.ORDINARY;
        this.levelMask = new LevelMask(0);
        this.hashes = new byte[0];
        this.depthLevels = new int[0];
    }

    public Cell(int bitSize) {
        this.bits = new BitString(bitSize);
        this.refs = new ArrayList<Cell>(4);
        this.exotic = false;
        this.type = CellType.ORDINARY;
        this.levelMask = this.resolveMask();
        this.hashes = new byte[0];
        this.depthLevels = new int[0];
    }

    public Cell(BitString bits, List<Cell> refs) {
        this.bits = new BitString(bits.getUsedBits());
        this.bits.writeBitString(bits.clone());
        this.refs = refs == null || refs.isEmpty() ? Collections.emptyList() : (refs.size() <= 4 ? new ArrayList<Cell>(refs) : new ArrayList<Cell>(refs));
        this.exotic = false;
        this.type = CellType.ORDINARY;
        this.levelMask = new LevelMask(0);
        this.hashes = new byte[0];
        this.depthLevels = new int[0];
    }

    public Cell(BitString bits, List<Cell> refs, int cellType) {
        this.bits = new BitString(bits.getUsedBits());
        this.bits.writeBitString(bits.clone());
        this.refs = new ArrayList<Cell>(refs);
        this.exotic = false;
        this.type = Cell.toCellType(cellType);
        this.levelMask = new LevelMask(0);
    }

    public Cell(BitString bits, int bitSize, List<Cell> refs, boolean exotic, LevelMask levelMask) {
        this.bits = new BitString(bitSize);
        this.bits.writeBitString(bits);
        this.refs = new ArrayList<Cell>(refs);
        this.exotic = exotic;
        this.type = CellType.ORDINARY;
        this.levelMask = levelMask;
    }

    public Cell(BitString bits, int bitSize, List<Cell> refs, boolean exotic, CellType cellType) {
        this.bits = new BitString(bitSize);
        this.bits.writeBitString(bits);
        this.refs = new ArrayList<Cell>(refs);
        this.exotic = exotic;
        this.type = cellType;
        this.levelMask = this.resolveMask();
    }

    public Cell(BitString bits, int bitSize, List<Cell> refs, CellType cellType) {
        this.bits = new BitString(bitSize);
        this.bits.writeBitString(bits);
        this.refs = new ArrayList<Cell>(refs);
        this.type = cellType;
        this.levelMask = this.resolveMask();
    }

    public static CellType toCellType(int cellType) {
        switch (cellType) {
            case -1: {
                return CellType.ORDINARY;
            }
            case 1: {
                return CellType.PRUNED_BRANCH;
            }
            case 2: {
                return CellType.LIBRARY;
            }
            case 3: {
                return CellType.MERKLE_PROOF;
            }
            case 4: {
                return CellType.MERKLE_UPDATE;
            }
        }
        return CellType.UNKNOWN;
    }

    public LevelMask resolveMask() {
        if (this.type == CellType.ORDINARY) {
            int mask = 0;
            for (Cell r : this.refs) {
                mask |= r.getMaxLevel();
            }
            return new LevelMask(mask);
        }
        if (this.type == CellType.PRUNED_BRANCH) {
            if (!this.refs.isEmpty()) {
                throw new Error("Pruned branch must not have refs");
            }
            BitString bs = this.bits.clone();
            bs.readUint8();
            return new LevelMask(bs.readUint8().intValue());
        }
        if (this.type == CellType.MERKLE_PROOF) {
            return new LevelMask(this.refs.get((int)0).levelMask.getMask() >> 1);
        }
        if (this.type == CellType.MERKLE_UPDATE) {
            return new LevelMask(this.refs.get((int)0).levelMask.getMask() | this.refs.get((int)1).levelMask.getMask() >> 1);
        }
        if (this.type == CellType.LIBRARY) {
            return new LevelMask(0);
        }
        throw new Error("Unknown cell type " + String.valueOf((Object)this.type));
    }

    public void calculateHashes() {
        int totalHashCount = this.levelMask.getHashIndex() + 1;
        this.hashes = new byte[32 * totalHashCount];
        this.depthLevels = new int[totalHashCount];
        int hashCount = totalHashCount;
        if (this.type == CellType.PRUNED_BRANCH) {
            hashCount = 1;
        }
        int hashIndexOffset = totalHashCount - hashCount;
        int hashIndex = 0;
        int level = this.levelMask.getLevel();
        for (int li = 0; li <= level; ++li) {
            int off;
            if (!this.levelMask.isSignificant(li)) continue;
            if (hashIndex < hashIndexOffset) {
                ++hashIndex;
                continue;
            }
            byte[] dsc = this.getDescriptors(this.levelMask.apply(li).getLevel());
            byte[] hash = new byte[]{};
            hash = Utils.concatBytes((byte[])hash, (byte[])dsc);
            if (hashIndex == hashIndexOffset) {
                if (li != 0 && this.type != CellType.PRUNED_BRANCH) {
                    throw new Error("invalid cell");
                }
                byte[] data = this.getDataBytes();
                hash = Utils.concatBytes((byte[])hash, (byte[])data);
            } else {
                if (li == 0 && this.type == CellType.PRUNED_BRANCH) {
                    throw new Error("neither pruned nor 0");
                }
                off = hashIndex - hashIndexOffset - 1;
                byte[] partHash = new byte[32];
                System.arraycopy(this.hashes, off * 32, partHash, 0, (off + 1) * 32);
                hash = Utils.concatBytes((byte[])hash, (byte[])partHash);
            }
            int depth = 0;
            for (Cell r : this.refs) {
                int childDepth = this.type == CellType.MERKLE_PROOF || this.type == CellType.MERKLE_UPDATE ? r.getDepth(li + 1) : r.getDepth(li);
                hash = Utils.concatBytes((byte[])hash, (byte[])Utils.intToByteArray((int)childDepth));
                if (childDepth <= depth) continue;
                depth = childDepth;
            }
            if (!this.refs.isEmpty() && ++depth >= 1024) {
                throw new Error("depth is more than max depth (1023)");
            }
            for (Cell r : this.refs) {
                if (this.type == CellType.MERKLE_PROOF || this.type == CellType.MERKLE_UPDATE) {
                    hash = Utils.concatBytes((byte[])hash, (byte[])r.getHash(li + 1));
                    continue;
                }
                hash = Utils.concatBytes((byte[])hash, (byte[])r.getHash(li));
            }
            off = hashIndex - hashIndexOffset;
            this.depthLevels[off] = depth;
            System.arraycopy(Utils.sha256AsArray((byte[])hash), 0, this.hashes, off * 32, 32);
            ++hashIndex;
        }
    }

    void setCellType(CellType pCellType) {
        this.type = pCellType;
    }

    void setExotic(boolean pExotic) {
        this.exotic = pExotic;
    }

    public static Cell fromBoc(String data) {
        return Cell.fromBocMultiRoot(Utils.hexToSignedBytes((String)data)).get(0);
    }

    public static Cell fromBocBase64(String data) {
        return Cell.fromBocMultiRoot(Utils.base64ToSignedBytes((String)data)).get(0);
    }

    public static Cell fromBoc(byte[] data) {
        return Cell.fromBocMultiRoot(data).get(0);
    }

    public static List<Cell> fromBocMultiRoots(String data) {
        return Cell.fromBocMultiRoot(Utils.hexToSignedBytes((String)data));
    }

    public static List<Cell> fromBocMultiRoots(byte[] data) {
        return Cell.fromBocMultiRoot(data);
    }

    public String toString() {
        return this.bits.toHex();
    }

    public int getBitLength() {
        return this.bits.toBitString().length();
    }

    public Cell clone() {
        Cell c = new Cell();
        c.bits = this.bits.clone();
        c.refs = this.refs.isEmpty() ? new ArrayList<Cell>(4) : new ArrayList<Cell>(this.refs);
        c.exotic = this.exotic;
        c.type = this.type;
        c.levelMask = this.levelMask.clone();
        if (this.hashes.length > 0) {
            c.hashes = new byte[this.hashes.length];
            System.arraycopy(this.hashes, 0, c.hashes, 0, this.hashes.length);
        }
        if (this.depthLevels.length > 0) {
            c.depthLevels = new int[this.depthLevels.length];
            System.arraycopy(this.depthLevels, 0, c.depthLevels, 0, this.depthLevels.length);
        }
        return c;
    }

    public void writeCell(Cell anotherCell) {
        this.bits.writeBitString(anotherCell.bits);
        int refsSize = anotherCell.refs.size();
        if (refsSize == 0) {
            return;
        }
        if (refsSize == 1) {
            this.refs.add(anotherCell.refs.get(0));
        } else {
            this.refs.addAll(anotherCell.refs);
        }
    }

    public int getMaxRefs() {
        return 4;
    }

    public int getFreeRefs() {
        return this.getMaxRefs() - this.refs.size();
    }

    public int getUsedRefs() {
        return this.refs.size();
    }

    public static Cell fromHex(String hexBitString) {
        try {
            boolean incomplete = hexBitString.endsWith("_");
            hexBitString = hexBitString.replaceAll("_", "");
            int[] b = Utils.hexToInts((String)hexBitString);
            BitString bs = new BitString(b);
            Boolean[] ba = bs.toBooleanArray();
            int i = ba.length - 1;
            while (incomplete && !ba[i].booleanValue()) {
                ba = Arrays.copyOf(ba, ba.length - 1);
                --i;
            }
            if (incomplete) {
                ba = Arrays.copyOf(ba, ba.length - 1);
            }
            BitString bss = new BitString(ba.length);
            bss.writeBitArray(ba);
            return CellBuilder.beginCell().storeBitString(bss).endCell();
        }
        catch (Exception e) {
            throw new Error("Cannot convert hex BitString to Cell. Error " + e.getMessage());
        }
    }

    static List<Cell> fromBocMultiRoot(byte[] data) {
        UnsignedByteReader r;
        if (data.length < 10) {
            throw new Error("Invalid boc");
        }
        byte[] reachBocMagicPrefix = Utils.hexToSignedBytes((String)"B5EE9C72");
        if (!Utils.compareBytes((byte[])reachBocMagicPrefix, (byte[])(r = new UnsignedByteReader(data)).readBytes(4L))) {
            throw new Error("Invalid boc magic header");
        }
        BocFlags bocFlags = Cell.parseBocFlags(r.readSignedByte());
        int dataSizeBytes = r.readByte();
        long cellsNum = Utils.dynInt((byte[])r.readSignedBytes(bocFlags.cellNumSizeBytes));
        long rootsNum = Utils.dynInt((byte[])r.readSignedBytes(bocFlags.cellNumSizeBytes));
        r.readBytes(bocFlags.cellNumSizeBytes);
        long dataLen = Utils.dynInt((byte[])r.readSignedBytes(dataSizeBytes));
        if (bocFlags.hasCrc32c) {
            byte[] bocWithoutCrc = Arrays.copyOfRange(data, 0, data.length - 4);
            byte[] crcInBoc = Arrays.copyOfRange(data, data.length - 4, data.length);
            byte[] crc32 = Utils.getCRC32ChecksumAsBytesReversed((byte[])bocWithoutCrc);
            if (!Utils.compareBytes((byte[])crc32, (byte[])crcInBoc)) {
                throw new Error("Crc32c hashsum mismatch");
            }
        }
        int[] rootsIndex = new int[(int)rootsNum];
        int i = 0;
        while ((long)i < rootsNum) {
            rootsIndex[i] = Utils.dynInt((byte[])r.readSignedBytes(bocFlags.cellNumSizeBytes));
            ++i;
        }
        if (bocFlags.hasCacheBits && !bocFlags.hasIndex) {
            throw new Error("cache flag cant be set without index flag");
        }
        int[] index = new int[]{};
        int j = 0;
        if (bocFlags.hasIndex) {
            index = new int[(int)cellsNum];
            byte[] idxData = r.readSignedBytes(cellsNum * (long)dataSizeBytes);
            int i2 = 0;
            while ((long)i2 < cellsNum) {
                int off = i2 * dataSizeBytes;
                int val = Utils.dynInt((byte[])Arrays.copyOfRange(idxData, off, off + dataSizeBytes));
                if (bocFlags.hasCacheBits) {
                    val /= 2;
                }
                index[j++] = val;
                ++i2;
            }
        }
        if (cellsNum > dataLen / 2L) {
            throw new Error("cells num looks malicious: data len " + String.valueOf(data) + ", cells " + cellsNum);
        }
        byte[] payload = r.readBytes(dataLen);
        return Cell.parseCells(rootsIndex, rootsNum, cellsNum, bocFlags.cellNumSizeBytes, payload, index);
    }

    private static List<Cell> parseCells(int[] rootsIndex, long rootsNum, long cellsNum, int refSzBytes, byte[] data, int[] index) {
        int i;
        Cell[] cells = new Cell[(int)cellsNum];
        int i2 = 0;
        while ((long)i2 < cellsNum) {
            cells[i2] = new Cell();
            ++i2;
        }
        int offset = 0;
        int i3 = 0;
        while ((long)i3 < cellsNum) {
            int sz;
            int ln;
            if (data.length - offset < 2) {
                throw new Error("failed to parse cell header, corrupted data");
            }
            if (Objects.nonNull(index) && index.length != 0) {
                offset = 0;
                if (i3 > 0) {
                    offset = index[i3 - 1];
                }
            }
            byte flags = data[offset];
            int refsNum = flags & 7;
            boolean special = (flags & 8) != 0;
            boolean withHashes = (flags & 0x10) != 0;
            LevelMask levelMask = new LevelMask(flags >> 5);
            if (refsNum > 4) {
                throw new Error("too many refs in cell");
            }
            int oneMore = ln % 2;
            if (data.length - (offset += 2) < (sz = (ln = data[offset + 1] & 0xFF) / 2 + oneMore)) {
                throw new Error("failed to parse cell payload, corrupted data");
            }
            if (withHashes) {
                int maskBits = (int)Math.ceil(Math.log(levelMask.mask + 1) / Math.log(2.0));
                int hashesNum = maskBits + 1;
                offset += hashesNum * 32 + hashesNum * 2;
            }
            byte[] payload = Arrays.copyOfRange(data, offset, offset + sz);
            if (data.length - (offset += sz) < refsNum * refSzBytes) {
                throw new Error("failed to parse cell refs, corrupted data");
            }
            int[] refsIndex = new int[refsNum];
            int x = 0;
            for (int j = 0; j < refsNum; ++j) {
                byte[] t = Arrays.copyOfRange(data, offset, offset + refSzBytes);
                refsIndex[x++] = Utils.dynInt((byte[])t);
                offset += refSzBytes;
            }
            Cell[] refs = new Cell[refsIndex.length];
            for (int y = 0; y < refsIndex.length; ++y) {
                if (refsIndex[y] < i3 && Objects.isNull(index)) {
                    throw new Error("reference to index which is behind parent cell");
                }
                if (refsIndex[y] >= cells.length) {
                    throw new Error("invalid index, out of scope");
                }
                refs[y] = cells[refsIndex[y]];
            }
            int bitSz = ln * 4;
            if (ln % 2 != 0) {
                for (int y = 0; y < 8; ++y) {
                    if ((payload[payload.length - 1] >> y & 1) != 1) continue;
                    bitSz += 3 - y;
                    break;
                }
            }
            cells[i3].bits = new BitString(payload, bitSz);
            cells[i3].refs = Arrays.asList(refs);
            cells[i3].exotic = special;
            cells[i3].levelMask = levelMask;
            cells[i3].type = cells[i3].getCellType();
            ++i3;
        }
        Cell[] roots = new Cell[rootsIndex.length];
        for (i = cells.length - 1; i >= 0; --i) {
            cells[i].calculateHashes();
        }
        for (i = 0; i < rootsIndex.length; ++i) {
            roots[i] = cells[rootsIndex[i]];
        }
        return Arrays.asList(roots);
    }

    static BocFlags parseBocFlags(byte data) {
        BocFlags bocFlags = new BocFlags();
        bocFlags.hasIndex = (data & 0x80) > 0;
        bocFlags.hasCrc32c = (data & 0x40) > 0;
        bocFlags.hasCacheBits = (data & 0x20) > 0;
        bocFlags.cellNumSizeBytes = data & 7;
        return bocFlags;
    }

    public String print(String indent) {
        String t = "x";
        if (this.type == CellType.MERKLE_PROOF) {
            t = "p";
        } else if (this.type == CellType.MERKLE_UPDATE) {
            t = "u";
        } else if (this.type == CellType.PRUNED_BRANCH) {
            t = "P";
        }
        StringBuilder s = new StringBuilder(indent + t + "{" + this.bits.toHex() + "}\n");
        if (Objects.nonNull(this.refs) && !this.refs.isEmpty()) {
            for (Cell i : this.refs) {
                if (!Objects.nonNull(i)) continue;
                s.append(i.print(indent + " "));
            }
        }
        return s.toString();
    }

    public String print() {
        return this.print("");
    }

    public void toFile(String filename, boolean withCrc) {
        byte[] boc = this.toBoc(withCrc);
        try {
            Files.write(Paths.get(filename, new String[0]), boc, new OpenOption[0]);
        }
        catch (Exception e) {
            log.error("Cannot write to file. Error: {} ", (Object)e.getMessage());
        }
    }

    public void toFile(String filename) {
        this.toFile(filename, true);
    }

    public String toHex(boolean withCrc) {
        return Utils.bytesToHex((byte[])this.toBoc(withCrc));
    }

    public String toHex(boolean withCrc, boolean withIndex) {
        return Utils.bytesToHex((byte[])this.toBoc(withCrc, withIndex));
    }

    public String toHex(boolean withCrc, boolean withIndex, boolean withCacheBits) {
        return Utils.bytesToHex((byte[])this.toBoc(withCrc, withIndex, withCacheBits));
    }

    public String toHex(boolean withCrc, boolean withIndex, boolean withCacheBits, boolean withTopHash) {
        return Utils.bytesToHex((byte[])this.toBoc(withCrc, withIndex, withCacheBits, withTopHash));
    }

    public String toHex(boolean withCrc, boolean withIndex, boolean withCacheBits, boolean withTopHash, boolean withIntHashes) {
        return Utils.bytesToHex((byte[])this.toBoc(withCrc, withIndex, withCacheBits, withTopHash, withIntHashes));
    }

    public String toHex() {
        return Utils.bytesToHex((byte[])this.toBoc(true));
    }

    public String bitStringToHex() {
        return this.bits.toHex();
    }

    public String toBitString() {
        return this.bits.toBitString();
    }

    public String toBase64() {
        return Utils.bytesToBase64((byte[])this.toBoc(true));
    }

    public String toBase64UrlSafe() {
        return Utils.bytesToBase64SafeUrl((byte[])this.toBoc(true));
    }

    public String toBase64(boolean withCrc) {
        return Utils.bytesToBase64((byte[])this.toBoc(withCrc));
    }

    public byte[] hash() {
        return this.getHash();
    }

    public byte[] getHash() {
        return this.getHash(this.levelMask.getLevel());
    }

    public String getShortHash() {
        String hashHex = Utils.bytesToHex((byte[])this.getHash(this.levelMask.getLevel()));
        return hashHex.substring(0, 4) + ".." + hashHex.substring(hashHex.length() - 5, hashHex.length() - 1);
    }

    public byte[] getHash(int lvl) {
        int hashIndex = this.levelMask.apply(lvl).getHashIndex();
        if (this.type == CellType.PRUNED_BRANCH) {
            int prunedHashIndex = this.levelMask.getHashIndex();
            if (hashIndex != prunedHashIndex) {
                return Arrays.copyOfRange(this.getDataBytes(), 2 + hashIndex * 32, 2 + (hashIndex + 1) * 32);
            }
            hashIndex = 0;
        }
        return Utils.slice((byte[])this.hashes, (int)(hashIndex * 32), (int)32);
    }

    public byte[] getRefsDescriptor(int lvl) {
        byte[] d1 = new byte[]{(byte)(Objects.isNull(this.refs) ? 0 : this.refs.size() + (this.exotic ? 1 : 0) * 8 + lvl * 32)};
        return d1;
    }

    public byte[] getBitsDescriptor() {
        int bitsLength = this.bits.getUsedBits();
        byte d3 = (byte)(bitsLength / 8 * 2);
        if (bitsLength % 8 != 0) {
            d3 = (byte)(d3 + 1);
        }
        return new byte[]{d3};
    }

    public int getMaxLevel() {
        int maxLevel = 0;
        for (Cell i : this.refs) {
            if (i.getMaxLevel() <= maxLevel) continue;
            maxLevel = i.getMaxLevel();
        }
        return maxLevel;
    }

    public byte[] toBoc() {
        return this.toBoc(true, false, false, false, false);
    }

    public byte[] toBoc(boolean withCRC) {
        return this.toBoc(withCRC, false, false, false, false);
    }

    public byte[] toBoc(boolean withCRC, boolean withIdx) {
        return this.toBoc(withCRC, withIdx, false, false, false);
    }

    public byte[] toBoc(boolean withCRC, boolean withIdx, boolean withCacheBits) {
        return this.toBoc(withCRC, withIdx, withCacheBits, false, false);
    }

    public byte[] toBoc(boolean withCRC, boolean withIdx, boolean withCacheBits, boolean withTopHash) {
        return this.toBoc(withCRC, withIdx, withCacheBits, withTopHash, false);
    }

    private byte[] internalToBoc(List<Cell> roots, boolean hasCrc32c, boolean hasIdx, boolean hasCacheBits, boolean hasTopHash, boolean hasIntHashes) {
        Pair<List<IdxItem>, Map<String, IdxItem>> sortedCellsAndIndex = this.flattenIndex(roots, hasTopHash);
        List sortedCells = (List)sortedCellsAndIndex.getLeft();
        Map index = (Map)sortedCellsAndIndex.getRight();
        int cellSizeBits = Utils.log2((int)(sortedCells.size() + 1));
        int cellSizeBytes = (int)Math.ceil((double)cellSizeBits / 8.0);
        byte[] payload = new byte[]{};
        for (IdxItem sortedCell : sortedCells) {
            payload = Utils.concatBytes((byte[])payload, (byte[])sortedCell.getCell().serialize(cellSizeBytes, index, sortedCell.isWithHash()));
            sortedCell.dataIndex = payload.length;
        }
        int sizeBits = Utils.log2Ceil((int)(payload.length + 1));
        byte sizeBytes = (byte)((sizeBits + 7) / 8);
        byte flagsByte = 0;
        if (hasIdx) {
            flagsByte = (byte)(flagsByte | 0x80);
        }
        if (hasCrc32c) {
            flagsByte = (byte)(flagsByte | 0x40);
        }
        if (hasCacheBits) {
            flagsByte = (byte)(flagsByte | 0x20);
        }
        flagsByte = (byte)(flagsByte | cellSizeBytes);
        byte[] data = new byte[]{};
        byte[] bocMagic = new byte[]{-75, -18, -100, 114};
        data = Utils.concatBytes((byte[])data, (byte[])bocMagic);
        data = Utils.concatBytes((byte[])data, (byte[])Utils.byteToBytes((byte)flagsByte));
        data = Utils.concatBytes((byte[])data, (byte[])Utils.byteToBytes((byte)sizeBytes));
        data = Utils.concatBytes((byte[])data, (byte[])Utils.dynamicIntBytes((BigInteger)BigInteger.valueOf(sortedCells.size()), (int)cellSizeBytes));
        data = Utils.concatBytes((byte[])data, (byte[])Utils.dynamicIntBytes((BigInteger)BigInteger.ONE, (int)cellSizeBytes));
        data = Utils.concatBytes((byte[])data, (byte[])Utils.dynamicIntBytes((BigInteger)BigInteger.ZERO, (int)cellSizeBytes));
        data = Utils.concatBytes((byte[])data, (byte[])Utils.dynamicIntBytes((BigInteger)BigInteger.valueOf(payload.length), (int)sizeBytes));
        for (Cell c : roots) {
            data = Utils.concatBytes((byte[])data, (byte[])Utils.dynamicIntBytes((BigInteger)((IdxItem)index.get((Object)Utils.bytesToHex((byte[])c.getHash()))).index, (int)cellSizeBytes));
        }
        if (hasIdx) {
            for (IdxItem sc : sortedCells) {
                long idx = sc.dataIndex;
                if (hasCacheBits) {
                    idx *= 2L;
                    if (sc.repeats > 0L) {
                        ++idx;
                    }
                }
                data = Utils.concatBytes((byte[])data, (byte[])Utils.dynamicIntBytes((BigInteger)BigInteger.valueOf(idx), (int)sizeBytes));
            }
        }
        data = Utils.appendByteArray((byte[])data, (byte[])payload);
        if (hasCrc32c) {
            data = Utils.appendByteArray((byte[])data, (byte[])Utils.getCRC32ChecksumAsBytesReversed((byte[])data));
        }
        return data;
    }

    public byte[] toBoc(boolean hasCrc32c, boolean hasIdx, boolean hasCacheBits, boolean hasTopHash, boolean hasIntHashes) {
        return this.internalToBoc(Collections.singletonList(this), hasCrc32c, hasIdx, hasCacheBits, hasTopHash, hasIntHashes);
    }

    public byte[] toBocMultiRoot(List<Cell> roots, boolean hasCrc32c, boolean hasIdx, boolean hasCacheBits, boolean hasTopHash, boolean hasIntHashes) {
        return this.internalToBoc(roots, hasCrc32c, hasIdx, hasCacheBits, hasTopHash, hasIntHashes);
    }

    /*
     * WARNING - void declaration
     */
    private Pair<List<IdxItem>, Map<String, IdxItem>> flattenIndex(List<Cell> roots, boolean hasTopHash) {
        void var7_14;
        HashMap<String, IdxItem> index = new HashMap<String, IdxItem>();
        BigInteger idx = BigInteger.ZERO;
        while (roots.size() > 0) {
            ArrayList<Cell> next = new ArrayList<Cell>(roots.size() * 4);
            for (Cell cell : roots) {
                String hash = Utils.bytesToHex((byte[])cell.getHash());
                IdxItem v = (IdxItem)index.get(hash);
                if (Objects.nonNull(v)) {
                    ++v.repeats;
                    continue;
                }
                index.put(hash, IdxItem.builder().cell(cell).index(idx).withHash(hasTopHash).build());
                idx = idx.add(BigInteger.ONE);
                next.addAll(cell.getRefs());
            }
            roots = next;
        }
        ArrayList<IdxItem> idxSlice = new ArrayList<IdxItem>(index.size());
        for (Map.Entry entry : index.entrySet()) {
            String key = (String)entry.getKey();
            IdxItem value = (IdxItem)entry.getValue();
            idxSlice.add(value);
        }
        idxSlice.sort(Comparator.comparing(lhs -> lhs.index));
        boolean verifyOrder = true;
        while (verifyOrder) {
            verifyOrder = false;
            for (IdxItem id : idxSlice) {
                for (Cell ref : id.getCell().getRefs()) {
                    IdxItem idRef = (IdxItem)index.get(Utils.bytesToHex((byte[])ref.getHash()));
                    if (idRef.index.compareTo(id.index) >= 0) continue;
                    idRef.index = idx;
                    idx = idx.add(BigInteger.ONE);
                    verifyOrder = true;
                }
            }
        }
        idxSlice.sort(Comparator.comparing(lhs -> lhs.index));
        boolean bl = false;
        while (var7_14 < idxSlice.size()) {
            ((IdxItem)idxSlice.get((int)var7_14)).index = BigInteger.valueOf((long)var7_14);
            ++var7_14;
        }
        return Pair.of(idxSlice, index);
    }

    private byte[] serialize(int refIndexSzBytes, Map<String, IdxItem> index, boolean hasHash) {
        byte[] body = this.getBits().toByteArray();
        int unusedBits = 8 - this.bits.getUsedBits() % 8;
        if (unusedBits != 8) {
            int n = body.length - 1;
            body[n] = (byte)(body[n] + (1 << unusedBits - 1));
        }
        int refsLn = this.getRefs().size() * refIndexSzBytes;
        int bufLn = 2 + body.length + refsLn;
        byte[] descriptors = this.getDescriptors(this.levelMask.getMask());
        byte[] data = Utils.concatBytes((byte[])descriptors, (byte[])body);
        long refsOffset = bufLn - refsLn;
        for (int i = 0; i < this.refs.size(); ++i) {
            String refIndex = Utils.bytesToHex((byte[])this.refs.get(i).getHash());
            byte[] src = Utils.dynamicIntBytes((BigInteger)index.get((Object)refIndex).index, (int)refIndexSzBytes);
            data = Utils.concatBytes((byte[])data, (byte[])src);
        }
        return data;
    }

    private byte[] getDescriptors(int lvl) {
        return Utils.concatBytes((byte[])this.getRefsDescriptor(lvl), (byte[])this.getBitsDescriptor());
    }

    private int getDepth(int lvlMask) {
        int prunedHashIndex;
        int hashIndex = this.levelMask.apply(lvlMask).getHashIndex();
        if (this.type == CellType.PRUNED_BRANCH && hashIndex != (prunedHashIndex = this.levelMask.getHashIndex())) {
            int off = 2 + 32 * prunedHashIndex + hashIndex * 2;
            return Utils.bytesToIntX((byte[])Utils.slice((byte[])this.getDataBytes(), (int)off, (int)2));
        }
        return this.depthLevels[hashIndex];
    }

    private byte[] getDataBytes() {
        if (this.bits.getUsedBits() % 8 > 0) {
            byte[] a = this.bits.toBitString().getBytes(StandardCharsets.UTF_8);
            int sz = a.length;
            byte[] b = new byte[sz + 1];
            System.arraycopy(a, 0, b, 0, sz);
            b[sz] = 49;
            int mod = b.length % 8;
            if (mod > 0) {
                b = Utils.rightPadBytes((byte[])b, (int)(b.length + (8 - mod)), (char)'0');
            }
            return Utils.bitStringToByteArray((String)new String(b));
        }
        return this.bits.toByteArray();
    }

    public static CellType getCellType(Cell c) {
        return c.getCellType();
    }

    public CellType getCellType() {
        if (!this.exotic) {
            return CellType.ORDINARY;
        }
        if (this.bits.getUsedBits() < 8) {
            return CellType.UNKNOWN;
        }
        BitString clonedBits = this.bits.clone();
        CellType cellType = Cell.toCellType(clonedBits.readUint(8).intValue());
        switch (cellType) {
            case PRUNED_BRANCH: {
                LevelMask msk;
                int lvl;
                if (this.bits.getUsedBits() >= 288 && (lvl = (msk = new LevelMask(clonedBits.readUint(8).intValue())).getLevel()) > 0 && lvl <= 3 && this.bits.getUsedBits() >= 16 + 272 * msk.apply(lvl - 1).getHashIndex() + 1) {
                    return CellType.PRUNED_BRANCH;
                }
            }
            case MERKLE_PROOF: {
                if (this.refs.size() == 1 && this.bits.getUsedBits() == 280) {
                    return CellType.MERKLE_PROOF;
                }
            }
            case MERKLE_UPDATE: {
                if (this.refs.size() == 2 && this.bits.getUsedBits() == 552) {
                    return CellType.MERKLE_UPDATE;
                }
            }
            case LIBRARY: {
                if (this.bits.getUsedBits() != 264) break;
                return CellType.LIBRARY;
            }
        }
        return CellType.UNKNOWN;
    }
}

