/*
 * Decompiled with CFR 0.152.
 */
package io.github.joealisson.mmocore.internal;

import io.github.joealisson.mmocore.ResourcePool;
import io.github.joealisson.mmocore.internal.InternalWritableBuffer;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Objects;

public class DynamicPacketBuffer
extends InternalWritableBuffer {
    private PacketNode[] nodes = new PacketNode[8];
    private PacketNode currentNode;
    private final ResourcePool resourcePool;
    private int nodeCount;
    private int bufferIndex;
    private int limit;

    public DynamicPacketBuffer(ByteBuffer buffer, ResourcePool resourcePool) {
        this.resourcePool = resourcePool;
        this.newNode(buffer, 0);
    }

    private void newNode(ByteBuffer buffer, int initialIndex) {
        if (this.nodes.length == this.nodeCount) {
            this.nodes = Arrays.copyOf(this.nodes, (int)((double)(this.nodes.length + 1) * 1.2));
        }
        PacketNode node = new PacketNode(buffer, initialIndex, this.nodeCount);
        this.nodes[this.nodeCount++] = node;
        this.limit = node.endIndex;
    }

    @Override
    public void writeByte(byte value) {
        this.ensureSize(this.bufferIndex + 1);
        this.setByte(this.bufferIndex++, value);
    }

    @Override
    public void writeByte(int index, byte value) {
        this.checkBounds(index, 1);
        this.setByte(index, value);
    }

    private void checkBounds(int index, int length) {
        if (index < 0 || index + length > this.limit) {
            throw new IndexOutOfBoundsException("Trying access index " + index + " until index " + (index + length) + " , max accessible index is " + this.limit);
        }
    }

    private void setByte(int index, byte value) {
        PacketNode node = this.indexToNode(index);
        node.buffer.put(node.idx(index), value);
    }

    @Override
    public void writeBytes(byte[] bytes) {
        if (Objects.isNull(bytes) || bytes.length == 0) {
            return;
        }
        this.ensureSize(this.bufferIndex + bytes.length);
        this.setBytes(this.bufferIndex, bytes);
        this.bufferIndex += bytes.length;
    }

    private void setBytes(int index, byte[] bytes) {
        int available;
        PacketNode node = this.indexToNode(index);
        int length = bytes.length;
        int offset = 0;
        do {
            available = Math.min(length, node.endIndex - index);
            node.buffer.position(node.idx(index));
            node.buffer.put(bytes, offset, available);
            node.buffer.position(0);
            offset += available;
            index += available;
            node = this.nodes[Math.min(node.offset + 1, this.nodes.length - 1)];
        } while ((length -= available) > 0);
    }

    @Override
    public void writeShort(short value) {
        this.ensureSize(this.bufferIndex + 2);
        this.setShort(this.bufferIndex, value);
        this.bufferIndex += 2;
    }

    @Override
    public void writeShort(int index, short value) {
        this.checkBounds(index, 2);
        this.setShort(index, value);
    }

    private void setShort(int index, short value) {
        PacketNode node = this.indexToNode(index);
        if (index + 2 <= node.endIndex) {
            node.buffer.putShort(node.idx(index), value);
        } else {
            this.setByte(index, (byte)value);
            this.setByte(index + 1, (byte)(value >>> 8));
        }
    }

    @Override
    public void writeChar(char value) {
        this.writeShort((short)value);
    }

    @Override
    public void writeInt(int value) {
        this.ensureSize(this.bufferIndex + 4);
        this.setInt(this.bufferIndex, value);
        this.bufferIndex += 4;
    }

    @Override
    public void writeInt(int index, int value) {
        this.checkBounds(index, 4);
        this.setInt(index, value);
    }

    private void setInt(int index, int value) {
        PacketNode node = this.indexToNode(index);
        if (index + 4 <= node.endIndex) {
            node.buffer.putInt(node.idx(index), value);
        } else {
            this.setShort(index, (short)value);
            this.setShort(index + 2, (short)(value >>> 16));
        }
    }

    @Override
    public void writeFloat(float value) {
        this.writeInt(Float.floatToRawIntBits(value));
    }

    @Override
    public void writeLong(long value) {
        this.ensureSize(this.bufferIndex + 8);
        this.setLong(this.bufferIndex, value);
        this.bufferIndex += 8;
    }

    private void setLong(int index, long value) {
        PacketNode node = this.indexToNode(index);
        if (index + 8 <= node.endIndex) {
            node.buffer.putLong(node.idx(index), value);
        } else {
            this.setInt(index, (int)value);
            this.setInt(index + 4, (int)(value >>> 32));
        }
    }

    @Override
    public void writeDouble(double value) {
        this.writeLong(Double.doubleToRawLongBits(value));
    }

    @Override
    public int position() {
        return this.bufferIndex;
    }

    @Override
    public void position(int pos) {
        this.bufferIndex = pos;
    }

    @Override
    public byte readByte(int index) {
        this.checkSize(index + 1);
        return this.getByte(index);
    }

    private byte getByte(int index) {
        PacketNode node = this.indexToNode(index);
        return node.buffer.get(node.idx(index));
    }

    public void readBytes(int index, byte[] data) {
        int available;
        this.checkSize(index + data.length);
        PacketNode node = this.indexToNode(index);
        int length = data.length;
        int offset = 0;
        do {
            available = Math.min(length, node.endIndex - index);
            node.buffer.position(node.idx(index));
            node.buffer.get(data, offset, available);
            offset += available;
            index += available;
            node = this.nodes[Math.min(node.offset + 1, this.nodes.length - 1)];
        } while ((length -= available) > 0);
    }

    @Override
    public short readShort(int index) {
        this.checkSize(index + 2);
        return this.getShort(index);
    }

    private short getShort(int index) {
        PacketNode node = this.indexToNode(index);
        if (index + 2 <= node.endIndex) {
            return node.buffer.getShort(node.idx(index));
        }
        return (short)(this.getByte(index) & 0xFF | (this.getByte(index + 1) & 0xFF) << 8);
    }

    @Override
    public int readInt(int index) {
        this.checkSize(index + 4);
        return this.getInt(index);
    }

    private int getInt(int index) {
        PacketNode node = this.indexToNode(index);
        if (index + 4 <= node.endIndex) {
            return node.buffer.getInt(node.idx(index));
        }
        return this.getShort(index) & 0xFFFF | (this.getShort(index + 2) & 0xFFFF) << 16;
    }

    public float readFloat(int index) {
        return Float.intBitsToFloat(this.readInt(index));
    }

    public long readLong(int index) {
        this.checkSize(index + 8);
        PacketNode node = this.indexToNode(index);
        if (index + 8 <= node.endIndex) {
            return node.buffer.getLong(node.idx(index));
        }
        return (long)this.getInt(index) & 0xFFFFFFFFL | ((long)this.getInt(index + 4) & 0xFFFFFFFFL) << 32;
    }

    public double readDouble(int index) {
        return Double.longBitsToDouble(this.readLong(index));
    }

    @Override
    public int limit() {
        return this.limit;
    }

    @Override
    public void limit(int newLimit) {
        if (this.limit != this.capacity()) {
            PacketNode node = this.indexToNode(this.limit);
            node.buffer.clear();
        }
        this.ensureSize(newLimit + 1);
        this.limit = newLimit;
        this.limitBuffer();
    }

    public int capacity() {
        return this.nodes[this.nodeCount - 1].endIndex;
    }

    @Override
    public void mark() {
        this.limit = this.bufferIndex;
        this.limitBuffer();
    }

    private void limitBuffer() {
        PacketNode node = this.indexToNode(this.limit - 1);
        node.buffer.limit(node.idx(this.limit));
    }

    private void ensureSize(int sizeRequired) {
        if (this.capacity() < sizeRequired) {
            int newSize;
            for (newSize = 64; newSize < sizeRequired; newSize <<= 1) {
            }
            this.increaseBuffers(newSize);
        }
    }

    private void increaseBuffers(int size) {
        int diffSize = size - this.capacity();
        ByteBuffer buffer = this.resourcePool.getBuffer(diffSize);
        PacketNode lastNode = this.nodes[this.nodeCount - 1];
        this.newNode(buffer, lastNode.endIndex);
    }

    private void checkSize(int size) {
        if (this.limit < size || size < 0) {
            throw new IndexOutOfBoundsException("Trying access index " + size + ", max size is " + this.limit);
        }
    }

    private PacketNode indexToNode(int index) {
        if (Objects.nonNull(this.currentNode) && this.currentNode.initialIndex <= index && this.currentNode.endIndex > index) {
            return this.currentNode;
        }
        int min = 0;
        int max = this.nodeCount - 1;
        while (min <= max) {
            int mid = min + max >>> 1;
            PacketNode node = this.nodes[mid];
            if (index >= node.endIndex) {
                min = mid + 1;
                continue;
            }
            if (index < node.initialIndex) {
                max = mid - 1;
                continue;
            }
            this.currentNode = node;
            return node;
        }
        throw new IndexOutOfBoundsException("Could not map the index to a node: " + index);
    }

    @Override
    public ByteBuffer[] toByteBuffers() {
        int maxNode = this.indexToNode((int)(this.limit - 1)).offset;
        ByteBuffer[] buffers = new ByteBuffer[maxNode + 1];
        for (int i = 0; i <= maxNode; ++i) {
            buffers[i] = this.nodes[i].buffer;
        }
        return buffers;
    }

    @Override
    public void releaseResources() {
        for (int i = 0; i < this.nodeCount; ++i) {
            this.resourcePool.recycleBuffer(this.nodes[i].buffer);
            this.nodes[i] = null;
        }
        this.nodeCount = 0;
        this.bufferIndex = 0;
    }

    private static class PacketNode {
        private final ByteBuffer buffer;
        private final int initialIndex;
        private final int endIndex;
        private final int offset;

        private PacketNode(ByteBuffer buffer, int initialIndex, int offset) {
            this.buffer = buffer;
            this.initialIndex = initialIndex;
            this.endIndex = initialIndex + buffer.capacity();
            this.offset = offset;
        }

        public int idx(int index) {
            return index - this.initialIndex;
        }
    }
}

