package com.gongbo.export.core;

import com.gongbo.export.annotations.AutoExport;
import com.gongbo.export.annotations.AutoExports;
import com.gongbo.export.config.ExportConfig;
import com.gongbo.export.config.ResultHandler;
import com.gongbo.export.core.handler.FieldFilter;
import com.gongbo.export.core.provider.ExportProvider;
import com.gongbo.export.core.provider.easyexcel.EasyExcelProvider;
import com.gongbo.export.entity.ExportContext;
import com.gongbo.export.entity.ExportFieldInfo;
import com.gongbo.export.entity.ExportParam;
import com.gongbo.export.exception.ExportFailedException;
import com.gongbo.export.exception.NotSupportExportException;
import com.gongbo.export.utils.CollectionUtil;
import com.gongbo.export.utils.ExportUtils;
import com.gongbo.export.utils.ReflectUtil;
import com.gongbo.export.utils.StringUtil;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;

import javax.servlet.http.HttpServletResponse;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class ExportHelper {

    /**
     * 构建导出上下文
     *
     * @param exportParam
     * @param targetMethod
     * @param exportConfig
     * @param resultHandler
     * @return
     */
    public static ExportContext buildExportContext(ExportParam exportParam, Method targetMethod, ExportConfig exportConfig, ResultHandler resultHandler) {
        //查找对应EnableExport注解
        AutoExport autoExport = findAutoExportAnnotation(exportParam.getExportTag(), targetMethod);

        //获取对应模型类
        Class<?> modelClass = getModelClass(targetMethod, autoExport, exportConfig);

        //获取导出字段信息
        List<ExportFieldInfo> exportFieldInfos = getExportFieldInfos(autoExport, modelClass, EasyExcelProvider.getInstance());

        //获取导出文件名
        String fileName = buildFileName(autoExport);

        return ExportContext.builder()
                .exportConfig(exportConfig)
                .resultHandler(resultHandler)
                .modelClass(modelClass)
                .autoExport(autoExport)
                .fileName(fileName)
                .sheetName(autoExport.sheetName())
                .template(autoExport.template())
                .excelType(autoExport.excelType())
                .fieldInfos(exportFieldInfos)
                .exportParam(exportParam)
                .outputPath(autoExport.outputPath())
                .formula(autoExport.formula())
                .userContext(new HashMap<>())
                .build();
    }

    /**
     * 获取导出模型类
     *
     * @param targetMethod
     * @param autoExport
     * @param exportConfig
     * @return
     */
    private static Class<?> getModelClass(Method targetMethod, AutoExport autoExport, ExportConfig exportConfig) {
        //没有导出模型类
        if (autoExport.modelClass() == AutoExport.NoneModel.class) {
            return null;
        }
        //注解配置
        else if (autoExport.modelClass() != Object.class) {
            return autoExport.modelClass();
        }
        //根据方法返回类型查找
        else {
            return Optional.ofNullable(ExportUtils.getModelClass(targetMethod, exportConfig))
                    .orElseThrow(() -> new IllegalArgumentException("无法提取到导出模型参数，请检查导出方法或在EnableExport注解上添加modelClass属性！"));
        }
    }

    /**
     * 获取导出字段信息
     *
     * @param autoExport
     * @param clazz
     * @param exportProvider
     * @return
     */
    private static List<ExportFieldInfo> getExportFieldInfos(AutoExport autoExport, Class<?> clazz, ExportProvider exportProvider) {
        if (clazz == null) {
            return Collections.emptyList();
        }

        //字段过滤器
        FieldFilter fieldFilter = ExportHandlers.of(autoExport.fieldFilter());

        return ReflectUtil.getFields(clazz, true).stream()
                .map(field -> {
                    ExportFieldInfo exportFieldInfo = exportProvider.findExportFieldInfo(field);

                    return Optional.ofNullable(exportFieldInfo)
                            .filter(exportFieldInfo1 -> fieldFilter.predict(field))
                            .orElse(null);
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    /**
     * 根据exportTag获取对应AutoExport注解
     *
     * @param exportTag
     * @param targetMethod
     * @return
     */
    public static AutoExport findAutoExportAnnotation(String exportTag, Method targetMethod) {
        AutoExport[] autoExports = Optional.ofNullable(targetMethod.getAnnotation(AutoExports.class))
                .map(AutoExports::value)
                .orElseGet(() -> targetMethod.getAnnotationsByType(AutoExport.class));

        //没有找到注解
        if (autoExports == null || autoExports.length == 0) {
            throw new NotSupportExportException(MessageFormat.format("该方法[{0}]不支持导出，若要开启导出，请在对应请求方法上配置EnableExport注解开启导出", targetMethod.getName()));
        }

        //根据exportGroup过滤
        Predicate<AutoExport> filter;
        if (StringUtil.isNotEmpty(exportTag)) {
            filter = e -> exportTag.equals(e.tag());
        } else {
            filter = e -> StringUtil.isEmpty(e.tag());
        }

        List<AutoExport> autoExportList = Arrays.stream(autoExports)
                .filter(filter)
                .collect(Collectors.toList());

        //没有找到对应注解
        if (CollectionUtil.isEmpty(autoExportList)) {
            throw new NotSupportExportException(MessageFormat.format("在该方法[{0}]上没有匹配到对应导出分组:{1}", targetMethod.getName(), exportTag));
        }

        //多个注解匹配
        if (autoExportList.size() > 1) {
            throw new ExportFailedException(MessageFormat.format("在该方法[{0}]上匹配到多个导出分组都为:{1}", targetMethod.getName(), exportTag));
        }

        //返回匹配的注解
        return autoExportList.get(0);
    }

    /**
     * 获取文件名
     *
     * @param autoExport
     * @return
     */
    public static String buildFileName(AutoExport autoExport) {
        String name = ExportHandlers.of(autoExport.fileNameConvert())
                .apply(autoExport.fileName());
        if (StringUtil.isEmpty(name)) {
            name = String.valueOf(System.currentTimeMillis());
        }
        return name;
    }

    /**
     * 设置响应头信息
     *
     * @param serverHttpResponse
     * @param exportContext
     */
    public static void setDownloadResponseHeaders(HttpServletResponse serverHttpResponse, ExportContext exportContext) {
        String fileName = exportContext.getFileName();
        String excelFileSuffix = exportContext.getExcelType().getValue();
        setDownloadResponseHeaders(serverHttpResponse, fileName + excelFileSuffix);
    }

    /**
     * 设置响应头信息
     *
     * @param serverHttpResponse
     * @param fileName
     */
    public static void setDownloadResponseHeaders(HttpServletResponse serverHttpResponse, String fileName) {
        // 这里注意 有同学反应使用swagger 会导致各种问题，请直接用浏览器或者用postman
        serverHttpResponse.addHeader( "Content-Type", "application/vnd.ms-excel;charset=UTF-8");

        try {
            // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
            fileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8.name()).replaceAll("\\+", "%20");
        } catch (UnsupportedEncodingException ignored) {
        }

        serverHttpResponse.addHeader("Content-Disposition", "attachment;filename*=utf-8''" + fileName);
    }

}
