package plus.easydo.starter.oauth.server.serializer;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.parser.DefaultJSONParser;
import com.alibaba.fastjson.parser.Feature;
import com.alibaba.fastjson.parser.deserializer.ObjectDeserializer;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.OAuth2Request;
import org.springframework.security.oauth2.provider.TokenRequest;

import java.io.Serializable;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author suidd
 * name Oauth2AuthenticationSerializer
 * description 自定义OAuth2认证序列化工具类
 * date 2020/4/15 9:43
 * Version 1.0
 **/
public class Oauth2AuthenticationSerializer implements ObjectDeserializer {

    @Override
    public <T> T deserialze(DefaultJSONParser parser, Type type, Object fieldName) {
        if (type == OAuth2Authentication.class) {
            try {
                Object o = parse(parser);
                if (o == null) {
                    return null;
                } else if (o instanceof OAuth2Authentication) {
                    //noinspection unchecked
                    return (T) o;
                }

                JSONObject jsonObject = (JSONObject) o;
                OAuth2Request request = parseOauth2RequestAuth2Request(jsonObject);

                //判断json节点userAuthentication的类型，根据类型动态取值
                //UsernamePasswordAuthenticationToken 密码模式/授权码模式下，存储类型为UsernamePasswordAuthenticationToken
                //PreAuthenticatedAuthenticationToken 刷新token模式下，存储类型为PreAuthenticatedAuthenticationToken
                Object autoType = jsonObject.get("userAuthentication");
                if (autoType != null) {
                    //noinspection unchecked
                    return (T) new OAuth2Authentication(request, jsonObject.getObject("userAuthentication", (Type) autoType.getClass()));
                }else {
                    return (T) new OAuth2Authentication(request, null);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            return null;
        }
        return null;
    }

    private OAuth2Request parseOauth2RequestAuth2Request(JSONObject jsonObject) {
        JSONObject json = jsonObject.getObject("oAuth2Request", JSONObject.class);
        Map<String, String> requestParameters = json.getObject("requestParameters", Map.class);
        String clientId = json.getString("clientId");
        String grantType = json.getString("grantType");
        String redirectUri = json.getString("redirectUri");
        Boolean approved = json.getBoolean("approved");
        Set<String> responseTypes = json
                .getObject("responseTypes", new TypeReference<Set<String>>() {
                });
        Set<String> scope = parseScopeSet(json);
        Set<SimpleGrantedAuthority> grantedAuthorities = json.getObject("authorities", new TypeReference<Set<SimpleGrantedAuthority>>() {
        });
        Set<String> resourceIds = json
                .getObject("resourceIds", new TypeReference<Set<String>>() {
                });
        Map<String, Serializable> extensions = json
                .getObject("extensions", new TypeReference<Map<String, Serializable>>() {
                });

        OAuth2Request request = new OAuth2Request(requestParameters, clientId,
                grantedAuthorities, approved, scope, resourceIds, redirectUri, responseTypes, extensions);
        TokenRequest tokenRequest = new TokenRequest(requestParameters, clientId, scope, grantType);
        request.refresh(tokenRequest);
        return request;
    }


    @Override
    public int getFastMatchToken() {
        return 0;
    }

    private Object parse(DefaultJSONParser parse) {

        JSONObject object = new JSONObject(parse.lexer.isEnabled(Feature.OrderedField));
        Object parsedObject = parse.parseObject(object);
        if (parsedObject instanceof JSONObject) {
            return parsedObject;
        } else if (parsedObject instanceof OAuth2Authentication) {
            return parsedObject;
        } else {
            return parsedObject == null ? null : new JSONObject((Map) parsedObject);
        }
    }

    private Set<String> parseScopeSet(JSONObject json) {
        JSONArray scopes = json.getJSONArray("scope");
        return scopes.stream().map(Object::toString).collect(Collectors.toSet());
    }
}


