package io.github.douira.glsl_transformer.transform;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import org.antlr.v4.runtime.CharStreams;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.Parser;
import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.pattern.ParseTreeMatch;
import org.antlr.v4.runtime.tree.pattern.ParseTreePattern;
import org.antlr.v4.runtime.tree.xpath.XPath;

import io.github.douira.glsl_transformer.GLSLLexer;
import io.github.douira.glsl_transformer.GLSLParser;
import io.github.douira.glsl_transformer.GLSLParserBaseListener;
import io.github.douira.glsl_transformer.generic.EditContext;
import io.github.douira.glsl_transformer.generic.EmptyTerminalNode;

abstract class Phase extends GLSLParserBaseListener {
  private PhaseCollector parent;

  void setParent(PhaseCollector parent) {
    this.parent = parent;
  }

  protected EditContext getEditContext() {
    return parent.editContext;
  }

  protected Parser getParser() {
    return parent.parser;
  }

  protected static List<ParseTree> getSiblings(ParserRuleContext node) {
    return node.getParent().children;
  }

  private void replaceNode(ParserRuleContext removeNode, ParseTree newNode) {
    var children = getSiblings(removeNode);
    children.set(children.indexOf(removeNode), newNode);
    getEditContext().omitNodeTokens(removeNode);
  }

  /**
   * Replaces the given node in its parent with a new node generated by parsing
   * the given string with the given method of the parser. See
   * {@link #createLocalRoot} for details of creating parsed nodes.
   * 
   * @param node        The node to be replaced
   * @param newContents The string from which a new node is generated
   * @param parseMethod The method with which the string will be parsed
   */
  protected void replaceNode(ParserRuleContext node, String newContents,
      Function<GLSLParser, ParserRuleContext> parseMethod) {
    replaceNode(node, createLocalRoot(newContents, parseMethod));
  }

  protected void removeNode(ParserRuleContext node) {
    // the node needs to be replaced with something to preserve the containing
    // array's length or there's a NullPointerException in the walker
    replaceNode(node, new EmptyTerminalNode());
  }

  protected XPath compilePath(String xpath) {
    return new XPath(getParser(), xpath);
  }

  protected ParseTreePattern compilePattern(String pattern, int rootRule) {
    return getParser().compileParseTreePattern(pattern, rootRule);
  }

  /**
   * This method uses a statically constructed xpath so it doesn't need to be
   * repeatedly constructed. The subtrees yielded by the xpath need to start with
   * the rule that the pattern was constructed with or nothing will match.
   * 
   * Adapted from ANTLR's implementation of
   * {@link org.antlr.v4.runtime.tree.pattern.ParseTreePattern#findAll}.
   * 
   * @param tree    The parse tree to find and match in
   * @param xpath   The xpath that leads to a subtree for matching
   * @param pattern The pattern that tests the subtrees for matches
   * @return A list of all matches resulting from the subtrees
   */
  public List<ParseTreeMatch> findAndMatch(ParseTree tree, XPath xpath, ParseTreePattern pattern) {
    var subtrees = xpath.evaluate(tree);
    var matches = new ArrayList<ParseTreeMatch>();
    for (ParseTree sub : subtrees) {
      ParseTreeMatch match = pattern.match(sub);
      if (match.succeeded()) {
        matches.add(match);
      }
    }
    return matches;
  }

  /**
   * Overwrite this method to add a check of if this phase should be run at all.
   * Especially for WalkPhase this is important since it reduces the number of
   * listeners that need to be processed.
   * 
   * @return If the phase should run. {@code true} by default.
   */
  protected boolean isActive() {
    return true;
  }

  protected void init() {
    // to be possibly overwritten by the implementing class
  }

  /**
   * Parses the given string using the given parser method. Since the parser
   * doesn't know which part of the parse tree any string would be part of, we
   * need to tell it. In many cases multiple methods would produce a correct
   * result. However, this can lead to a truncated parse tree when the resulting
   * node is inserted into a bigger parse tree. The parsing method should be
   * chosen such that when the resulting node is inserted into a parse tree, the
   * tree has the same structure as if it had been parsed as one piece.
   * 
   * For example, the code fragment {@code foo()} could be parsed as a
   * {@code functionCall}, a {@code primaryExpression}, an {@code expression} or
   * other enclosing parse rules. If it's inserted into an expression, it should
   * be parsed as an {@code expression} so that this rule isn't missing from the
   * parse tree. Using the wrong parse method often doesn't matter but it can
   * cause tree matchers to not find the node if they are, for example, looking
   * for an {@code expression} specifically.
   * 
   * @param str         The string to be parsed
   * @param parseMethod The parser method with which the string is parsed
   * @return The resulting parsed node
   */
  public ParserRuleContext createLocalRoot(String str, Function<GLSLParser, ParserRuleContext> parseMethod) {
    var input = CharStreams.fromString(str);
    var lexer = new GLSLLexer(input);
    var commonTokenStream = new CommonTokenStream(lexer);
    var node = parseMethod.apply(new GLSLParser(commonTokenStream));
    getEditContext().registerLocalRoot(node, commonTokenStream);
    return node;
  }
}
