/*
 * Decompiled with CFR 0.152.
 */
package io.github.bonigarcia.seljup;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.async.ResultCallback;
import com.github.dockerjava.api.command.CreateContainerCmd;
import com.github.dockerjava.api.command.ExecCreateCmdResponse;
import com.github.dockerjava.api.exception.DockerException;
import com.github.dockerjava.api.model.Bind;
import com.github.dockerjava.api.model.Capability;
import com.github.dockerjava.api.model.ContainerNetwork;
import com.github.dockerjava.api.model.ExposedPort;
import com.github.dockerjava.api.model.Frame;
import com.github.dockerjava.api.model.HostConfig;
import com.github.dockerjava.api.model.Ports;
import com.github.dockerjava.api.model.PullResponseItem;
import com.github.dockerjava.core.DefaultDockerClientConfig;
import com.github.dockerjava.core.DockerClientBuilder;
import com.github.dockerjava.core.DockerClientConfig;
import com.github.dockerjava.httpclient5.ApacheDockerHttpClient;
import com.github.dockerjava.transport.DockerHttpClient;
import com.google.common.base.Strings;
import io.github.bonigarcia.seljup.DockerContainer;
import io.github.bonigarcia.seljup.DockerHost;
import io.github.bonigarcia.seljup.InternalPreferences;
import io.github.bonigarcia.seljup.SeleniumJupiterException;
import io.github.bonigarcia.seljup.config.Config;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DockerService {
    final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private Config config;
    private String dockerDefaultSocket;
    private int dockerWaitTimeoutSec;
    private int dockerPollTimeMs;
    private DockerClient dockerClient;
    private InternalPreferences preferences;
    private boolean localDaemon = true;
    private URI dockerHostUri;

    public DockerService(Config config, InternalPreferences preferences) {
        this.config = config;
        this.preferences = preferences;
        this.dockerDefaultSocket = this.getConfig().getDockerDefaultSocket();
        this.dockerWaitTimeoutSec = this.getConfig().getDockerWaitTimeoutSec();
        this.dockerPollTimeMs = this.getConfig().getDockerPollTimeMs();
        DockerHost dockerHostFromEnv = DockerHost.fromEnv();
        this.dockerClient = this.getDockerClient(dockerHostFromEnv.endpoint());
    }

    private DockerClient getDockerClient(String dockerHost) {
        DefaultDockerClientConfig.Builder dockerClientConfigBuilder = DefaultDockerClientConfig.createDefaultConfigBuilder();
        if (!Strings.isNullOrEmpty((String)dockerHost)) {
            dockerClientConfigBuilder.withDockerHost(dockerHost);
        }
        DefaultDockerClientConfig dockerClientConfig = dockerClientConfigBuilder.build();
        this.dockerHostUri = dockerClientConfig.getDockerHost();
        ApacheDockerHttpClient dockerHttpClient = new ApacheDockerHttpClient.Builder().dockerHost(this.dockerHostUri).build();
        return DockerClientBuilder.getInstance((DockerClientConfig)dockerClientConfig).withDockerHttpClient((DockerHttpClient)dockerHttpClient).build();
    }

    public String getHost(String containerId, String network) throws DockerException {
        String dockerHost = this.getConfig().getDockerHost();
        if (!dockerHost.isEmpty()) {
            return dockerHost;
        }
        return SystemUtils.IS_OS_LINUX ? ((ContainerNetwork)this.dockerClient.inspectContainerCmd(containerId).exec().getNetworkSettings().getNetworks().get(network)).getGateway() : Optional.ofNullable(this.dockerHostUri.getHost()).orElse("localhost");
    }

    public synchronized String startContainer(DockerContainer dockerContainer) throws DockerException {
        String imageId = dockerContainer.getImageId();
        this.log.info("Starting Docker container {}", (Object)imageId);
        HostConfig hostConfigBuilder = new HostConfig();
        String containerId = null;
        try (CreateContainerCmd containerConfigBuilder = this.dockerClient.createContainerCmd(imageId);){
            Optional<List<String>> entryPoint;
            Optional<List<String>> cmd;
            Optional<List<String>> envs;
            Optional<List<Bind>> binds;
            List<String> exposedPorts;
            Optional<String> network;
            boolean privileged = dockerContainer.isPrivileged();
            if (privileged) {
                this.log.trace("Using privileged mode");
                hostConfigBuilder.withPrivileged(Boolean.valueOf(true));
                hostConfigBuilder.withCapAdd(new Capability[]{Capability.NET_ADMIN, Capability.NET_RAW});
            }
            if ((network = dockerContainer.getNetwork()).isPresent()) {
                this.log.trace("Using network: {}", (Object)network.get());
                hostConfigBuilder.withNetworkMode(network.get());
            }
            if (!(exposedPorts = dockerContainer.getExposedPorts()).isEmpty()) {
                this.log.trace("Using exposed ports: {}", exposedPorts);
                containerConfigBuilder.withExposedPorts(exposedPorts.stream().map(ExposedPort::parse).collect(Collectors.toList()));
                hostConfigBuilder.withPublishAllPorts(Boolean.valueOf(true));
            }
            if ((binds = dockerContainer.getBinds()).isPresent()) {
                this.log.trace("Using binds: {}", binds.get());
                hostConfigBuilder.withBinds(binds.get());
            }
            if ((envs = dockerContainer.getEnvs()).isPresent()) {
                this.log.trace("Using envs: {}", envs.get());
                containerConfigBuilder.withEnv(envs.get().toArray(new String[0]));
            }
            if ((cmd = dockerContainer.getCmd()).isPresent()) {
                this.log.trace("Using cmd: {}", cmd.get());
                containerConfigBuilder.withCmd(cmd.get().toArray(new String[0]));
            }
            if ((entryPoint = dockerContainer.getEntryPoint()).isPresent()) {
                this.log.trace("Using entryPoint: {}", entryPoint.get());
                containerConfigBuilder.withEntrypoint(entryPoint.get().toArray(new String[0]));
            }
            containerId = containerConfigBuilder.withHostConfig(hostConfigBuilder).exec().getId();
            this.dockerClient.startContainerCmd(containerId).exec();
        }
        return containerId;
    }

    public String execCommandInContainer(String containerId, String ... command) throws DockerException {
        String commandStr = Arrays.toString(command);
        this.log.trace("Running command {} in container {}", (Object)commandStr, (Object)containerId);
        String execId = ((ExecCreateCmdResponse)this.dockerClient.execCreateCmd(containerId).withCmd(command).withAttachStdout(Boolean.valueOf(true)).withAttachStderr(Boolean.valueOf(true)).exec()).getId();
        final StringBuilder output = new StringBuilder();
        this.dockerClient.execStartCmd(execId).exec((ResultCallback)new ResultCallback.Adapter<Frame>(){

            public void onNext(Frame object) {
                output.append(new String(object.getPayload(), StandardCharsets.UTF_8));
                super.onNext((Object)object);
            }
        });
        this.log.trace("Result of command {} in container {}: {}", new Object[]{commandStr, containerId, output});
        return output.toString();
    }

    public String getBindPort(String containerId, String exposed) throws DockerException {
        Ports ports = this.dockerClient.inspectContainerCmd(containerId).exec().getNetworkSettings().getPorts();
        Ports.Binding[] exposedPort = (Ports.Binding[])ports.getBindings().get(ExposedPort.parse((String)exposed));
        this.log.trace("Port list {} -- Exposed port {} = {}", new Object[]{ports, exposed, exposedPort});
        if (ports.getBindings().isEmpty() || exposedPort.length == 0) {
            String dockerImage = this.dockerClient.inspectContainerCmd(containerId).exec().getConfig().getImage();
            throw new SeleniumJupiterException("Port " + exposed + " is not bindable in container " + dockerImage);
        }
        return exposedPort[0].getHostPortSpec();
    }

    public void pullImage(String imageId) throws DockerException, InterruptedException {
        if (!(this.preferences.checkKeyInPreferences(imageId) && this.getConfig().isUsePreferences() && this.localDaemon)) {
            try {
                this.log.info("Pulling Docker image {}", (Object)imageId);
                (this.dockerClient.pullImageCmd(imageId).exec((ResultCallback)new ResultCallback.Adapter<PullResponseItem>(){})).awaitCompletion();
                this.log.trace("Docker image {} downloaded", (Object)imageId);
                if (this.getConfig().isUsePreferences() && this.localDaemon) {
                    this.preferences.putValueInPreferencesIfEmpty(imageId, "pulled");
                }
            }
            catch (Exception e) {
                this.log.warn("Exception pulling image {}: {}", (Object)imageId, (Object)e.getMessage());
            }
        }
    }

    public synchronized void stopAndRemoveContainer(String containerId, String imageId) {
        this.log.info("Stopping Docker container {}", (Object)imageId);
        try {
            this.stopContainer(containerId);
            this.removeContainer(containerId);
        }
        catch (Exception e) {
            this.log.warn("Exception stopping container {}", (Object)imageId, (Object)e);
        }
    }

    public synchronized void stopContainer(String containerId) throws DockerException {
        int stopTimeoutSec = this.getConfig().getDockerStopTimeoutSec();
        if (stopTimeoutSec == 0) {
            this.log.trace("Killing container {}", (Object)containerId);
            this.dockerClient.killContainerCmd(containerId).exec();
        } else {
            this.log.trace("Stopping container {} (timeout {} seconds)", (Object)containerId, (Object)stopTimeoutSec);
            this.dockerClient.stopContainerCmd(containerId).withTimeout(Integer.valueOf(stopTimeoutSec)).exec();
        }
    }

    public synchronized void removeContainer(String containerId) throws DockerException {
        this.log.trace("Removing container {}", (Object)containerId);
        int stopTimeoutSec = this.getConfig().getDockerStopTimeoutSec();
        if (stopTimeoutSec == 0) {
            this.dockerClient.removeContainerCmd(containerId).withForce(Boolean.valueOf(true)).exec();
        } else {
            this.dockerClient.removeContainerCmd(containerId).exec();
        }
    }

    public String getDockerDefaultSocket() {
        return this.dockerDefaultSocket;
    }

    public int getDockerWaitTimeoutSec() {
        return this.dockerWaitTimeoutSec;
    }

    public int getDockerPollTimeMs() {
        return this.dockerPollTimeMs;
    }

    public void close() throws IOException {
        this.dockerClient.close();
    }

    public void updateDockerClient(String url) {
        if (this.localDaemon) {
            this.log.debug("Updating Docker client using URL {}", (Object)url);
            this.dockerClient = this.getDockerClient(url);
            this.localDaemon = false;
        }
    }

    public Config getConfig() {
        return this.config;
    }
}

