package io.github.baharmc.chat

import org.hjson.JsonArray
import org.hjson.JsonObject
import org.hjson.JsonValue
import org.hjson.Stringify
import java.util.*

open class ChatComponent {

    companion object {

        fun create() = ChatComponent()

        fun empty() = "".also { create().text = it }

        fun text(text: String) = create().setText(text)

        fun fromJson(json: JsonObject): ChatComponent {
            val cc = create()
            val text = json["text"]
            if (text != null) {
                cc.text = text.asString()
            }
            val translate = json["translate"]
            if (translate != null) {
                cc.translate = translate.asString()
            }
            val with = json["with"]
            if (with != null) {
                val array = with.asArray()
                var i = 0
                val j = array.size()
                while (i < j) {
                    val el = array[i]
                    if (el.isString || el.isNumber || el.isBoolean) cc.addWith(el.asString()) else if (el.isObject) cc.addWith(
                        fromJson(el.asObject())
                    )
                    i++
                }
            }
            val score = json["score"]
            if (score != null) {
                val scoreArray = score.asObject()
                cc.scoreUsername = scoreArray["name"].asString()
                cc.scoreObjective = scoreArray["objective"].asString()
            }
            val selector = json["selector"]
            if (selector != null) {
                cc.selector = selector.asString()
            }
            val extra = json["extra"]
            if (extra != null) {
                val array = extra.asArray()
                var i = 0
                val j = array.size()
                while (i < j) {
                    val el = array[i]
                    if (el.isString || el.isNumber || el.isBoolean) cc.addExtra(el.asString()) else if (el.isObject) cc.addExtra(
                        fromJson(el.asObject())
                    )
                    i++
                }
            }
            val bold = json["bold"]
            if (bold != null) {
                cc.setBold(bold.asBoolean())
            }
            val italic = json["italic"]
            if (italic != null) {
                cc.setItalic(italic.asBoolean())
            }
            val underlined = json["underlined"]
            if (underlined != null) {
                cc.setUnderlined(underlined.asBoolean())
            }
            val strikethrough = json["strikethrough"]
            if (strikethrough != null) {
                cc.setStrikethrough(strikethrough.asBoolean())
            }
            val obfuscated = json["obfuscated"]
            if (obfuscated != null) {
                cc.setObfuscated(obfuscated.asBoolean())
            }
            val color = json["color"]
            if (color != null) {
                cc.color = ChatColor.valueOf(color.asString().toUpperCase())
            }
            val clickEvent = json["clickEvent"]
            if (clickEvent != null) {
                cc.clickEvent = ClickEvent.fromJson(clickEvent.asObject())
            }
            val hoverEvent = json["hoverEvent"]
            if (hoverEvent != null) {
                cc.hoverEvent = HoverEvent.fromJson(hoverEvent.asObject())
            }
            val insertion = json["insertion"]
            if (insertion != null) {
                cc.insertion = insertion.asString()
            }
            return cc
        }

        fun fromFormat(format: String): ChatComponent {
            val chars = format.toCharArray()
            var currentText = ""
            var currentColor: ChatColor? = null
            val component = create()
            var currentComponent: ChatComponent? = null
            val bold: Boolean
            var italic: Boolean
            var underline: Boolean
            var strikethrough: Boolean
            var obfuscate: Boolean
            bold = false.also { obfuscate = it }.also { strikethrough = it }.also { underline = it }
                .also { italic = it }
            IntRange(0, chars.size).forEach {
                val prevSection = it != 0 && chars[it - 1] == '\u00A7'
                val c = chars[it]
                if (prevSection) {
                    val chatColor = ChatColor.of(c)
                    var curr = currentComponent ?: component
                    if (!currentText.isEmpty()) {
                        curr.setText(currentText).setColor(currentColor)
                        if (currentComponent != null) {
                            component.addExtra(currentComponent!!)
                        }
                        currentComponent = create()
                        curr = currentComponent!!
                        currentText = ""
                    }
                    if (chatColor.isColor()) {
                        currentColor = chatColor
                        // disable all formatting
                        if (bold)
                            curr.setBold(false)
                        if (italic)
                            curr.setItalic(false)
                        if (underline)
                            curr.setUnderlined(false)
                        if (strikethrough)
                            curr.setStrikethrough(false)
                        if (obfuscate)
                            curr.setObfuscated(false)
                    } else {
                        when (chatColor) {
                            ChatColor.BOLD -> {
                                if (!bold) {
                                    curr.setBold(true)
                                }
                            }
                            ChatColor.ITALIC -> {
                                if (!italic) {
                                    curr.setItalic(true)
                                }
                            }
                            ChatColor.UNDERLINE -> {
                                if (!underline) {
                                    curr.setUnderlined(true)
                                }
                            }
                            ChatColor.STRIKETHROUGH -> {
                                if (!strikethrough) {
                                    curr.setStrikethrough(true)
                                }
                            }
                            ChatColor.OBFUSCATED -> {
                                if (!obfuscate) {
                                    curr.setObfuscated(true)
                                }
                            }
                            ChatColor.RESET -> {
                                currentColor = null
                                // disable all formatting
                                if (bold) {
                                    curr.setBold(false)
                                }
                                if (italic) {
                                    curr.setItalic(false)
                                }
                                if (underline) {
                                    curr.setUnderlined(false)
                                }
                                if (strikethrough) {
                                    curr.setStrikethrough(false)
                                }
                                if (obfuscate) {
                                    curr.setObfuscated(false)
                                }
                            }
                            else -> {
                            }
                        }
                    }
                } else if (c != '\u00A7') {
                    currentText += c
                }
            }
            if (currentText.isNotEmpty()) {
                currentComponent?.setText(currentText)?.setColor(currentColor)
                    .let { it?.let { component.addExtra(it) } }
            }
            return component
        }

    }

    private var text: String? = null

    private var translate: String? = null

    private var selector: String? = null

    private var insertion: String? = null

    private var scoreUsername: String? = null

    private var scoreObjective: String? = null

    private var color: ChatColor? = null

    private var clickEvent: ClickEvent? = null

    private var hoverEvent: HoverEvent? = null

    private val with = ArrayList<ChatComponent>()

    private val extra = ArrayList<ChatComponent>()

    var bold = false

    var italic = false

    var underlined = false

    var strikethrough = false

    var obfuscated = false

    fun getWith(): List<ChatComponent> = Collections.unmodifiableList(with)

    fun getExtra(): List<ChatComponent> = Collections.unmodifiableList(extra)

    fun addWith(component: ChatComponent) =
        if (!this.hasWith(component, true)) {
            with.add(component)
            this
        } else {
            this
        }

    fun addWith(string: String) = this.addWith(StringChatComponent(string))

    fun hasWith(component: ChatComponent, recursive: Boolean): Boolean =
        if (component === this || with.contains(component)) {
            true
        } else if (!recursive) {
            false
        } else {
            with.stream().anyMatch { it.hasWith(component, true) }
        }

    fun addExtra(component: ChatComponent) =
        if (!hasExtra(component, true)) {
            extra.add(component)
            this
        } else {
            this
        }

    fun addExtra(string: String) = addExtra(StringChatComponent(string))

    fun hasExtra(component: ChatComponent, recursive: Boolean): Boolean =
        if (component === this || extra.contains(component)) {
            true
        } else if (!recursive) {
            false
        } else {
            extra.stream().anyMatch { it.hasExtra(component, true) }
        }

    fun setBold(bold: Boolean): ChatComponent {
        this.bold = bold
        return this
    }

    fun setItalic(italic: Boolean): ChatComponent {
        this.italic = italic
        return this
    }

    fun setUnderlined(underlined: Boolean): ChatComponent {
        this.underlined = underlined
        return this
    }

    fun setStrikethrough(strikethrough: Boolean): ChatComponent {
        this.strikethrough = strikethrough
        return this
    }

    fun setObfuscated(obfuscated: Boolean): ChatComponent {
        this.obfuscated = obfuscated
        return this
    }

    fun setText(text: String?): ChatComponent {
        this.text = text
        return this
    }

    fun setTranslate(translate: String?): ChatComponent {
        this.translate = translate
        return this
    }

    fun setSelector(selector: String?): ChatComponent {
        this.selector = selector
        return this
    }

    fun setInsertion(insertion: String?): ChatComponent {
        this.insertion = insertion
        return this
    }

    fun setScoreUsername(scoreUsername: String?): ChatComponent {
        this.scoreUsername = scoreUsername
        return this
    }

    fun setScoreObjective(scoreObjective: String?): ChatComponent {
        this.scoreObjective = scoreObjective
        return this
    }

    fun setColor(color: ChatColor?): ChatComponent {
        this.color = color
        return this
    }

    fun setClickEvent(clickEvent: ClickEvent?): ChatComponent {
        this.clickEvent = clickEvent
        return this
    }

    fun setHoverEvent(hoverEvent: HoverEvent?): ChatComponent {
        this.hoverEvent = hoverEvent
        return this
    }

    open fun stripColor(): JsonValue {
        val json = JsonObject()
        if (text != null) {
            json.add("text", text!!)
        }
        if (translate != null) {
            val array = JsonArray()
            json.add("translate", translate!!)
            with.forEach { array.add(it.stripColor()) }
            json.add("with", array)
        }
        if (scoreUsername != null && scoreObjective != null) {
            val score = JsonObject()
            score.add("name", scoreUsername!!)
            score.add("objective", scoreObjective!!)
            json.add("score", score)
        }
        if (selector != null) {
            json.add("selector", selector!!)
        }
        if (extra.isNotEmpty()) {
            val extraArray = JsonArray()
            extra.stream()
                .map { it.stripColor() }
                .forEach { extraArray.add(it) }
            json.add("extra", extraArray)
        }
        if (clickEvent != null) {
            json.add("clickEvent", clickEvent!!.asJson())
        }
        if (hoverEvent != null) {
            json.add("hoverEvent", hoverEvent!!.asJson())
        }
        if (insertion != null) {
            json.add("insertion", insertion!!)
        }
        return json
    }

    open fun asJson(): JsonValue {
        val json = JsonObject()
        if (text != null) {
            json.add("text", text!!)
        }
        if (translate != null) {
            val array = JsonArray()
            json.add("translate", translate!!)
            with.forEach { array.add(it.asJson()) }
            json.add("with", array)
        }
        if (scoreUsername != null && scoreObjective != null) {
            val score = JsonObject()
            score.add("name", scoreUsername!!)
            score.add("objective", scoreObjective!!)
            json.add("score", score)
        }
        if (selector != null) {
            json.add("selector", selector!!)
        }
        if (extra.isNotEmpty()) {
            val extraArray = JsonArray()
            extra.forEach { extraArray.add(it.asJson()) }
            json.add("extra", extraArray)
        }
        json.add("bold", bold)
        json.add("italic", italic)
        json.add("underlined", underlined)
        json.add("strikethrough", strikethrough)
        json.add("obfuscated", obfuscated)
        if (color != null && !color!!.isFormat()) {
            json.add("color", color!!.name.toLowerCase())
        }
        if (clickEvent != null) {
            json.add("clickEvent", clickEvent!!.asJson())
        }
        if (hoverEvent != null) {
            json.add("hoverEvent", hoverEvent!!.asJson())
        }
        if (insertion != null) {
            json.add("insertion", insertion!!)
        }
        return json
    }

    override fun toString(): String {
        return asJson().toString(Stringify.PLAIN)
    }

    override fun equals(other: Any?) =
        if (other == null || other === this || other !is ChatComponent) {
            other === this
        } else {
            asJson() == other.asJson()
        }

    override fun hashCode() = asJson().hashCode()

    private class StringChatComponent(val string: String) : ChatComponent() {

        override fun stripColor(): JsonValue {
            return asJson()
        }

        override fun asJson(): JsonValue {
            return JsonValue.valueOf(string)
        }
    }

}