package io.ahmer.utils.subutil;

import static android.Manifest.permission.ACCESS_COARSE_LOCATION;
import static android.Manifest.permission.ACCESS_FINE_LOCATION;

import android.content.Context;
import android.content.Intent;
import android.location.Address;
import android.location.Criteria;
import android.location.Geocoder;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.location.LocationProvider;
import android.os.Bundle;
import android.provider.Settings;
import android.util.Log;

import androidx.annotation.RequiresPermission;

import java.io.IOException;
import java.util.List;
import java.util.Locale;

import io.ahmer.utils.utilcode.Utils;

public final class LocationUtils {

    private static final int TWO_MINUTES = 1000 * 60 * 2;

    private static OnLocationChangeListener mListener;
    private static MyLocationListener myLocationListener;
    private static LocationManager mLocationManager;

    private LocationUtils() {
        throw new UnsupportedOperationException("You can't instantiate me...");
    }


//    /**
//     * you have to check for Location Permission before use this method
//     * add this code <uses-permission android:name="android.permission.ACCESS_FINE_LOCATION" /> to your Manifest file.
//     * you have also implement LocationListener and passed it to the method.
//     *
//     * @param Context
//     * @param LocationListener
//     * @return {@code Location}
//     */
//
//    @SuppressLint("MissingPermission")
//    public static Location getLocation(Context context, LocationListener listener) {
//        Location location = null;
//        try {
//            mLocationManager = (LocationManager) context.getSystemService(LOCATION_SERVICE);
//            if (!isLocationEnabled()) {
//                //no Network and GPS providers is enabled
//                Toast.makeText(context
//                        , " you have to open GPS or INTERNET"
//                        , Toast.LENGTH_LONG)
//                        .show();
//            } else {
//                if (isLocationEnabled()) {
//                    mLocationManager.requestLocationUpdates(
//                            LocationManager.NETWORK_PROVIDER,
//                            MIN_TIME_BETWEEN_UPDATES,
//                            MIN_DISTANCE_CHANGE_FOR_UPDATES,
//                            listener);
//
//                    if (mLocationManager != null) {
//                        location = mLocationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER);
//                        if (location != null) {
//                            mLocationManager.removeUpdates(listener);
//                            return location;
//                        }
//                    }
//                }
//                //when GPS is enabled.
//                if (isGpsEnabled()) {
//                    if (location == null) {
//                        mLocationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER,
//                                MIN_TIME_BETWEEN_UPDATES,
//                                MIN_DISTANCE_CHANGE_FOR_UPDATES,
//                                listener);
//
//                        if (mLocationManager != null) {
//                            location =
//                                    mLocationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER);
//                            if (location != null) {
//                                mLocationManager.removeUpdates(listener);
//                                return location;
//                            }
//                        }
//                    }
//                }
//
//            }
//        } catch (Exception e) {
//            e.printStackTrace();
//        }
//
//        return location;
//    }

    /**
     * Determine if GPS is available
     *
     * @return {@code true}: 是<br>{@code false}: 否
     */
    public static boolean isGpsEnabled() {
        LocationManager lm = (LocationManager) Utils.getApp().getSystemService(Context.LOCATION_SERVICE);
        return lm.isProviderEnabled(LocationManager.GPS_PROVIDER);
    }

    /**
     * Determine if positioning is available
     *
     * @return {@code true}: yes<br>{@code false}: no
     */
    public static boolean isLocationEnabled() {
        LocationManager lm = (LocationManager) Utils.getApp().getSystemService(Context.LOCATION_SERVICE);
        return lm.isProviderEnabled(LocationManager.NETWORK_PROVIDER)
                || lm.isProviderEnabled(LocationManager.GPS_PROVIDER);
    }

    /**
     * Open the GPS settings interface
     */
    public static void openGpsSettings() {
        Intent intent = new Intent(Settings.ACTION_LOCATION_SOURCE_SETTINGS);
        Utils.getApp().startActivity(intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK));
    }

    /**
     * register
     * <p>Remember to call {@link #unregister()} after use</p>
     * <p>Need to add permission {@code <uses-permission android:name="android.permission.INTERNET" />}</p>
     * <p>Permission required {@code <uses-permission android:name="android.permission.ACCESS_COARSE_LOCATION" />}</p>
     * <p>Add permission {@code <uses-permission android:name="android.permission.ACCESS_FINE_LOCATION" />}</p>
     * <p>If {@code minDistance} is 0, it will be updated regularly by {@code minTime};</p>
     * <p>{@code minDistance} is not 0, then {@code minDistance} shall prevail;</p>
     * <p>Both are 0, refresh at any time. </p>
     *
     * @param minTime     location information update period (unit: milliseconds)
     * @param minDistance position change minimum distance: when the position distance change exceeds this value, the position information will be updated (unit: meters)
     * @param listener    position refresh callback interface
     * @return {@code true}: Initialization succeeded<br>{@code false}: Initialization failed
     */
    @RequiresPermission(ACCESS_FINE_LOCATION)
    public static boolean register(long minTime, long minDistance, OnLocationChangeListener listener) {
        if (listener == null) return false;
        mLocationManager = (LocationManager) Utils.getApp().getSystemService(Context.LOCATION_SERVICE);
        if (!mLocationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)
                && !mLocationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
            Log.d("LocationUtils", "Unable to locate, please turn on location services");
            return false;
        }
        mListener = listener;
        String provider = mLocationManager.getBestProvider(getCriteria(), true);
        Location location = mLocationManager.getLastKnownLocation(provider);
        if (location != null) listener.getLastKnownLocation(location);
        if (myLocationListener == null) myLocationListener = new MyLocationListener();
        mLocationManager.requestLocationUpdates(provider, minTime, minDistance, myLocationListener);
        return true;
    }

    /**
     * Log out
     */
    @RequiresPermission(ACCESS_COARSE_LOCATION)
    public static void unregister() {
        if (mLocationManager != null) {
            if (myLocationListener != null) {
                mLocationManager.removeUpdates(myLocationListener);
                myLocationListener = null;
            }
            mLocationManager = null;
        }
        if (mListener != null) {
            mListener = null;
        }
    }

    /**
     * Set targeting parameters
     *
     * @return {@link Criteria}
     */
    private static Criteria getCriteria() {
        Criteria criteria = new Criteria();
        // Set the positioning accuracy Criteria.ACCURACY_COARSE is relatively rough, Criteria.ACCURACY_FINE is relatively fine
        criteria.setAccuracy(Criteria.ACCURACY_FINE);
        // Set whether to require speed
        criteria.setSpeedRequired(false);
        // Set whether to allow the operator to charge
        criteria.setCostAllowed(false);
        // Set whether to require orientation information
        criteria.setBearingRequired(false);
        // Set whether altitude information is required
        criteria.setAltitudeRequired(false);
        // set the power requirements
        criteria.setPowerRequirement(Criteria.POWER_LOW);
        return criteria;
    }

    /**
     * Get geographic location based on latitude and longitude
     *
     * @param latitude  latitude
     * @param longitude longitude
     * @return {@link Address}
     */
    public static Address getAddress(double latitude, double longitude) {
        Geocoder geocoder = new Geocoder(Utils.getApp(), Locale.getDefault());
        try {
            List<Address> addresses = geocoder.getFromLocation(latitude, longitude, 1);
            if (addresses.size() > 0) return addresses.get(0);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Get the country based on latitude and longitude
     *
     * @param latitude  latitude
     * @param longitude longitude
     * @return country
     */
    public static String getCountryName(double latitude, double longitude) {
        Address address = getAddress(latitude, longitude);
        return address == null ? "unknown" : address.getCountryName();
    }

    /**
     * Get the location based on latitude and longitude
     *
     * @param latitude  latitude
     * @param longitude longitude
     * @return location
     */
    public static String getLocality(double latitude, double longitude) {
        Address address = getAddress(latitude, longitude);
        return address == null ? "unknown" : address.getLocality();
    }

    /**
     * Get the street where you are based on latitude and longitude
     *
     * @param latitude  latitude
     * @param longitude longitude
     * @return the street
     */
    public static String getStreet(double latitude, double longitude) {
        Address address = getAddress(latitude, longitude);
        return address == null ? "unknown" : address.getAddressLine(0);
    }

    /**
     * Is there a better location
     *
     * @param newLocation         The new Location that you want to evaluate
     * @param currentBestLocation The current Location fix, to which you want to compare the new one
     * @return {@code true}: yes<br>{@code false}: no
     */
    public static boolean isBetterLocation(Location newLocation, Location currentBestLocation) {
        if (currentBestLocation == null) {
            // A new location is always better than no location
            return true;
        }

        // Check whether the new location fix is newer or older
        long timeDelta = newLocation.getTime() - currentBestLocation.getTime();
        boolean isSignificantlyNewer = timeDelta > TWO_MINUTES;
        boolean isSignificantlyOlder = timeDelta < -TWO_MINUTES;
        boolean isNewer = timeDelta > 0;

        // If it's been more than two minutes since the current location, use the new location
        // because the user has likely moved
        if (isSignificantlyNewer) {
            return true;
            // If the new location is more than two minutes older, it must be worse
        } else if (isSignificantlyOlder) {
            return false;
        }

        // Check whether the new location fix is more or less accurate
        int accuracyDelta = (int) (newLocation.getAccuracy() - currentBestLocation.getAccuracy());
        boolean isLessAccurate = accuracyDelta > 0;
        boolean isMoreAccurate = accuracyDelta < 0;
        boolean isSignificantlyLessAccurate = accuracyDelta > 200;

        // Check if the old and new location are from the same provider
        boolean isFromSameProvider = isSameProvider(newLocation.getProvider(), currentBestLocation.getProvider());

        // Determine location quality using a combination of timeliness and accuracy
        if (isMoreAccurate) {
            return true;
        } else if (isNewer && !isLessAccurate) {
            return true;
        } else if (isNewer && !isSignificantlyLessAccurate && isFromSameProvider) {
            return true;
        }
        return false;
    }

    /**
     * Whether the same provider
     *
     * @param provider0 provider 1
     * @param provider1 provider2
     * @return {@code true}: yes<br>{@code false}: no
     */
    public static boolean isSameProvider(String provider0, String provider1) {
        if (provider0 == null) {
            return provider1 == null;
        }
        return provider0.equals(provider1);
    }

    public interface OnLocationChangeListener {

        /**
         * Get the last reserved coordinates
         *
         * @param location coordinates
         */
        void getLastKnownLocation(Location location);

        /**
         * This function is triggered when the coordinates change, if the Provider passes the same coordinates, it will not be triggered
         *
         * @param location coordinates
         */
        void onLocationChanged(Location location);

        /**
         * The provider triggers this function when the three states of available, temporarily unavailable and no service are directly switched
         *
         * @param provider provider
         * @param status   status
         * @param extras   provider optional package
         */
        void onStatusChanged(String provider, int status, Bundle extras);// position state changed
    }

    private static class MyLocationListener
            implements LocationListener {
        /**
         * This function is triggered when the coordinates change, if the Provider passes the same coordinates, it will not be triggered
         *
         * @param location coordinates
         */
        @Override
        public void onLocationChanged(Location location) {
            if (mListener != null) {
                mListener.onLocationChanged(location);
            }
        }

        /**
         * The provider triggers this function when the three states of available, temporarily unavailable and no service are directly switched
         *
         * @param provider provider
         * @param status   status
         * @param extras   provider optional package
         */
        @Override
        public void onStatusChanged(String provider, int status, Bundle extras) {
            if (mListener != null) {
                mListener.onStatusChanged(provider, status, extras);
            }
            switch (status) {
                case LocationProvider.AVAILABLE:
                    Log.d("LocationUtils", "The current GPS status is visible");
                    break;
                case LocationProvider.OUT_OF_SERVICE:
                    Log.d("LocationUtils", "The current GPS status is out of service area");
                    break;
                case LocationProvider.TEMPORARILY_UNAVAILABLE:
                    Log.d("LocationUtils", "The current GPS status is suspended service status");
                    break;
            }
        }

        /**
         * This function is triggered when the provider is enabled, such as when GPS is turned on
         */
        @Override
        public void onProviderEnabled(String provider) {
        }

        /**
         * This function is triggered when the provider is disabled, such as GPS is turned off
         */
        @Override
        public void onProviderDisabled(String provider) {
        }
    }
}
