package io.ahmer.utils.subutil;

import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.net.Uri;
import android.util.Log;

import io.ahmer.utils.utilcode.AppUtils;
import io.ahmer.utils.utilcode.RomUtils;
import io.ahmer.utils.utilcode.Utils;

import java.util.List;

public final class AppStoreUtils {

    private static final String TAG = "AppStoreUtils";

    private static final String GOOGLE_PLAY_APP_STORE_PACKAGE_NAME = "com.android.vending";

    /**
     * Get the Intent to jump to the app store
     *
     * @return Jump to the Intent of the app store
     */
    public static Intent getAppStoreIntent() {
        return getAppStoreIntent(Utils.getApp().getPackageName(), false);
    }

    /**
     * Get the Intent to jump to the app store
     *
     * @param isIncludeGooglePlayStore whether to include Google Play Store
     * @return Jump to the Intent of the app store
     */
    public static Intent getAppStoreIntent(boolean isIncludeGooglePlayStore) {
        return getAppStoreIntent(Utils.getApp().getPackageName(), isIncludeGooglePlayStore);
    }

    /**
     * Get the Intent to jump to the app store
     *
     * @param packageName package name
     * @return Jump to the Intent of the app store
     */
    public static Intent getAppStoreIntent(final String packageName) {
        return getAppStoreIntent(packageName, false);
    }

    /**
     * Get the Intent to jump to the app store
     * <p>Priority jump to the mobile app market</p>
     *
     * @param packageName              package name
     * @param isIncludeGooglePlayStore whether to include Google Play Store
     * @return Jump to the Intent of the app store
     */
    public static Intent getAppStoreIntent(final String packageName, boolean isIncludeGooglePlayStore) {
        if (RomUtils.isSamsung()) {// Samsung handles jump to Samsung market separately
            Intent samsungAppStoreIntent = getSamsungAppStoreIntent(packageName);
            if (samsungAppStoreIntent != null) return samsungAppStoreIntent;
        }
        if (RomUtils.isLeeco()) {// LeTV handles jumping to LeTV market separately
            Intent leecoAppStoreIntent = getLeecoAppStoreIntent(packageName);
            if (leecoAppStoreIntent != null) return leecoAppStoreIntent;
        }

        Uri uri = Uri.parse("market://details?id=" + packageName);
        Intent intent = new Intent();
        intent.setData(uri);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        List<ResolveInfo> resolveInfos = Utils.getApp().getPackageManager()
                .queryIntentActivities(intent, PackageManager.MATCH_DEFAULT_ONLY);
        if (resolveInfos == null || resolveInfos.size() == 0) {
            Log.e(TAG, "No app store!");
            return null;
        }
        Intent googleIntent = null;
        for (ResolveInfo resolveInfo : resolveInfos) {
            String pkgName = resolveInfo.activityInfo.packageName;
            if (!GOOGLE_PLAY_APP_STORE_PACKAGE_NAME.equals(pkgName)) {
                if (AppUtils.isAppSystem(pkgName)) {
                    intent.setPackage(pkgName);
                    return intent;
                }
            } else {
                intent.setPackage(GOOGLE_PLAY_APP_STORE_PACKAGE_NAME);
                googleIntent = intent;
            }
        }
        if (isIncludeGooglePlayStore && googleIntent != null) {
            return googleIntent;
        }

        intent.setPackage(resolveInfos.get(0).activityInfo.packageName);
        return intent;
    }

    private static Intent getSamsungAppStoreIntent(final String packageName) {
        Intent intent = new Intent();
        intent.setClassName("com.sec.android.app.samsungapps", "com.sec.android.app.samsungapps.Main");
        intent.setData(Uri.parse("http://www.samsungapps.com/appquery/appDetail.as?appId=" + packageName));
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        if (getAvailableIntentSize(intent) > 0) {
            return intent;
        }
        return null;
    }

    private static Intent getLeecoAppStoreIntent(final String packageName) {
        Intent intent = new Intent();
        intent.setClassName("com.letv.app.appstore", "com.letv.app.appstore.appmodule.details.DetailsActivity");
        intent.setAction("com.letv.app.appstore.appdetailactivity");
        intent.putExtra("packageName", packageName);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        if (getAvailableIntentSize(intent) > 0) {
            return intent;
        }
        return null;
    }

    private static int getAvailableIntentSize(final Intent intent) {
        return Utils.getApp().getPackageManager()
                .queryIntentActivities(intent, PackageManager.MATCH_DEFAULT_ONLY)
                .size();
    }
}
