package io.ahmer.utils;

import android.content.Context;
import android.util.AttributeSet;

import androidx.appcompat.widget.AppCompatTextView;

import io.ahmer.utils.async.AsyncTask;
import io.ahmer.utils.utilcode.ThreadUtils;


public class VerticalMarqueeTextView extends AppCompatTextView {

    private boolean isPaused;
    private boolean isUserScrolling;
    private boolean stop;
    private int duration;
    private int pixelYOffSet;

    /**
     * Creates a vertically auto scrolling marquee of a TextView within an
     * Activity. The (long) duration in milliseconds between calls to the next
     * scrollBy(0, pixelYOffSet). Defaults to 65L. The (int) amount of Y pixels
     * to scroll by defaults to 1.
     */
    public VerticalMarqueeTextView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init();
    }

    public VerticalMarqueeTextView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public VerticalMarqueeTextView(Context context) {
        super(context);
        init();
    }

    public int getDuration() {
        return duration;
    }

    public void setDuration(int duration) {
        if (duration <= 0) {
            this.duration = 65;
        } else {
            this.duration = duration;
        }
    }

    public int getPixelYOffSet() {
        return pixelYOffSet;
    }

    public void setPixelYOffSet(int pixelYOffSet) {
        this.pixelYOffSet = Math.max(pixelYOffSet, 1);
    }

    private void startMarquee() {
        new AutoScrollTextView(VerticalMarqueeTextView.this).execute(null);
    }

    public void stopMarquee() {
        stop = true;
    }

    public void pauseMarquee() {
        isPaused = true;
    }

    public void resumeMarquee() {
        isPaused = false;
    }

    public boolean isPaused() {
        return isPaused || isUserScrolling;
    }

    /**
     * Initialize fields and start the marquee.
     * Defaults to 65L, But I need fast speed so I changed 65L to 25L.
     */
    private void init() {
        setDuration(65);
        setPixelYOffSet(1);
        isUserScrolling = isPaused = stop = false;
        startMarquee();
    }

    private static class AutoScrollTextView extends AsyncTask<Void, Void, Void> {

        private final VerticalMarqueeTextView vMTV;
        private boolean isNotDrawn;
        private int pixelCount;

        public AutoScrollTextView(VerticalMarqueeTextView vMTV) {
            this.vMTV = vMTV;
        }

        @Override
        protected void onPreExecute() {
            super.onPreExecute();
            isNotDrawn = true;
        }

        @Override
        protected Void doInBackground(Void aVoid) throws Exception {
            while (textViewNotDrawn()) {
                Thread.sleep(1000);
            }
            while (!vMTV.stop) {
                if (!(vMTV.isPressed() && vMTV.isUserScrolling && !vMTV.isPaused)) {
                    vMTV.isUserScrolling = false;
                }
                while (!vMTV.isUserScrolling && !vMTV.stop && !vMTV.isPaused) {
                    Thread.sleep(vMTV.duration);
                    ThreadUtils.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            if (vMTV.isPressed()) {
                                vMTV.isUserScrolling = true;
                            } else {
                                if (vMTV.getScrollY() >= pixelCount) {
                                    vMTV.scrollTo(0, 0);
                                } else {
                                    vMTV.scrollBy(0, vMTV.pixelYOffSet);
                                }
                                vMTV.invalidate();
                            }
                        }
                    });
                }
            }
            return null;
        }

        @Override
        protected void onPostExecute(Void aVoid) {
            super.onPostExecute(aVoid);
        }

        @Override
        protected void onBackgroundError(Exception e) {
            e.printStackTrace();
        }

        private boolean textViewNotDrawn() {
            ThreadUtils.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (vMTV.getLineCount() > 0) {
                        pixelCount = vMTV.getLineHeight() * vMTV.getLineCount();
                        isNotDrawn = false;
                    }
                }
            });
            return isNotDrawn;
        }
    }
}