/*
 * Decompiled with CFR 0.152.
 */
package io.etcd.jetcd.launcher;

import com.github.dockerjava.api.DockerClient;
import io.etcd.jetcd.launcher.EtcdCluster;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testcontainers.DockerClientFactory;
import org.testcontainers.containers.BindMode;
import org.testcontainers.containers.ContainerLaunchException;
import org.testcontainers.containers.FixedHostPortGenericContainer;
import org.testcontainers.containers.Network;
import org.testcontainers.containers.SelinuxContext;
import org.testcontainers.containers.output.OutputFrame;
import org.testcontainers.containers.output.WaitingConsumer;
import org.testcontainers.containers.wait.strategy.AbstractWaitStrategy;
import org.testcontainers.containers.wait.strategy.WaitStrategy;
import org.testcontainers.utility.LogUtils;

public class EtcdContainer
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(EtcdCluster.class);
    private static final int ETCD_CLIENT_PORT = 2379;
    private static final int ETCD_PEER_PORT = 2380;
    private static final String ETCD_DATA_DIR = "/data.etcd";
    public static final String ETCD_DOCKER_IMAGE_NAME = "gcr.io/etcd-development/etcd:v3.3";
    private final String endpoint;
    private final boolean ssl;
    private final FixedHostPortGenericContainer<?> container;
    private final LifecycleListener listener;
    private final Path dataDirectory;

    public EtcdContainer(Network network, LifecycleListener listener, boolean ssl, String clusterName, String endpoint, List<String> endpoints, boolean restartable) {
        this(network, listener, ssl, clusterName, endpoint, endpoints, restartable, ETCD_DOCKER_IMAGE_NAME, Collections.emptyList());
    }

    public EtcdContainer(Network network, LifecycleListener listener, boolean ssl, String clusterName, String endpoint, List<String> endpoints, boolean restartable, String ... additionalArgs) {
        this(network, listener, ssl, clusterName, endpoint, endpoints, restartable, ETCD_DOCKER_IMAGE_NAME, Arrays.asList(additionalArgs));
    }

    public EtcdContainer(Network network, LifecycleListener listener, boolean ssl, String clusterName, String endpoint, List<String> endpoints, boolean restartable, String image, List<String> additionalArgs) {
        this.endpoint = endpoint;
        this.ssl = ssl;
        this.listener = listener;
        String name = endpoint;
        ArrayList<String> command = new ArrayList<String>();
        this.container = new FixedHostPortGenericContainer(image);
        this.container.withExposedPorts(new Integer[]{2379, 2380});
        this.container.withNetwork(network);
        this.container.withNetworkAliases(new String[]{name});
        this.container.waitingFor(this.waitStrategy());
        this.container.withLogConsumer(this.logConsumer());
        command.add("etcd");
        command.add("--name");
        command.add(name);
        command.add("--advertise-client-urls");
        command.add((ssl ? "https" : "http") + "://0.0.0.0:" + 2379);
        command.add("--listen-client-urls");
        command.add((ssl ? "https" : "http") + "://0.0.0.0:" + 2379);
        if (restartable) {
            this.dataDirectory = EtcdContainer.createDataDirectory(name);
            this.container.addFileSystemBind(this.dataDirectory.toString(), ETCD_DATA_DIR, BindMode.READ_WRITE, SelinuxContext.SHARED);
            command.add("--data-dir");
            command.add(ETCD_DATA_DIR);
        } else {
            this.dataDirectory = null;
        }
        if (ssl) {
            this.container.withClasspathResourceMapping("ssl/cert/" + name + ".pem", "/etc/ssl/etcd/server.pem", BindMode.READ_ONLY, SelinuxContext.SHARED);
            this.container.withClasspathResourceMapping("ssl/cert/" + name + "-key.pem", "/etc/ssl/etcd/server-key.pem", BindMode.READ_ONLY, SelinuxContext.SHARED);
            command.add("--cert-file");
            command.add("/etc/ssl/etcd/server.pem");
            command.add("--key-file");
            command.add("/etc/ssl/etcd/server-key.pem");
        }
        if (endpoints.size() > 1) {
            command.add("--initial-advertise-peer-urls");
            command.add("http://" + name + ":" + 2380);
            command.add("--listen-peer-urls");
            command.add("http://0.0.0.0:2380");
            command.add("--initial-cluster-token");
            command.add(clusterName);
            command.add("--initial-cluster");
            command.add(endpoints.stream().map(e -> e + "=http://" + e + ":" + 2380).collect(Collectors.joining(",")));
            command.add("--initial-cluster-state");
            command.add("new");
        }
        command.addAll(additionalArgs);
        if (!command.isEmpty()) {
            this.container.withCommand(command.toArray(new String[command.size()]));
        }
    }

    public void start() {
        LOGGER.debug("starting etcd container {} with command: {}", (Object)this.endpoint, (Object)String.join((CharSequence)" ", this.container.getCommandParts()));
        try {
            this.container.start();
            this.listener.started(this);
            if (this.dataDirectory != null) {
                this.setDataDirectoryPermissions("o+rwx");
            }
        }
        catch (Exception exception) {
            this.listener.failedToStart(this, exception);
        }
    }

    public void restart() {
        if (this.dataDirectory == null) {
            throw new IllegalStateException("Container not restartable, please create it with restartable=true");
        }
        LOGGER.debug("restarting etcd container {} with command: {}", (Object)this.endpoint, (Object)String.join((CharSequence)" ", this.container.getCommandParts()));
        int port = this.container.getMappedPort(2379);
        this.container.stop();
        this.container.withExposedPorts(new Integer[]{2380});
        this.container.withFixedExposedPort(port, 2379);
        this.container.start();
    }

    @Override
    public void close() {
        if (this.container != null) {
            this.container.stop();
        }
        if (this.dataDirectory != null && Files.exists(this.dataDirectory, new LinkOption[0])) {
            EtcdContainer.deleteDataDirectory(this.dataDirectory);
        }
    }

    public URI clientEndpoint() {
        String host = this.container.getContainerIpAddress();
        int port = this.container.getMappedPort(2379);
        return this.newURI(host, port);
    }

    public URI peerEndpoint() {
        String host = this.container.getContainerIpAddress();
        int port = this.container.getMappedPort(2380);
        return this.newURI(host, port);
    }

    private URI newURI(String host, int port) {
        try {
            return new URI(this.ssl ? "https" : "http", null, host, port, null, null, null);
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException("URISyntaxException should never happen here", e);
        }
    }

    private WaitStrategy waitStrategy() {
        return new AbstractWaitStrategy(){

            protected void waitUntilReady() {
                DockerClient client = DockerClientFactory.instance().client();
                WaitingConsumer waitingConsumer = new WaitingConsumer();
                LogUtils.followOutput((DockerClient)client, (String)this.waitStrategyTarget.getContainerId(), (Consumer)waitingConsumer);
                try {
                    waitingConsumer.waitUntil(f -> f.getUtf8String().contains("ready to serve client requests"), this.startupTimeout.getSeconds(), TimeUnit.SECONDS, 1);
                }
                catch (TimeoutException e) {
                    throw new ContainerLaunchException("Timed out");
                }
            }
        };
    }

    private Consumer<OutputFrame> logConsumer() {
        Logger logger = LoggerFactory.getLogger(EtcdContainer.class);
        return outputFrame -> {
            OutputFrame.OutputType outputType = outputFrame.getType();
            String utf8String = outputFrame.getUtf8String().replaceAll("((\\r?\\n)|(\\r))$", "");
            switch (outputType) {
                case END: {
                    break;
                }
                case STDOUT: 
                case STDERR: {
                    logger.debug("{}{}: {}", new Object[]{this.endpoint, outputType, utf8String});
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unexpected outputType " + outputType);
                }
            }
        };
    }

    private void setDataDirectoryPermissions(String permissions) {
        try {
            this.container.execInContainer(new String[]{"chmod", permissions, "-R", ETCD_DATA_DIR});
        }
        catch (Exception e) {
            throw new ContainerLaunchException("Error changing permission to etcd data directory", (Throwable)e);
        }
    }

    private static Path createDataDirectory(String name) {
        try {
            Path path = Files.createTempDirectory("jetcd_test_" + name + "_", new FileAttribute[0]);
            return path.toRealPath(new LinkOption[0]);
        }
        catch (IOException e) {
            throw new ContainerLaunchException("Error creating data directory", (Throwable)e);
        }
    }

    private static void deleteDataDirectory(Path dir) {
        try (Stream<Path> stream = Files.walk(dir, new FileVisitOption[0]);){
            stream.sorted(Comparator.reverseOrder()).forEach(p -> p.toFile().delete());
        }
        catch (IOException e) {
            LOGGER.error("Error deleting directory " + dir.toString(), (Throwable)e);
        }
    }

    static interface LifecycleListener {
        public void started(EtcdContainer var1);

        public void failedToStart(EtcdContainer var1, Exception var2);

        public void stopped(EtcdContainer var1);
    }
}

