/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.storage.jdbc;

import io.debezium.storage.jdbc.offset.JdbcOffsetBackingStore;
import io.debezium.util.DelayStrategy;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.time.Duration;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetriableConnection
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(JdbcOffsetBackingStore.class);
    private final String url;
    private final String user;
    private final String pwd;
    private final Duration waitRetryDelay;
    private final int maxRetryCount;
    private Connection conn;

    public RetriableConnection(String url, String user, String pwd, Duration waitRetryDelay, int maxRetryCount) throws SQLException {
        this.url = url;
        this.user = user;
        this.pwd = pwd;
        this.waitRetryDelay = waitRetryDelay;
        this.maxRetryCount = maxRetryCount;
        try {
            this.createConnection();
        }
        catch (SQLException e) {
            LOGGER.error("Unable to create connection. It will be re-attempted during its first use: {}", (Object)e.getMessage(), (Object)e);
            this.close();
        }
    }

    private void createConnection() throws SQLException {
        this.conn = DriverManager.getConnection(this.url, this.user, this.pwd);
        this.conn.setAutoCommit(false);
    }

    @Override
    public void close() throws SQLException {
        if (this.isOpen()) {
            try {
                this.conn.close();
            }
            catch (Exception e) {
                LOGGER.warn("Exception while closing connection", (Throwable)e);
            }
        }
        this.conn = null;
    }

    public boolean isOpen() {
        try {
            return this.conn != null && !this.conn.isClosed();
        }
        catch (SQLException e) {
            LOGGER.warn("Exception while checking connection", (Throwable)e);
            this.conn = null;
            return false;
        }
    }

    public synchronized void executeWithRetry(ConnectionConsumer consumer, String name, boolean rollback) throws SQLException {
        this.executeWithRetry(null, consumer, name, rollback);
    }

    public synchronized <T> T executeWithRetry(ConnectionFunction<T> func, String name, boolean rollback) throws SQLException {
        return this.executeWithRetry(func, null, name, rollback);
    }

    private synchronized <T> T executeWithRetry(ConnectionFunction<T> func, ConnectionConsumer consumer, String name, boolean rollback) throws SQLException {
        int attempt = 1;
        while (true) {
            if (!this.isOpen()) {
                LOGGER.debug("Trying to reconnect (attempt {}).", (Object)attempt);
                try {
                    this.createConnection();
                }
                catch (SQLException e) {
                    LOGGER.error("SQL Exception while trying to reconnect: {}", (Object)e.getMessage(), (Object)e);
                    this.close();
                    if (attempt >= this.maxRetryCount) {
                        throw e;
                    }
                    ++attempt;
                    LOGGER.debug("Waiting for reconnect for {} ms.", (Object)this.waitRetryDelay);
                    DelayStrategy delayStrategy = DelayStrategy.constant((Duration)this.waitRetryDelay);
                    delayStrategy.sleepWhen(true);
                    continue;
                }
            }
            try {
                if (func != null) {
                    return func.accept(this.conn);
                }
                if (consumer == null) continue;
                consumer.accept(this.conn);
                return null;
            }
            catch (SQLException e) {
                LOGGER.warn("Attempt {} to call '{}' failed.", new Object[]{attempt, name, e});
                if (rollback) {
                    LOGGER.warn("'{}': doing rollback.", (Object)name);
                    try {
                        this.conn.rollback();
                    }
                    catch (SQLException sQLException) {
                        // empty catch block
                    }
                }
                this.close();
                continue;
            }
            break;
        }
    }

    @FunctionalInterface
    public static interface ConnectionFunction<T> {
        public T accept(Connection var1) throws SQLException;
    }

    @FunctionalInterface
    public static interface ConnectionConsumer {
        public void accept(Connection var1) throws SQLException;

        default public ConnectionConsumer andThen(ConnectionConsumer after) {
            Objects.requireNonNull(after);
            return c -> {
                this.accept(c);
                after.accept(c);
            };
        }
    }
}

