/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.spanner.task;

import com.google.cloud.Timestamp;
import io.debezium.connector.spanner.SpannerConnectorConfig;
import io.debezium.connector.spanner.task.LowWatermarkCalculator;
import io.debezium.connector.spanner.task.LowWatermarkHolder;
import io.debezium.util.Clock;
import io.debezium.util.Metronome;
import io.debezium.util.Stopwatch;
import java.time.Duration;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LowWatermarkCalculationJob {
    private static final Logger LOGGER = LoggerFactory.getLogger(LowWatermarkCalculationJob.class);
    private volatile Thread calculationThread;
    private final Duration pollInterval = Duration.ofMillis(60000L);
    private final Consumer<Throwable> errorHandler;
    private final boolean enabled;
    private final long period;
    private final LowWatermarkCalculator lowWatermarkCalculator;
    private final LowWatermarkHolder lowWatermarkHolder;
    private final String taskUid;
    private final Clock clock;
    private final Duration sleepInterval = Duration.ofMillis(100L);

    public LowWatermarkCalculationJob(SpannerConnectorConfig connectorConfig, Consumer<Throwable> errorHandler, LowWatermarkCalculator lowWatermarkCalculator, LowWatermarkHolder lowWatermarkHolder, String taskUid) {
        this.errorHandler = errorHandler;
        this.lowWatermarkCalculator = lowWatermarkCalculator;
        this.lowWatermarkHolder = lowWatermarkHolder;
        this.enabled = connectorConfig.isLowWatermarkEnabled();
        this.period = connectorConfig.getLowWatermarkUpdatePeriodMs();
        this.taskUid = taskUid;
        this.clock = Clock.system();
    }

    private Thread createCalculationThread() {
        Thread thread = new Thread(() -> {
            try {
                Stopwatch sw = Stopwatch.accumulating().start();
                Metronome metronome = Metronome.sleeper((Duration)Duration.ofMillis(this.period), (Clock)this.clock);
                LOGGER.info("Task {}, beginning calculation of low watermark", (Object)this.taskUid);
                while (!Thread.currentThread().isInterrupted()) {
                    try {
                        Duration totalDuration = sw.stop().durations().statistics().getTotal();
                        boolean printOffsets = false;
                        if (totalDuration.toMillis() >= this.pollInterval.toMillis()) {
                            printOffsets = true;
                            sw = Stopwatch.accumulating().start();
                        } else {
                            sw.start();
                        }
                        this.getLowWatermark(printOffsets);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        LOGGER.info("Task {}, interrupted low watermark calculation", (Object)this.taskUid);
                        LOGGER.info("Task {}, ended calculation of low watermark", (Object)this.taskUid);
                        return;
                    }
                    try {
                        metronome.pause();
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        LOGGER.info("Task {}, ended calculation of low watermark", (Object)this.taskUid);
                        return;
                    }
                }
            }
            finally {
                LOGGER.info("Task {}, ended calculation of low watermark", (Object)this.taskUid);
            }
        }, "SpannerConnector-WatermarkCalculationJob-Calculation");
        thread.setUncaughtExceptionHandler((t, e) -> {
            LOGGER.error("Task {}, caught exception during low watermark calculation {}", (Object)this.taskUid, (Object)e);
            this.errorHandler.accept(e);
        });
        return thread;
    }

    private void getLowWatermark(boolean printOffsets) throws InterruptedException {
        Timestamp timestamp;
        Metronome metronome = Metronome.sleeper((Duration)Duration.ofMillis(100L), (Clock)this.clock);
        do {
            try {
                timestamp = this.lowWatermarkCalculator.calculateLowWatermark(printOffsets);
                if (timestamp != null) break;
                metronome.pause();
            }
            catch (InterruptedException e) {
                LOGGER.warn("Task {}, low watermark calculation was interrupted", (Object)this.taskUid);
                throw e;
            }
        } while (!Thread.currentThread().isInterrupted());
        this.lowWatermarkHolder.setLowWatermark(timestamp);
    }

    public void start() {
        if (!this.enabled) {
            return;
        }
        LOGGER.info("Task {}, Started low watermark calculation", (Object)this.taskUid);
        this.calculationThread = this.createCalculationThread();
        this.calculationThread.start();
    }

    public void stop() {
        if (this.calculationThread != null) {
            this.calculationThread.interrupt();
            Metronome metronome = Metronome.sleeper((Duration)this.sleepInterval, (Clock)this.clock);
            LOGGER.info("Task {}, stopping low watermark calculation thread ", (Object)this.taskUid);
            while (!this.calculationThread.getState().equals((Object)Thread.State.TERMINATED)) {
                try {
                    metronome.pause();
                    this.calculationThread.interrupt();
                    LOGGER.info("Task {}, still waiting for low watermark calculation thread to die", (Object)this.taskUid);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
            LOGGER.info("Task {}, stopped low watermark calculation thread ", (Object)this.taskUid);
            this.calculationThread = null;
        }
    }
}

