/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.postgresql.connection;

import io.debezium.config.Configuration;
import io.debezium.config.Field;
import io.debezium.connector.postgresql.PostgresType;
import io.debezium.connector.postgresql.TypeRegistry;
import io.debezium.connector.postgresql.connection.ServerInfo;
import io.debezium.connector.postgresql.spi.SlotState;
import io.debezium.jdbc.JdbcConfiguration;
import io.debezium.jdbc.JdbcConnection;
import io.debezium.relational.Column;
import io.debezium.relational.ColumnEditor;
import io.debezium.relational.TableId;
import io.debezium.relational.Tables;
import io.debezium.util.Clock;
import io.debezium.util.Metronome;
import java.nio.charset.Charset;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.kafka.connect.errors.ConnectException;
import org.postgresql.Driver;
import org.postgresql.jdbc.PgConnection;
import org.postgresql.replication.LogSequenceNumber;
import org.postgresql.util.PSQLState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PostgresConnection
extends JdbcConnection {
    private static Logger LOGGER = LoggerFactory.getLogger(PostgresConnection.class);
    private static final String URL_PATTERN = "jdbc:postgresql://${" + JdbcConfiguration.HOSTNAME + "}:${" + JdbcConfiguration.PORT + "}/${" + JdbcConfiguration.DATABASE + "}";
    protected static final JdbcConnection.ConnectionFactory FACTORY = JdbcConnection.patternBasedFactory((String)URL_PATTERN, (String)Driver.class.getName(), (ClassLoader)PostgresConnection.class.getClassLoader(), (Field[])new Field[0]);
    private static final int MAX_ATTEMPTS_FOR_OBTAINING_REPLICATION_SLOT = 900;
    private static final Duration PAUSE_BETWEEN_REPLICATION_SLOT_RETRIEVAL_ATTEMPTS = Duration.ofSeconds(2L);
    private final TypeRegistry typeRegistry = new TypeRegistry(this);
    private final Charset databaseCharset = this.determineDatabaseCharset();

    public PostgresConnection(Configuration config) {
        super(config, FACTORY, PostgresConnection::validateServerVersion, PostgresConnection::defaultSettings);
    }

    public String connectionString() {
        return this.connectionString(URL_PATTERN);
    }

    public ServerInfo.ReplicaIdentity readReplicaIdentityInfo(TableId tableId) throws SQLException {
        String statement = "SELECT relreplident FROM pg_catalog.pg_class c LEFT JOIN pg_catalog.pg_namespace n ON c.relnamespace=n.oid WHERE n.nspname=? and c.relname=?";
        String schema = tableId.schema() != null && tableId.schema().length() > 0 ? tableId.schema() : "public";
        StringBuilder replIdentity = new StringBuilder();
        this.prepareQuery(statement, stmt -> {
            stmt.setString(1, schema);
            stmt.setString(2, tableId.table());
        }, rs -> {
            if (rs.next()) {
                replIdentity.append(rs.getString(1));
            } else {
                LOGGER.warn("Cannot determine REPLICA IDENTITY information for table '{}'", (Object)tableId);
            }
        });
        return ServerInfo.ReplicaIdentity.parseFromDB(replIdentity.toString());
    }

    public SlotState getReplicationSlotState(String slotName, String pluginName) throws SQLException {
        try {
            ServerInfo.ReplicationSlot slot = this.readReplicationSlotInfo(slotName, pluginName);
            if (slot.equals(ServerInfo.ReplicationSlot.INVALID)) {
                return null;
            }
            return slot.asSlotState();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ConnectException("Interrupted while waiting for valid replication slot info", (Throwable)e);
        }
    }

    private ServerInfo.ReplicationSlot fetchReplicationSlotInfo(String slotName, String pluginName) throws SQLException {
        String database = this.database();
        ServerInfo.ReplicationSlot slot = this.queryForSlot(slotName, database, pluginName, (JdbcConnection.ResultSetMapper<ServerInfo.ReplicationSlot>)((JdbcConnection.ResultSetMapper)rs -> {
            if (rs.next()) {
                boolean active = rs.getBoolean("active");
                Long confirmedFlushedLsn = this.parseConfirmedFlushLsn(slotName, pluginName, database, rs);
                if (confirmedFlushedLsn == null) {
                    return null;
                }
                Long restartLsn = this.parseRestartLsn(slotName, pluginName, database, rs);
                if (restartLsn == null) {
                    return null;
                }
                Long xmin = rs.getLong("catalog_xmin");
                return new ServerInfo.ReplicationSlot(active, confirmedFlushedLsn, restartLsn, xmin);
            }
            LOGGER.debug("No replication slot '{}' is present for plugin '{}' and database '{}'", new Object[]{slotName, pluginName, database});
            return ServerInfo.ReplicationSlot.INVALID;
        }));
        return slot;
    }

    ServerInfo.ReplicationSlot readReplicationSlotInfo(String slotName, String pluginName) throws SQLException, InterruptedException {
        String database = this.database();
        Metronome metronome = Metronome.parker((Duration)PAUSE_BETWEEN_REPLICATION_SLOT_RETRIEVAL_ATTEMPTS, (Clock)Clock.SYSTEM);
        for (int attempt = 1; attempt <= 900; ++attempt) {
            ServerInfo.ReplicationSlot slot = this.fetchReplicationSlotInfo(slotName, pluginName);
            if (slot != null) {
                LOGGER.info("Obtained valid replication slot {}", (Object)slot);
                return slot;
            }
            LOGGER.warn("Cannot obtain valid replication slot '{}' for plugin '{}' and database '{}' [during attempt {} out of {}, concurrent tx probably blocks taking snapshot.", new Object[]{slotName, pluginName, database, attempt, 900});
            metronome.pause();
        }
        throw new ConnectException("Unable to obtain valid replication slot. Make sure there are no long-running transactions running in parallel as they may hinder the allocation of the replication slot when starting this connector");
    }

    protected ServerInfo.ReplicationSlot queryForSlot(String slotName, String database, String pluginName, JdbcConnection.ResultSetMapper<ServerInfo.ReplicationSlot> map) throws SQLException {
        return (ServerInfo.ReplicationSlot)this.prepareQueryAndMap("select * from pg_replication_slots where slot_name = ? and database = ? and plugin = ?", statement -> {
            statement.setString(1, slotName);
            statement.setString(2, database);
            statement.setString(3, pluginName);
        }, map);
    }

    private Long parseConfirmedFlushLsn(String slotName, String pluginName, String database, ResultSet rs) {
        Long confirmedFlushedLsn = null;
        try {
            confirmedFlushedLsn = this.tryParseLsn(slotName, pluginName, database, rs, "confirmed_flush_lsn");
        }
        catch (SQLException e) {
            LOGGER.info("unable to find confirmed_flushed_lsn, falling back to restart_lsn");
            try {
                confirmedFlushedLsn = this.tryParseLsn(slotName, pluginName, database, rs, "restart_lsn");
            }
            catch (SQLException e2) {
                throw new ConnectException("Neither confirmed_flush_lsn nor restart_lsn could be found");
            }
        }
        return confirmedFlushedLsn;
    }

    private Long parseRestartLsn(String slotName, String pluginName, String database, ResultSet rs) {
        Long restartLsn = null;
        try {
            restartLsn = this.tryParseLsn(slotName, pluginName, database, rs, "restart_lsn");
        }
        catch (SQLException e) {
            throw new ConnectException("restart_lsn could be found");
        }
        return restartLsn;
    }

    private Long tryParseLsn(String slotName, String pluginName, String database, ResultSet rs, String column) throws ConnectException, SQLException {
        Long lsn = null;
        String lsnStr = rs.getString(column);
        if (lsnStr == null) {
            return null;
        }
        try {
            lsn = LogSequenceNumber.valueOf((String)lsnStr).asLong();
        }
        catch (Exception e) {
            throw new ConnectException("Value " + column + " in the pg_replication_slots table for slot = '" + slotName + "', plugin = '" + pluginName + "', database = '" + database + "' is not valid. This is an abnormal situation and the database status should be checked.");
        }
        if (lsn.longValue() == LogSequenceNumber.INVALID_LSN.asLong()) {
            throw new ConnectException("Invalid LSN returned from database");
        }
        return lsn;
    }

    public boolean dropReplicationSlot(String slotName) {
        try {
            this.execute(new String[]{"select pg_drop_replication_slot('" + slotName + "')"});
            return true;
        }
        catch (SQLException e) {
            if (PSQLState.OBJECT_IN_USE.getState().equals(e.getSQLState())) {
                LOGGER.warn("Cannot drop replication slot '{}' because it's still in use", (Object)slotName);
                return false;
            }
            if (PSQLState.UNDEFINED_OBJECT.getState().equals(e.getSQLState())) {
                LOGGER.debug("Replication slot {} has already been dropped", (Object)slotName);
                return false;
            }
            LOGGER.error("Unexpected error while attempting to drop replication slot", (Throwable)e);
            return false;
        }
    }

    public boolean dropPublication(String publicationName) {
        try {
            LOGGER.debug("Dropping publication '{}'", (Object)publicationName);
            this.execute(new String[]{"DROP PUBLICATION " + publicationName});
            return true;
        }
        catch (SQLException e) {
            if (PSQLState.UNDEFINED_OBJECT.getState().equals(e.getSQLState())) {
                LOGGER.debug("Publication {} has already been dropped", (Object)publicationName);
            } else {
                LOGGER.error("Unexpected error while attempting to drop publication", (Throwable)e);
            }
            return false;
        }
    }

    public synchronized void close() {
        try {
            super.close();
        }
        catch (SQLException e) {
            LOGGER.error("Unexpected error while closing Postgres connection", (Throwable)e);
        }
    }

    public Long currentTransactionId() throws SQLException {
        AtomicLong txId = new AtomicLong(0L);
        this.query("select * from txid_current()", rs -> {
            if (rs.next()) {
                txId.compareAndSet(0L, rs.getLong(1));
            }
        });
        long value = txId.get();
        return value > 0L ? Long.valueOf(value) : null;
    }

    public long currentXLogLocation() throws SQLException {
        AtomicLong result = new AtomicLong(0L);
        int majorVersion = this.connection().getMetaData().getDatabaseMajorVersion();
        this.query(majorVersion >= 10 ? "select * from pg_current_wal_lsn()" : "select * from pg_current_xlog_location()", rs -> {
            if (!rs.next()) {
                throw new IllegalStateException("there should always be a valid xlog position");
            }
            result.compareAndSet(0L, LogSequenceNumber.valueOf((String)rs.getString(1)).asLong());
        });
        return result.get();
    }

    public ServerInfo serverInfo() throws SQLException {
        ServerInfo serverInfo = new ServerInfo();
        this.query("SELECT version(), current_user, current_database()", rs -> {
            if (rs.next()) {
                serverInfo.withServer(rs.getString(1)).withUsername(rs.getString(2)).withDatabase(rs.getString(3));
            }
        });
        String username = serverInfo.username();
        if (username != null) {
            this.query("SELECT oid, rolname, rolsuper, rolinherit, rolcreaterole, rolcreatedb, rolcanlogin, rolreplication FROM pg_roles WHERE pg_has_role('" + username + "', oid, 'member')", rs -> {
                while (rs.next()) {
                    String roleInfo = "superuser: " + rs.getBoolean(3) + ", replication: " + rs.getBoolean(8) + ", inherit: " + rs.getBoolean(4) + ", create role: " + rs.getBoolean(5) + ", create db: " + rs.getBoolean(6) + ", can log in: " + rs.getBoolean(7);
                    String roleName = rs.getString(2);
                    serverInfo.addRole(roleName, roleInfo);
                }
            });
        }
        return serverInfo;
    }

    public Charset getDatabaseCharset() {
        return this.databaseCharset;
    }

    private Charset determineDatabaseCharset() {
        try {
            return Charset.forName(((PgConnection)this.connection()).getEncoding().name());
        }
        catch (SQLException e) {
            throw new RuntimeException("Couldn't obtain encoding for database " + this.database(), e);
        }
    }

    protected static void defaultSettings(Configuration.Builder builder) {
        builder.with("assumeMinServerVersion", "9.4");
    }

    private static void validateServerVersion(Statement statement) throws SQLException {
        DatabaseMetaData metaData = statement.getConnection().getMetaData();
        int majorVersion = metaData.getDatabaseMajorVersion();
        int minorVersion = metaData.getDatabaseMinorVersion();
        if (majorVersion < 9 || majorVersion == 9 && minorVersion < 4) {
            throw new SQLException("Cannot connect to a version of Postgres lower than 9.4");
        }
    }

    protected int resolveNativeType(String typeName) {
        return this.getTypeRegistry().get(typeName).getRootType().getOid();
    }

    protected int resolveJdbcType(int metadataJdbcType, int nativeType) {
        return this.getTypeRegistry().get(nativeType).getRootType().getJdbcId();
    }

    protected Optional<ColumnEditor> readTableColumn(ResultSet columnMetadata, TableId tableId, Tables.ColumnNameFilter columnFilter) throws SQLException {
        String columnName = columnMetadata.getString(4);
        if (columnFilter == null || columnFilter.matches(tableId.catalog(), tableId.schema(), tableId.table(), columnName)) {
            ColumnEditor column = Column.editor().name(columnName);
            column.type(columnMetadata.getString(6));
            column.length(columnMetadata.getInt(7));
            if (columnMetadata.getObject(9) != null) {
                column.scale(Integer.valueOf(columnMetadata.getInt(9)));
            }
            column.optional(PostgresConnection.isNullable((int)columnMetadata.getInt(11)));
            column.position(columnMetadata.getInt(17));
            column.autoIncremented("YES".equalsIgnoreCase(columnMetadata.getString(23)));
            String autogenerated = null;
            try {
                autogenerated = columnMetadata.getString(24);
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            column.generated("YES".equalsIgnoreCase(autogenerated));
            PostgresType nativeType = this.getTypeRegistry().get(column.typeName());
            column.nativeType(nativeType.getRootType().getOid());
            column.jdbcType(nativeType.getRootType().getJdbcId());
            if (2001 == nativeType.getJdbcId()) {
                column.length(nativeType.getDefaultLength());
                column.scale(Integer.valueOf(nativeType.getDefaultScale()));
            }
            return Optional.of(column);
        }
        return Optional.empty();
    }

    public TypeRegistry getTypeRegistry() {
        return this.typeRegistry;
    }
}

