/*
 * Decompiled with CFR 0.152.
 */
package io.awspring.cloud.sqs.listener.errorhandler;

import io.awspring.cloud.sqs.MessageHeaderUtils;
import io.awspring.cloud.sqs.listener.BatchVisibility;
import io.awspring.cloud.sqs.listener.Visibility;
import io.awspring.cloud.sqs.listener.errorhandler.AsyncErrorHandler;
import io.awspring.cloud.sqs.listener.errorhandler.ErrorHandlerVisibilityHelper;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.messaging.Message;
import org.springframework.util.Assert;

public class ExponentialBackoffErrorHandler<T>
implements AsyncErrorHandler<T> {
    private static final Logger logger = LoggerFactory.getLogger(ExponentialBackoffErrorHandler.class);
    private final int initialVisibilityTimeoutSeconds;
    private final double multiplier;
    private final int maxVisibilityTimeoutSeconds;

    private ExponentialBackoffErrorHandler(int initialVisibilityTimeoutSeconds, double multiplier, int maxVisibilityTimeoutSeconds) {
        this.initialVisibilityTimeoutSeconds = initialVisibilityTimeoutSeconds;
        this.multiplier = multiplier;
        this.maxVisibilityTimeoutSeconds = maxVisibilityTimeoutSeconds;
    }

    @Override
    public CompletableFuture<Void> handle(Message<T> message, Throwable t) {
        return this.applyExponentialBackoffVisibilityTimeout(message).thenCompose(theVoid -> CompletableFuture.failedFuture(t));
    }

    @Override
    public CompletableFuture<Void> handle(Collection<Message<T>> messages, Throwable t) {
        return this.applyExponentialBackoffVisibilityTimeout(messages).thenCompose(theVoid -> CompletableFuture.failedFuture(t));
    }

    private CompletableFuture<Void> applyExponentialBackoffVisibilityTimeout(Collection<Message<T>> messages) {
        CompletableFuture[] futures = (CompletableFuture[])ErrorHandlerVisibilityHelper.groupMessagesByReceiveMessageCount(messages).entrySet().stream().map(entry -> {
            int timeout = this.calculateTimeout((Long)entry.getKey());
            return this.applyBatchVisibilityChange((Collection)entry.getValue(), timeout);
        }).toArray(CompletableFuture[]::new);
        return CompletableFuture.allOf(futures);
    }

    private CompletableFuture<Void> applyBatchVisibilityChange(Collection<Message<T>> messages, int timeout) {
        logger.debug("Changing batch visibility timeout to {} - Messages Id {}", (Object)timeout, (Object)MessageHeaderUtils.getId(messages));
        BatchVisibility visibility = ErrorHandlerVisibilityHelper.getVisibility(messages);
        return visibility.changeToAsync(timeout).exceptionallyCompose(throwable -> {
            logger.warn("Failed to change batch visibility timeout to {} - Messages Id {}", new Object[]{timeout, MessageHeaderUtils.getId(messages), throwable});
            return CompletableFuture.failedFuture(throwable);
        });
    }

    private CompletableFuture<Void> applyExponentialBackoffVisibilityTimeout(Message<T> message) {
        int timeout = this.calculateTimeout(message);
        Visibility visibility = ErrorHandlerVisibilityHelper.getVisibility(message);
        logger.debug("Changing visibility timeout to {} - Message Id {}", (Object)timeout, (Object)message.getHeaders().getId());
        return visibility.changeToAsync(timeout).exceptionallyCompose(throwable -> {
            logger.warn("Failed to change visibility timeout to {} - Message Id {}", new Object[]{timeout, message.getHeaders().getId(), throwable});
            return CompletableFuture.failedFuture(throwable);
        });
    }

    private int calculateTimeout(Message<T> message) {
        long receiveMessageCount = ErrorHandlerVisibilityHelper.getReceiveMessageCount(message);
        return this.calculateTimeout(receiveMessageCount);
    }

    private int calculateTimeout(long receiveMessageCount) {
        double exponential = (double)this.initialVisibilityTimeoutSeconds * Math.pow(this.multiplier, receiveMessageCount - 1L);
        int seconds = (int)Math.min(exponential, 2.147483647E9);
        return Math.min(seconds, this.maxVisibilityTimeoutSeconds);
    }

    public static <T> Builder<T> builder() {
        return new Builder();
    }

    public static class Builder<T> {
        private static final int DEFAULT_INITIAL_VISIBILITY_TIMEOUT_SECONDS = 100;
        private static final double DEFAULT_MULTIPLIER = 2.0;
        private int initialVisibilityTimeoutSeconds = 100;
        private double multiplier = 2.0;
        private int maxVisibilityTimeoutSeconds = 43200;

        public Builder<T> initialVisibilityTimeoutSeconds(int initialVisibilityTimeoutSeconds) {
            this.checkVisibilityTimeout(initialVisibilityTimeoutSeconds);
            this.initialVisibilityTimeoutSeconds = initialVisibilityTimeoutSeconds;
            return this;
        }

        public Builder<T> multiplier(double multiplier) {
            Assert.isTrue((multiplier >= 1.0 ? 1 : 0) != 0, () -> "Invalid multiplier '" + multiplier + "'. Should be greater than or equal to 1.");
            this.multiplier = multiplier;
            return this;
        }

        public Builder<T> maxVisibilityTimeoutSeconds(int maxVisibilityTimeoutSeconds) {
            this.checkVisibilityTimeout(maxVisibilityTimeoutSeconds);
            this.maxVisibilityTimeoutSeconds = maxVisibilityTimeoutSeconds;
            return this;
        }

        public ExponentialBackoffErrorHandler<T> build() {
            Assert.isTrue((this.initialVisibilityTimeoutSeconds <= this.maxVisibilityTimeoutSeconds ? 1 : 0) != 0, (String)"Initial visibility timeout must not exceed max visibility timeout");
            return new ExponentialBackoffErrorHandler(this.initialVisibilityTimeoutSeconds, this.multiplier, this.maxVisibilityTimeoutSeconds);
        }

        private void checkVisibilityTimeout(long visibilityTimeout) {
            Assert.isTrue((visibilityTimeout > 0L ? 1 : 0) != 0, () -> "Invalid visibility timeout '" + visibilityTimeout + "'. Should be greater than 0 ");
            Assert.isTrue((visibilityTimeout <= 43200L ? 1 : 0) != 0, () -> "Invalid visibility timeout '" + visibilityTimeout + "'. Should be less than or equal to 43200");
        }
    }
}

