/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.protocols.raft.session.impl;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.primitives.Longs;
import io.atomix.cluster.MemberId;
import io.atomix.primitive.PrimitiveState;
import io.atomix.primitive.PrimitiveType;
import io.atomix.primitive.service.ServiceConfig;
import io.atomix.primitive.session.SessionId;
import io.atomix.protocols.raft.RaftClient;
import io.atomix.protocols.raft.RaftException;
import io.atomix.protocols.raft.ReadConsistency;
import io.atomix.protocols.raft.protocol.CloseSessionRequest;
import io.atomix.protocols.raft.protocol.HeartbeatRequest;
import io.atomix.protocols.raft.protocol.HeartbeatResponse;
import io.atomix.protocols.raft.protocol.KeepAliveRequest;
import io.atomix.protocols.raft.protocol.OpenSessionRequest;
import io.atomix.protocols.raft.protocol.RaftClientProtocol;
import io.atomix.protocols.raft.protocol.RaftResponse;
import io.atomix.protocols.raft.session.CommunicationStrategy;
import io.atomix.protocols.raft.session.impl.MemberSelectorManager;
import io.atomix.protocols.raft.session.impl.RaftSessionConnection;
import io.atomix.protocols.raft.session.impl.RaftSessionState;
import io.atomix.utils.concurrent.Futures;
import io.atomix.utils.concurrent.Scheduled;
import io.atomix.utils.concurrent.ThreadContext;
import io.atomix.utils.concurrent.ThreadContextFactory;
import io.atomix.utils.logging.ContextualLoggerFactory;
import io.atomix.utils.logging.LoggerContext;
import io.atomix.utils.serializer.Namespace;
import io.atomix.utils.serializer.Serializer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.slf4j.Logger;

public class RaftSessionManager {
    private static final double TIMEOUT_FACTOR = 0.5;
    private static final long MIN_TIMEOUT_DELTA = 2500L;
    private final Logger log;
    private final String clientId;
    private final MemberId memberId;
    private final RaftClientProtocol protocol;
    private final RaftSessionConnection connection;
    private final ThreadContextFactory threadContextFactory;
    private final ThreadContext threadContext;
    private final MemberSelectorManager selectorManager;
    private final Map<Long, RaftSessionState> sessions = new ConcurrentHashMap<Long, RaftSessionState>();
    private final Map<Long, Scheduled> keepAliveTimers = new ConcurrentHashMap<Long, Scheduled>();
    private final AtomicBoolean open = new AtomicBoolean();

    public RaftSessionManager(String clientId, MemberId memberId, RaftClientProtocol protocol, MemberSelectorManager selectorManager, ThreadContextFactory threadContextFactory) {
        this.clientId = (String)Preconditions.checkNotNull((Object)clientId, (Object)"clientId cannot be null");
        this.memberId = (MemberId)Preconditions.checkNotNull((Object)memberId, (Object)"memberId cannot be null");
        this.protocol = (RaftClientProtocol)Preconditions.checkNotNull((Object)protocol, (Object)"protocol cannot be null");
        this.selectorManager = (MemberSelectorManager)Preconditions.checkNotNull((Object)selectorManager, (Object)"selectorManager cannot be null");
        this.threadContext = threadContextFactory.createContext();
        this.log = ContextualLoggerFactory.getLogger(this.getClass(), (LoggerContext)LoggerContext.builder(RaftClient.class).addValue((Object)clientId).build());
        this.connection = new RaftSessionConnection(protocol, selectorManager.createSelector(CommunicationStrategy.LEADER), threadContextFactory.createContext(), LoggerContext.builder(RaftClient.class).addValue((Object)clientId).build());
        protocol.registerHeartbeatHandler(this::handleHeartbeat);
        this.threadContextFactory = (ThreadContextFactory)Preconditions.checkNotNull((Object)threadContextFactory, (Object)"threadContextFactory cannot be null");
    }

    public long term() {
        return 0L;
    }

    public MemberId leader() {
        return this.selectorManager.leader();
    }

    public void resetConnections() {
        this.selectorManager.resetAll();
    }

    public void resetConnections(MemberId leader, Collection<MemberId> servers) {
        this.selectorManager.resetAll(leader, servers);
    }

    public CompletableFuture<Void> open() {
        this.open.set(true);
        return CompletableFuture.completedFuture(null);
    }

    public CompletableFuture<RaftSessionState> openSession(String serviceName, PrimitiveType primitiveType, ServiceConfig config, ReadConsistency readConsistency, CommunicationStrategy communicationStrategy, Duration minTimeout, Duration maxTimeout) {
        Preconditions.checkNotNull((Object)serviceName, (Object)"serviceName cannot be null");
        Preconditions.checkNotNull((Object)primitiveType, (Object)"serviceType cannot be null");
        Preconditions.checkNotNull((Object)((Object)communicationStrategy), (Object)"communicationStrategy cannot be null");
        Preconditions.checkNotNull((Object)maxTimeout, (Object)"timeout cannot be null");
        this.log.debug("Opening session; name: {}, type: {}", (Object)serviceName, (Object)primitiveType);
        OpenSessionRequest request = OpenSessionRequest.builder().withMemberId(this.memberId).withServiceName(serviceName).withServiceType(primitiveType).withServiceConfig(Serializer.using((Namespace)primitiveType.namespace()).encode((Object)config)).withReadConsistency(readConsistency).withMinTimeout(minTimeout.toMillis()).withMaxTimeout(maxTimeout.toMillis()).build();
        CompletableFuture<RaftSessionState> future = new CompletableFuture<RaftSessionState>();
        ThreadContext proxyContext = this.threadContextFactory.createContext();
        this.connection.openSession(request).whenCompleteAsync((response, error) -> {
            if (error == null) {
                if (response.status() == RaftResponse.Status.OK) {
                    RaftSessionState state = new RaftSessionState(this.clientId, SessionId.from((long)response.session()), serviceName, primitiveType, response.timeout());
                    this.sessions.put((Long)state.getSessionId().id(), state);
                    state.addStateChangeListener(s -> {
                        if (s == PrimitiveState.EXPIRED || s == PrimitiveState.CLOSED) {
                            this.sessions.remove(state.getSessionId().id());
                        }
                    });
                    this.keepAliveSessions(System.currentTimeMillis(), state.getSessionTimeout());
                    future.complete(state);
                } else {
                    future.completeExceptionally(new RaftException.Unavailable(response.error().message(), new Object[0]));
                }
            } else {
                future.completeExceptionally(new RaftException.Unavailable(error.getMessage(), new Object[0]));
            }
        }, (Executor)proxyContext);
        return future;
    }

    public CompletableFuture<Void> closeSession(SessionId sessionId, boolean delete) {
        RaftSessionState state = this.sessions.get(sessionId.id());
        if (state == null) {
            return Futures.exceptionalFuture((Throwable)new RaftException.UnknownSession("Unknown session: " + sessionId, new Object[0]));
        }
        this.log.debug("Closing session {}", (Object)sessionId);
        CloseSessionRequest request = ((CloseSessionRequest.Builder)CloseSessionRequest.builder().withSession((Long)sessionId.id())).withDelete(delete).build();
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.connection.closeSession(request).whenComplete((response, error) -> {
            this.sessions.remove(sessionId.id());
            if (error == null) {
                if (response.status() == RaftResponse.Status.OK) {
                    future.complete(null);
                } else {
                    future.completeExceptionally((Throwable)response.error().createException());
                }
            } else {
                future.completeExceptionally((Throwable)error);
            }
        });
        return future;
    }

    private synchronized void resetAllIndexes() {
        ArrayList sessions = Lists.newArrayList(this.sessions.values());
        if (sessions.isEmpty()) {
            return;
        }
        long[] sessionIds = new long[sessions.size()];
        long[] commandResponses = new long[sessions.size()];
        long[] eventIndexes = new long[sessions.size()];
        int i = 0;
        for (RaftSessionState sessionState : sessions) {
            sessionIds[i] = (Long)sessionState.getSessionId().id();
            commandResponses[i] = sessionState.getCommandResponse();
            eventIndexes[i] = sessionState.getEventIndex();
            ++i;
        }
        this.log.trace("Resetting {} sessions", (Object)sessionIds.length);
        KeepAliveRequest request = KeepAliveRequest.builder().withSessionIds(sessionIds).withCommandSequences(commandResponses).withEventIndexes(eventIndexes).build();
        this.connection.keepAlive(request);
    }

    CompletableFuture<Void> resetIndexes(SessionId sessionId) {
        RaftSessionState sessionState = this.sessions.get(sessionId.id());
        if (sessionState == null) {
            return Futures.exceptionalFuture((Throwable)new IllegalArgumentException("Unknown session: " + sessionId));
        }
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        KeepAliveRequest request = KeepAliveRequest.builder().withSessionIds(new long[]{(Long)sessionId.id()}).withCommandSequences(new long[]{sessionState.getCommandResponse()}).withEventIndexes(new long[]{sessionState.getEventIndex()}).build();
        this.connection.keepAlive(request).whenComplete((response, error) -> {
            if (error == null) {
                if (response.status() == RaftResponse.Status.OK) {
                    future.complete(null);
                } else {
                    future.completeExceptionally((Throwable)response.error().createException());
                }
            } else {
                future.completeExceptionally((Throwable)error);
            }
        });
        return future;
    }

    private synchronized void keepAliveSessions(long lastKeepAliveTime, long sessionTimeout) {
        List needKeepAlive = this.sessions.values().stream().filter(session -> session.getSessionTimeout() == sessionTimeout).collect(Collectors.toList());
        if (needKeepAlive.isEmpty()) {
            return;
        }
        long[] sessionIds = new long[needKeepAlive.size()];
        long[] commandResponses = new long[needKeepAlive.size()];
        long[] eventIndexes = new long[needKeepAlive.size()];
        int i = 0;
        for (RaftSessionState sessionState : needKeepAlive) {
            sessionIds[i] = (Long)sessionState.getSessionId().id();
            commandResponses[i] = sessionState.getCommandResponse();
            eventIndexes[i] = sessionState.getEventIndex();
            ++i;
        }
        this.log.trace("Keeping {} sessions alive", (Object)sessionIds.length);
        KeepAliveRequest request = KeepAliveRequest.builder().withSessionIds(sessionIds).withCommandSequences(commandResponses).withEventIndexes(eventIndexes).build();
        long keepAliveTime = System.currentTimeMillis();
        this.connection.keepAlive(request).whenComplete((response, error) -> {
            if (this.open.get()) {
                long delta = System.currentTimeMillis() - keepAliveTime;
                if (error == null) {
                    if (response.status() == RaftResponse.Status.OK) {
                        this.selectorManager.resetAll(response.leader(), response.members());
                        HashSet keptAliveSessions = Sets.newHashSet((Iterable)Longs.asList((long[])response.sessionIds()));
                        for (RaftSessionState session : needKeepAlive) {
                            if (keptAliveSessions.contains(session.getSessionId().id())) {
                                session.setState(PrimitiveState.CONNECTED);
                                continue;
                            }
                            session.setState(PrimitiveState.EXPIRED);
                        }
                        this.scheduleKeepAlive(System.currentTimeMillis(), sessionTimeout, delta);
                    } else if (System.currentTimeMillis() - lastKeepAliveTime < sessionTimeout) {
                        this.selectorManager.resetAll(null, this.connection.members());
                        this.keepAliveSessions(lastKeepAliveTime, sessionTimeout);
                    } else {
                        needKeepAlive.forEach(s -> s.setState(PrimitiveState.SUSPENDED));
                        this.selectorManager.resetAll();
                        this.scheduleKeepAlive(lastKeepAliveTime, sessionTimeout, delta);
                    }
                } else if (System.currentTimeMillis() - lastKeepAliveTime < sessionTimeout && this.connection.leader() != null) {
                    this.selectorManager.resetAll(null, this.connection.members());
                    this.keepAliveSessions(lastKeepAliveTime, sessionTimeout);
                } else {
                    needKeepAlive.forEach(s -> s.setState(PrimitiveState.SUSPENDED));
                    this.selectorManager.resetAll();
                    this.scheduleKeepAlive(lastKeepAliveTime, sessionTimeout, delta);
                }
            }
        });
    }

    private synchronized void scheduleKeepAlive(long lastKeepAliveTime, long timeout, long delta) {
        Scheduled keepAliveFuture = this.keepAliveTimers.remove(timeout);
        if (keepAliveFuture != null) {
            keepAliveFuture.cancel();
        }
        this.keepAliveTimers.put(timeout, this.threadContext.schedule(Duration.ofMillis(Math.max(Math.max((long)((double)timeout * 0.5) - delta, timeout - 2500L - delta), 0L)), () -> {
            if (this.open.get()) {
                this.keepAliveSessions(lastKeepAliveTime, timeout);
            }
        }));
    }

    private CompletableFuture<HeartbeatResponse> handleHeartbeat(HeartbeatRequest request) {
        this.log.trace("Received {}", (Object)request);
        boolean newLeader = !Objects.equals(this.selectorManager.leader(), request.leader());
        this.selectorManager.resetAll(request.leader(), request.members());
        HeartbeatResponse response = ((HeartbeatResponse.Builder)HeartbeatResponse.builder().withStatus(RaftResponse.Status.OK)).build();
        if (newLeader) {
            this.resetAllIndexes();
        }
        this.log.trace("Sending {}", (Object)response);
        return CompletableFuture.completedFuture(response);
    }

    public CompletableFuture<Void> close() {
        if (this.open.compareAndSet(true, false)) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            this.threadContext.execute(() -> {
                RaftSessionManager raftSessionManager = this;
                synchronized (raftSessionManager) {
                    for (Scheduled keepAliveFuture : this.keepAliveTimers.values()) {
                        keepAliveFuture.cancel();
                    }
                    this.protocol.unregisterHeartbeatHandler();
                }
                future.complete(null);
            });
            return future;
        }
        return CompletableFuture.completedFuture(null);
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("client", (Object)this.clientId).toString();
    }
}

