/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.protocols.raft.service.impl;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import io.atomix.protocols.raft.RaftException;
import io.atomix.protocols.raft.ReadConsistency;
import io.atomix.protocols.raft.cluster.MemberId;
import io.atomix.protocols.raft.impl.OperationResult;
import io.atomix.protocols.raft.impl.RaftContext;
import io.atomix.protocols.raft.operation.OperationId;
import io.atomix.protocols.raft.operation.OperationType;
import io.atomix.protocols.raft.operation.RaftOperation;
import io.atomix.protocols.raft.service.RaftService;
import io.atomix.protocols.raft.service.ServiceContext;
import io.atomix.protocols.raft.service.ServiceId;
import io.atomix.protocols.raft.service.ServiceType;
import io.atomix.protocols.raft.service.impl.DefaultCommit;
import io.atomix.protocols.raft.service.impl.DefaultServiceSessions;
import io.atomix.protocols.raft.session.RaftSession;
import io.atomix.protocols.raft.session.RaftSessions;
import io.atomix.protocols.raft.session.SessionId;
import io.atomix.protocols.raft.session.impl.RaftSessionContext;
import io.atomix.protocols.raft.session.impl.RaftSessionManager;
import io.atomix.protocols.raft.storage.snapshot.Snapshot;
import io.atomix.protocols.raft.storage.snapshot.SnapshotReader;
import io.atomix.protocols.raft.storage.snapshot.SnapshotWriter;
import io.atomix.time.LogicalClock;
import io.atomix.time.LogicalTimestamp;
import io.atomix.time.WallClock;
import io.atomix.time.WallClockTimestamp;
import io.atomix.utils.SlidingWindowCounter;
import io.atomix.utils.concurrent.ThreadContext;
import io.atomix.utils.concurrent.ThreadContextFactory;
import io.atomix.utils.logging.ContextualLoggerFactory;
import io.atomix.utils.logging.LoggerContext;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentSkipListMap;
import org.slf4j.Logger;

public class DefaultServiceContext
implements ServiceContext {
    private static final int WINDOW_SIZE = 5;
    private static final int LOAD_WINDOW = 2;
    private static final int HIGH_LOAD_THRESHOLD = 2;
    private final Logger log;
    private final ServiceId serviceId;
    private final String serviceName;
    private final ServiceType serviceType;
    private final RaftService service;
    private final RaftContext server;
    private final DefaultServiceSessions sessions;
    private final ThreadContext serviceExecutor;
    private final ThreadContext snapshotExecutor;
    private final ThreadContextFactory threadContextFactory;
    private final SlidingWindowCounter loadCounter;
    private final Map<Long, PendingSnapshot> pendingSnapshots = new ConcurrentSkipListMap<Long, PendingSnapshot>();
    private long snapshotIndex;
    private long currentIndex;
    private long currentTimestamp;
    private OperationType currentOperation;
    private final LogicalClock logicalClock = new LogicalClock(){

        public LogicalTimestamp getTime() {
            return new LogicalTimestamp(DefaultServiceContext.this.currentIndex);
        }
    };
    private final WallClock wallClock = new WallClock(){

        public WallClockTimestamp getTime() {
            return new WallClockTimestamp(DefaultServiceContext.this.currentTimestamp);
        }
    };

    public DefaultServiceContext(ServiceId serviceId, String serviceName, ServiceType serviceType, RaftService service, RaftContext server, RaftSessionManager sessionManager, ThreadContextFactory threadContextFactory) {
        this.serviceId = (ServiceId)((Object)Preconditions.checkNotNull((Object)((Object)serviceId)));
        this.serviceName = (String)Preconditions.checkNotNull((Object)serviceName);
        this.serviceType = (ServiceType)Preconditions.checkNotNull((Object)serviceType);
        this.service = (RaftService)Preconditions.checkNotNull((Object)service);
        this.server = (RaftContext)Preconditions.checkNotNull((Object)server);
        this.sessions = new DefaultServiceSessions(serviceId, sessionManager);
        this.serviceExecutor = threadContextFactory.createContext();
        this.snapshotExecutor = threadContextFactory.createContext();
        this.loadCounter = new SlidingWindowCounter(5, this.serviceExecutor);
        this.threadContextFactory = threadContextFactory;
        this.log = ContextualLoggerFactory.getLogger(this.getClass(), (LoggerContext)LoggerContext.builder(RaftService.class).addValue((Object)serviceId).add("type", (Object)serviceType).add("name", (Object)serviceName).build());
        this.init();
    }

    private void init() {
        this.sessions.addListener(this.service);
        this.service.init(this);
    }

    @Override
    public ServiceId serviceId() {
        return this.serviceId;
    }

    @Override
    public String serviceName() {
        return this.serviceName;
    }

    @Override
    public ServiceType serviceType() {
        return this.serviceType;
    }

    @Override
    public long currentIndex() {
        return this.currentIndex;
    }

    @Override
    public OperationType currentOperation() {
        return this.currentOperation;
    }

    @Override
    public LogicalClock logicalClock() {
        return this.logicalClock;
    }

    @Override
    public WallClock wallClock() {
        return this.wallClock;
    }

    @Override
    public RaftSessions sessions() {
        return this.sessions;
    }

    public boolean isUnderHighLoad() {
        return this.loadCounter.get(2) > 2L;
    }

    public ThreadContext executor() {
        return this.serviceExecutor;
    }

    private void setOperation(OperationType operation) {
        this.currentOperation = operation;
    }

    private void tick(long index, long timestamp) {
        this.currentIndex = index;
        this.currentTimestamp = Math.max(this.currentTimestamp, timestamp);
        this.setOperation(OperationType.COMMAND);
    }

    private void expireSessions(long timestamp) {
        for (RaftSessionContext session : this.sessions.getSessions()) {
            long lastUpdated = session.getTimestamp();
            if (lastUpdated <= 0L || timestamp - lastUpdated <= session.timeout()) continue;
            this.log.debug("Detected expired session {}", (Object)session);
            this.log.debug("Closing session {}", (Object)session.sessionId());
            this.sessions.expireSession(session);
        }
    }

    private void maybeCompleteSnapshot(long index) {
        if (!this.pendingSnapshots.isEmpty()) {
            long lastCompleted = index;
            for (RaftSessionContext session : this.sessions.getSessions()) {
                lastCompleted = Math.min(lastCompleted, session.getLastCompleted());
            }
            for (PendingSnapshot pendingSnapshot : this.pendingSnapshots.values()) {
                Snapshot snapshot = pendingSnapshot.snapshot;
                if (!snapshot.isPersisted() || lastCompleted < snapshot.index()) continue;
                this.log.debug("Completing snapshot {}", (Object)snapshot.index());
                snapshot.complete();
                this.snapshotIndex = snapshot.index();
                pendingSnapshot.future.complete(null);
            }
        }
    }

    private void maybeInstallSnapshot(long index) {
        Snapshot snapshot = this.server.getSnapshotStore().getSnapshotById(this.serviceId);
        if (snapshot != null && snapshot.index() > this.snapshotIndex && snapshot.index() < index) {
            this.log.debug("Installing snapshot {}", (Object)snapshot.index());
            try (SnapshotReader reader = snapshot.openReader();){
                reader.skip(8);
                ServiceType serviceType = ServiceType.from(reader.readString());
                String serviceName = reader.readString();
                int sessionCount = reader.readInt();
                for (int i = 0; i < sessionCount; ++i) {
                    SessionId sessionId = SessionId.from(reader.readLong());
                    MemberId node = MemberId.from(reader.readString());
                    ReadConsistency readConsistency = ReadConsistency.valueOf(reader.readString());
                    long sessionTimeout = reader.readLong();
                    long sessionTimestamp = reader.readLong();
                    RaftSessionContext session = new RaftSessionContext(sessionId, node, serviceName, serviceType, readConsistency, sessionTimeout, this, this.server, this.threadContextFactory);
                    session.setTimestamp(sessionTimestamp);
                    session.setRequestSequence(reader.readLong());
                    session.setCommandSequence(reader.readLong());
                    session.setEventIndex(reader.readLong());
                    session.setLastCompleted(reader.readLong());
                    session.setLastApplied(snapshot.index());
                    this.sessions.openSession(session);
                }
                this.service.install(reader);
            }
            catch (Exception e) {
                this.log.error("Snapshot installation failed: {}", (Throwable)e);
            }
            this.snapshotIndex = snapshot.index();
        }
    }

    public CompletableFuture<Long> takeSnapshot() {
        CompletableFuture<Long> future = new CompletableFuture<Long>();
        this.serviceExecutor.execute(() -> {
            if (this.currentIndex == 0L) {
                return;
            }
            long snapshotIndex = this.currentIndex;
            this.log.debug("Taking snapshot {}", (Object)snapshotIndex);
            Snapshot snapshot = this.server.getSnapshotStore().newTemporarySnapshot(this.serviceId, snapshotIndex, WallClockTimestamp.from((long)this.currentTimestamp));
            PendingSnapshot pendingSnapshot = new PendingSnapshot(snapshot);
            this.pendingSnapshots.put(snapshotIndex, pendingSnapshot);
            pendingSnapshot.future.whenComplete((r, e) -> this.pendingSnapshots.remove(snapshotIndex));
            try (SnapshotWriter writer = snapshot.openWriter();){
                writer.writeLong((Long)this.serviceId.id());
                writer.writeString((String)((Object)this.serviceType.id()));
                writer.writeString(this.serviceName);
                writer.writeInt(this.sessions.getSessions().size());
                for (RaftSessionContext session : this.sessions.getSessions()) {
                    writer.writeLong((Long)session.sessionId().id());
                    writer.writeString((String)((Object)session.memberId().id()));
                    writer.writeString(session.readConsistency().name());
                    writer.writeLong(session.timeout());
                    writer.writeLong(session.getTimestamp());
                    writer.writeLong(session.getRequestSequence());
                    writer.writeLong(session.getCommandSequence());
                    writer.writeLong(session.getEventIndex());
                    writer.writeLong(session.getLastCompleted());
                }
                this.service.snapshot(writer);
            }
            catch (Exception e2) {
                this.log.error("Snapshot failed: {}", (Throwable)e2);
            }
            this.snapshotExecutor.execute(() -> {
                pendingSnapshot.persist();
                future.complete(snapshotIndex);
            });
        });
        return future;
    }

    public CompletableFuture<Void> completeSnapshot(long index) {
        PendingSnapshot pendingSnapshot = this.pendingSnapshots.get(index);
        if (pendingSnapshot == null) {
            return CompletableFuture.completedFuture(null);
        }
        this.serviceExecutor.execute(() -> this.maybeCompleteSnapshot(index));
        return pendingSnapshot.future;
    }

    public CompletableFuture<Long> openSession(long index, long timestamp, RaftSessionContext session) {
        CompletableFuture<Long> future = new CompletableFuture<Long>();
        this.serviceExecutor.execute(() -> {
            this.log.debug("Opening session {}", (Object)session.sessionId());
            session.setTimestamp(timestamp);
            this.tick(index, timestamp);
            this.maybeInstallSnapshot(index);
            this.expireSessions(this.currentTimestamp);
            this.sessions.openSession(session);
            this.commit();
            this.maybeCompleteSnapshot(index);
            future.complete((Long)session.sessionId().id());
        });
        return future;
    }

    public CompletableFuture<Boolean> keepAlive(long index, long timestamp, RaftSessionContext session, long commandSequence, long eventIndex) {
        CompletableFuture<Boolean> future = new CompletableFuture<Boolean>();
        this.serviceExecutor.execute(() -> {
            this.tick(index, timestamp);
            this.maybeInstallSnapshot(index);
            if (session.getState() != RaftSession.State.CLOSED) {
                session.setTimestamp(timestamp);
                session.clearResults(commandSequence);
                session.resendEvents(eventIndex);
                session.resetRequestSequence(commandSequence);
                session.setCommandSequence(commandSequence);
                future.complete(true);
            } else {
                future.complete(false);
            }
        });
        return future;
    }

    public CompletableFuture<Void> completeKeepAlive(long index, long timestamp) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.serviceExecutor.execute(() -> {
            this.tick(index, timestamp);
            this.expireSessions(this.currentTimestamp);
            this.commit();
            this.maybeCompleteSnapshot(index);
            future.complete(null);
        });
        return future;
    }

    public CompletableFuture<Void> keepAliveSessions(long index, long timestamp) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.serviceExecutor.execute(() -> {
            this.log.debug("Resetting session timeouts");
            this.currentIndex = index;
            this.currentTimestamp = Math.max(this.currentTimestamp, timestamp);
            for (RaftSessionContext session : this.sessions.getSessions()) {
                session.setTimestamp(timestamp);
            }
        });
        return future;
    }

    public CompletableFuture<Void> closeSession(long index, long timestamp, RaftSessionContext session) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.serviceExecutor.execute(() -> {
            this.log.debug("Closing session {}", (Object)session.sessionId());
            session.setTimestamp(timestamp);
            this.tick(index, timestamp);
            this.maybeInstallSnapshot(index);
            this.expireSessions(this.currentTimestamp);
            this.sessions.closeSession(session);
            this.commit();
            this.maybeCompleteSnapshot(index);
            future.complete(null);
        });
        return future;
    }

    public CompletableFuture<OperationResult> executeCommand(long index, long sequence, long timestamp, RaftSessionContext session, RaftOperation operation) {
        CompletableFuture<OperationResult> future = new CompletableFuture<OperationResult>();
        this.serviceExecutor.execute(() -> this.executeCommand(index, sequence, timestamp, session, operation, future));
        return future;
    }

    private void executeCommand(long index, long sequence, long timestamp, RaftSessionContext session, RaftOperation operation, CompletableFuture<OperationResult> future) {
        session.setTimestamp(timestamp);
        this.tick(index, timestamp);
        this.maybeInstallSnapshot(index);
        if (!session.getState().active()) {
            future.completeExceptionally(new RaftException.UnknownSession("Unknown session: " + (Object)((Object)session.sessionId()), new Object[0]));
            return;
        }
        if (sequence > 0L && sequence < session.nextCommandSequence()) {
            this.sequenceCommand(index, sequence, session, future);
        } else {
            this.applyCommand(index, sequence, timestamp, operation, session, future);
            session.setCommandSequence(sequence);
        }
    }

    private void sequenceCommand(long index, long sequence, RaftSessionContext session, CompletableFuture<OperationResult> future) {
        OperationResult result = session.getResult(sequence);
        if (result == null) {
            this.log.debug("Missing command result at index {}", (Object)index);
        }
        future.complete(result);
    }

    private void applyCommand(long index, long sequence, long timestamp, RaftOperation operation, RaftSessionContext session, CompletableFuture<OperationResult> future) {
        OperationResult result;
        if (operation.id().equals(OperationId.NOOP)) {
            future.complete(OperationResult.noop(index, session.getEventIndex()));
            return;
        }
        DefaultCommit<byte[]> commit = new DefaultCommit<byte[]>(index, operation.id(), operation.value(), session, timestamp);
        long eventIndex = session.getEventIndex();
        try {
            byte[] output = this.service.apply(commit);
            result = OperationResult.succeeded(index, eventIndex, output);
        }
        catch (Exception e) {
            result = OperationResult.failed(index, eventIndex, e);
        }
        this.commit();
        session.registerResult(sequence, result);
        future.complete(result);
    }

    public CompletableFuture<OperationResult> executeQuery(long index, long sequence, long timestamp, RaftSessionContext session, RaftOperation operation) {
        CompletableFuture<OperationResult> future = new CompletableFuture<OperationResult>();
        this.serviceExecutor.execute(() -> this.executeQuery(index, sequence, timestamp, session, operation, future));
        return future;
    }

    private void executeQuery(long index, long sequence, long timestamp, RaftSessionContext session, RaftOperation operation, CompletableFuture<OperationResult> future) {
        if (!session.getState().active()) {
            this.log.warn("Inactive session: " + (Object)((Object)session.sessionId()));
            future.completeExceptionally(new RaftException.UnknownSession("Unknown session: " + (Object)((Object)session.sessionId()), new Object[0]));
            return;
        }
        this.sequenceQuery(index, sequence, timestamp, session, operation, future);
    }

    private void sequenceQuery(long index, long sequence, long timestamp, RaftSessionContext session, RaftOperation operation, CompletableFuture<OperationResult> future) {
        long commandSequence = session.getCommandSequence();
        if (sequence > commandSequence) {
            this.log.trace("Registering query with sequence number " + sequence + " > " + commandSequence);
            session.registerSequenceQuery(sequence, () -> this.indexQuery(index, timestamp, session, operation, future));
        } else {
            this.indexQuery(index, timestamp, session, operation, future);
        }
    }

    private void indexQuery(long index, long timestamp, RaftSessionContext session, RaftOperation operation, CompletableFuture<OperationResult> future) {
        if (index > this.currentIndex) {
            this.log.trace("Registering query with index " + index + " > " + this.currentIndex);
            session.registerIndexQuery(index, () -> this.applyQuery(timestamp, session, operation, future));
        } else {
            this.applyQuery(timestamp, session, operation, future);
        }
    }

    private void applyQuery(long timestamp, RaftSessionContext session, RaftOperation operation, CompletableFuture<OperationResult> future) {
        OperationResult result;
        if (!session.getState().active()) {
            this.log.warn("Inactive session: " + (Object)((Object)session.sessionId()));
            future.completeExceptionally(new RaftException.UnknownSession("Unknown session: " + (Object)((Object)session.sessionId()), new Object[0]));
            return;
        }
        this.setOperation(OperationType.QUERY);
        DefaultCommit<byte[]> commit = new DefaultCommit<byte[]>(session.getLastApplied(), operation.id(), operation.value(), session, timestamp);
        long eventIndex = session.getEventIndex();
        try {
            result = OperationResult.succeeded(this.currentIndex, eventIndex, this.service.apply(commit));
        }
        catch (Exception e) {
            result = OperationResult.failed(this.currentIndex, eventIndex, e);
        }
        future.complete(result);
    }

    private void commit() {
        long index = this.currentIndex;
        for (RaftSessionContext session : this.sessions.getSessions()) {
            session.commit(index);
        }
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("server", (Object)this.server.getName()).add("type", (Object)this.serviceType).add("name", (Object)this.serviceName).add("id", (Object)this.serviceId).toString();
    }

    private class PendingSnapshot {
        private volatile Snapshot snapshot;
        private final CompletableFuture<Void> future = new CompletableFuture();

        public PendingSnapshot(Snapshot snapshot) {
            this.snapshot = snapshot;
        }

        void persist() {
            this.snapshot = this.snapshot.persist();
        }
    }
}

