package com.atlassian.vcache.utils.cas;

import com.atlassian.vcache.CasIdentifier;
import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.PutPolicy;
import io.atlassian.fugue.Either;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletionException;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import static com.atlassian.vcache.VCacheUtils.fold;
import static io.atlassian.fugue.Either.left;
import static io.atlassian.fugue.Either.right;

/**
 * The purpose of this class is to provide a set of convenience functions to perform CAS operations
 * on a DirectExternalCache.
 */
public class VCacheCasUtils {
    private static final Clock clock = Clock.systemDefaultZone();

    private VCacheCasUtils() {
        throw new AssertionError("This class should not be instantiated!");
    }

    /**
     * Performs an atomic replacement on the cache by repeatedly calling the DirectExternalCache.replaceIf method
     * until it succeeds. The user provides a function which, given the current value in the cache, returns
     * a value to replace the current value.
     *
     * If there is no entry in the cache at the given key, the user provided defaultSupplier will be used to
     * populate the cache
     *
     * @param vcache           the DirectExternalCache to operate on
     * @param key              the key to write to within the cache
     * @param updater          the user provided replacement function
     * @param defaultSupplier  used to generate the value, if one does not exist already for the key. The supplier may not
     *                         return {@code null}.
     * @param maxDuration      the maximum duration to loop for before giving up; at least one loop is attempted
     *
     * @return either the Throwable if an exception occurred, or the value that was finally written to the cache; an
     *         empty optional means no change was made
     */
    public static <V> Either<Throwable, V> atomicReplace(final DirectExternalCache<V> vcache, final String key, final Function<V, V> updater, final Supplier<V> defaultSupplier, final Duration maxDuration) {
        return VCacheCasUtils.internalAtomicLoop(
                maxDuration,
                () -> fold(vcache.getIdentified(key),
                        identifiedValue -> {
                            if (identifiedValue.isPresent()) {
                                // There currently is a entry present in the cache so we will use replaceIf
                                final CasIdentifier identifier = identifiedValue.get().identifier();
                                final V value = identifiedValue.get().value();
                                final V replacement = updater.apply(value);

                                return fold(vcache.replaceIf(key, identifier, replacement),
                                        // Return the value written to the cache if successful
                                        res -> res ? Optional.of(right(replacement)) : Optional.empty(),
                                        err -> Optional.of(left(err))
                                );
                            } else {
                                // There currently is no entry in the cache so we will use put with the ADD_ONLY policy
                                final V item = defaultSupplier.get();

                                return fold(vcache.put(key, item, PutPolicy.ADD_ONLY),
                                        // Return the value written to the cache if successful
                                        res -> res ? Optional.of(right(item)) : Optional.empty(),
                                        err -> Optional.of(left(err))
                                );
                            }
                        },
                        err -> Optional.of(left(err))
                )
        );
    }

    /**
     * Performs an atomic removal on the cache by repeatedly calling the DirectExternalCache.removeIf method until
     * it succeeds. The user provides a predicate which, given the current value in the cache, returns true to remove
     * that value or false to perform no operation.
     *
     * If there is currently no entry in the cache at the given key, no operation will be performed and an empty
     * Optional will be returned.
     *
     * @param vcache      the DirectExternalCache to operate on
     * @param key         the key to write to within the cache
     * @param shouldRemove    the user provided predicate to decide whether to remove the entry or not
     * @param maxDuration the maximum duration to loop for before giving up; at least one loop is attempted
     *
     * @return either the Throwable if an exception occurred, or the value that was finally removed from the cache; an
     *         empty Optional means no change was made
     */
    public static <V> Either<Throwable, Optional<V>> atomicRemoveIf(final DirectExternalCache<V> vcache, final String key, final Predicate<V> shouldRemove, final Duration maxDuration) {
        return VCacheCasUtils.internalAtomicLoop(
                maxDuration,
                () -> fold(vcache.getIdentified(key),
                        identifiedValue -> {
                            if (!identifiedValue.isPresent()) {
                                // There currently is no entry present in the cache, nothing to remove
                                return Optional.of(right(Optional.empty()));
                            }

                            final V curCacheValue = identifiedValue.get().value();
                            if (!shouldRemove.test(curCacheValue)) {
                                // The entry should not be removed
                                return Optional.of(right(Optional.empty()));
                            }

                            return fold(vcache.removeIf(key, identifiedValue.get().identifier()),
                                    // Return removed value if it succeeded
                                    res -> res ? Optional.of(right(Optional.of(curCacheValue))) : Optional.empty(),
                                    err -> Optional.of(left(err))
                            );
                        },
                        err -> Optional.of(left(err))
                )
        );
    }

    private static <T> Either<Throwable, T> internalAtomicLoop(final Duration maxDuration, final Supplier<Optional<Either<Throwable, T>>> cacheOperation) {
        final Instant completeBy = clock.instant().plus(maxDuration);

        do {
            final Optional<Either<Throwable, T>> result = cacheOperation.get();
            if (result.isPresent()) {
                return result.get();
            }
        } while (clock.instant().isBefore(completeBy));

        // Exceeded maxDuration - return an error
        return left(new CompletionException(new RuntimeException("Timed out")));
    }
}
