/*
 * Decompiled with CFR 0.152.
 */
package io.apigee.trireme.kernel.fs;

import io.apigee.trireme.kernel.OSException;
import io.apigee.trireme.kernel.fs.AbstractFileHandle;
import io.apigee.trireme.kernel.fs.BasicFileHandle;
import io.apigee.trireme.kernel.fs.FileStats;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class BasicFilesystem {
    protected static final int FIRST_FD = 4;
    private static final Logger log = LoggerFactory.getLogger(BasicFilesystem.class);
    protected final AtomicInteger nextFd = new AtomicInteger(4);
    protected final ConcurrentHashMap<Integer, AbstractFileHandle> descriptors = new ConcurrentHashMap();

    public int open(File path, String origPath, int flags, int mode, int umask) throws OSException {
        if (log.isDebugEnabled()) {
            log.debug("open({}, {}, {})", new Object[]{origPath, flags, mode});
        }
        if ((flags & 0x200) != 0) {
            boolean justCreated;
            try {
                justCreated = path.createNewFile();
            }
            catch (IOException e) {
                if (log.isDebugEnabled()) {
                    log.debug("Error in createNewFile: {}", (Object)e, (Object)e);
                }
                throw new OSException(-5, e, origPath);
            }
            if (justCreated) {
                this.chmod(path, origPath, mode, umask, false);
            } else if ((flags & 0x800) != 0) {
                throw new OSException(-17, origPath);
            }
        } else if (!path.exists()) {
            throw new OSException(-2, origPath);
        }
        RandomAccessFile file = null;
        if (path.isFile()) {
            String modeStr = (flags & 2) != 0 ? "rw" : ((flags & 1) != 0 ? "rw" : "r");
            if ((flags & 0x80) != 0) {
                modeStr = "rws";
            }
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Opening {} with {}", (Object)path.getPath(), (Object)modeStr);
                }
                file = new RandomAccessFile(path, modeStr);
                if ((flags & 0x400) != 0) {
                    file.setLength(0L);
                } else if ((flags & 8) != 0 && file.length() > 0L) {
                    file.seek(file.length());
                }
            }
            catch (FileNotFoundException fnfe) {
                if (log.isDebugEnabled()) {
                    log.debug("File not found: {}", (Object)path);
                }
                throw new OSException(-2, origPath);
            }
            catch (IOException ioe) {
                if (log.isDebugEnabled()) {
                    log.debug("I/O error: {}", (Object)ioe, (Object)ioe);
                }
                throw new OSException(-5, ioe, origPath);
            }
        }
        BasicFileHandle handle = new BasicFileHandle(path, origPath, file);
        try {
            if ((flags & 8) != 0 && file != null && file.length() > 0L) {
                if (log.isDebugEnabled()) {
                    log.debug("  setting file position to {}", (Object)file.length());
                }
                handle.setPosition(file.length());
            }
        }
        catch (IOException ioe) {
            try {
                file.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            if (log.isDebugEnabled()) {
                log.debug("Error getting file length after opening it: {}", (Throwable)ioe);
            }
            throw new OSException(-5, ioe, origPath);
        }
        int fd = this.nextFd.getAndIncrement();
        handle.setFd(fd);
        this.descriptors.put(fd, handle);
        if (log.isDebugEnabled()) {
            log.debug("Opened FD {}", (Object)fd);
        }
        return fd;
    }

    public void close(int fd) throws OSException {
        AbstractFileHandle h = this.ensureHandle(fd);
        this.descriptors.remove(fd);
        try {
            if (h.getChannel() != null) {
                h.getChannel().close();
            }
        }
        catch (IOException ioe) {
            throw new OSException(-5);
        }
    }

    public long updatePosition(int fd, int delta) throws OSException {
        AbstractFileHandle h = this.ensureHandle(fd);
        long oldPos = h.getPosition();
        h.setPosition(oldPos + (long)delta);
        return oldPos;
    }

    public long getPosition(int fd) throws OSException {
        AbstractFileHandle h = this.ensureHandle(fd);
        return h.getPosition();
    }

    public int write(int fd, ByteBuffer buf, long pos) throws OSException {
        int written;
        AbstractFileHandle handle = this.ensureRegularFileHandle(fd);
        int origLen = buf.remaining();
        try {
            written = handle.getChannel().write(buf, pos);
        }
        catch (IOException ioe) {
            throw new OSException(-5, (Throwable)ioe);
        }
        if (log.isTraceEnabled()) {
            log.trace("write({}, {}) = {}", new Object[]{pos, origLen, written});
        }
        return written;
    }

    public int read(int fd, ByteBuffer buf, long pos) throws OSException {
        int read;
        AbstractFileHandle handle = this.ensureRegularFileHandle(fd);
        int origLen = buf.remaining();
        try {
            read = handle.getChannel().read(buf, pos);
        }
        catch (IOException ioe) {
            throw new OSException(-5, (Throwable)ioe);
        }
        if (log.isTraceEnabled()) {
            log.trace("read({}, {}) = {}", new Object[]{pos, origLen, read});
        }
        return read < 0 ? 0 : read;
    }

    public FileStats stat(File f, String origPath, boolean noFollow) throws OSException {
        if (!f.exists()) {
            throw new OSException(-2, origPath);
        }
        return new FileStats(f);
    }

    public FileStats fstat(int fd, boolean noFollow) throws OSException {
        AbstractFileHandle handle = this.ensureHandle(fd);
        return this.stat(handle.getFile(), handle.getOrigPath(), noFollow);
    }

    public void utimes(File f, String origPath, long atime, long mtime) throws OSException {
        if (!f.exists()) {
            OSException ne = new OSException(-2);
            ne.setPath(origPath);
            throw ne;
        }
        f.setLastModified(mtime);
    }

    public void futimes(int fd, long atime, long mtime) throws OSException {
        AbstractFileHandle handle = this.ensureHandle(fd);
        this.utimes(handle.getFile(), handle.getOrigPath(), atime, mtime);
    }

    public void chmod(File f, String origPath, int origMode, int umask, boolean nofollow) throws OSException {
        int mode = origMode & ~umask;
        if ((mode & 4) != 0 || (mode & 0x20) != 0) {
            f.setReadable(true, false);
        } else if ((mode & 0x100) != 0) {
            f.setReadable(true, true);
        } else {
            f.setReadable(false, true);
        }
        if ((mode & 2) != 0 || (mode & 0x10) != 0) {
            f.setWritable(true, false);
        } else if ((mode & 0x80) != 0) {
            f.setWritable(true, true);
        } else {
            f.setWritable(false, true);
        }
        if ((mode & 1) != 0 || (mode & 8) != 0) {
            f.setExecutable(true, false);
        } else if ((mode & 0x40) != 0) {
            f.setExecutable(true, true);
        } else {
            f.setExecutable(false, true);
        }
    }

    public void fchmod(int fd, int mode, int umask) throws OSException {
        AbstractFileHandle handle = this.ensureHandle(fd);
        this.chmod(handle.getFile(), handle.getOrigPath(), mode, umask, false);
    }

    public void mkdir(File file, String origPath, int mode, int umask) throws OSException {
        if (file.exists()) {
            throw new OSException(-17, origPath);
        }
        if (file.getParentFile() != null && !file.getParentFile().exists()) {
            throw new OSException(-2, origPath);
        }
        if (!file.mkdir()) {
            throw new OSException(-5, origPath);
        }
        this.chmod(file, origPath, mode, umask, false);
    }

    public void unlink(File file, String origPath) throws OSException {
        if (!file.exists()) {
            throw new OSException(-2, origPath);
        }
        if (!file.delete()) {
            throw new OSException(-5, origPath);
        }
    }

    public void rmdir(File file, String origPath) throws OSException {
        if (!file.exists()) {
            throw new OSException(-2, origPath);
        }
        if (!file.isDirectory()) {
            throw new OSException(-20, origPath);
        }
        if (!file.delete()) {
            throw new OSException(-5, origPath);
        }
    }

    public void rename(File oldFile, String oldPath, File newFile, String newPath) throws OSException {
        if (!oldFile.exists()) {
            throw new OSException(-2, oldPath);
        }
        if (newFile.getParentFile() != null && !newFile.getParentFile().exists()) {
            throw new OSException(-2, newPath);
        }
        if (!oldFile.renameTo(newFile)) {
            throw new OSException(-5, newPath);
        }
    }

    public void ftruncate(int fd, long len) throws OSException {
        BasicFileHandle handle = (BasicFileHandle)this.ensureRegularFileHandle(fd);
        try {
            handle.getFileHandle().setLength(len);
        }
        catch (IOException e) {
            throw new OSException(-5, e, handle.getOrigPath());
        }
    }

    public void fsync(int fd, boolean syncMetadata) throws OSException {
        AbstractFileHandle handle = this.ensureRegularFileHandle(fd);
        try {
            handle.getChannel().force(syncMetadata);
        }
        catch (IOException e) {
            throw new OSException(-5, (Throwable)e);
        }
    }

    public List<String> readdir(File f, String origPath) throws OSException {
        if (!f.isDirectory()) {
            throw new OSException(-20, origPath);
        }
        String[] files = f.list();
        if (files == null) {
            return Collections.emptyList();
        }
        return Arrays.asList(files);
    }

    public void chown(File file, String origPath, String uid, String gid, boolean noFollow) throws OSException {
        throw new OSException(-1, origPath);
    }

    public void fchown(int fd, String uid, String gid, boolean noFollow) throws OSException {
        AbstractFileHandle handle = this.ensureHandle(fd);
        this.chown(handle.getFile(), handle.getOrigPath(), uid, gid, noFollow);
    }

    public void link(File destFile, String destPath, File srcFile, String srcPath) throws OSException {
        throw new OSException(-13, srcPath);
    }

    public void symlink(File destFile, String destPath, File srcFile, String srcPath) throws OSException {
        throw new OSException(-13, srcPath);
    }

    public String readlink(File file, String origPath) throws OSException {
        throw new OSException(-13, origPath);
    }

    protected AbstractFileHandle ensureHandle(int fd) throws OSException {
        AbstractFileHandle handle = this.descriptors.get(fd);
        if (handle == null) {
            if (log.isDebugEnabled()) {
                log.debug("FD {} is not a valid handle", (Object)fd);
            }
            throw new OSException(-9);
        }
        return handle;
    }

    protected AbstractFileHandle ensureRegularFileHandle(int fd) throws OSException {
        AbstractFileHandle h = this.ensureHandle(fd);
        if (h.getChannel() == null) {
            throw new OSException(-9);
        }
        return h;
    }

    public void cleanup() {
        for (AbstractFileHandle handle : this.descriptors.values()) {
            BasicFileHandle bh;
            if (log.isDebugEnabled()) {
                log.debug("Closing leaked file descriptor " + handle);
            }
            if (!(handle instanceof BasicFileHandle) || (bh = (BasicFileHandle)handle).getFileHandle() == null) continue;
            try {
                bh.getFileHandle().close();
            }
            catch (IOException iOException) {}
        }
        this.descriptors.clear();
    }
}

