/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.memory.jetty;

import io.airlift.units.DataSize;
import java.lang.foreign.Arena;
import java.lang.foreign.MemorySegment;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.RetainableByteBuffer;

public class ConcurrentRetainableBufferPool
implements ByteBufferPool {
    private static final long DEFAULT_MAX_MEMORY = DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.MEGABYTE).toBytes();
    private static final int MIN_POOL_SIZE_POWER = 7;
    private static final int[] poolSizeShiftToSize = new int[25];
    private final ArenaBucket[] heapBuckets;
    private final ArenaBucket[] offHeapBuckets;
    private final AtomicBoolean evictor = new AtomicBoolean(false);
    private final int numBuckets = Runtime.getRuntime().availableProcessors() * 4;
    private final int checkMaxMemoryPoint = this.numBuckets * 100;
    private long maxHeapMemory;
    private long maxOffHeapMemory;
    private int checkCount;

    public ConcurrentRetainableBufferPool(long maxHeapMemory, long maxOffHeapMemory) {
        this.maxHeapMemory = maxHeapMemory > 0L ? maxHeapMemory : DEFAULT_MAX_MEMORY;
        this.maxOffHeapMemory = maxOffHeapMemory > 0L ? maxOffHeapMemory : DEFAULT_MAX_MEMORY;
        this.heapBuckets = new ArenaBucket[this.numBuckets];
        this.offHeapBuckets = new ArenaBucket[this.numBuckets];
        for (int bucketId = 0; bucketId < this.numBuckets; ++bucketId) {
            this.heapBuckets[bucketId] = new ArenaBucket(false, bucketId);
            this.offHeapBuckets[bucketId] = new ArenaBucket(true, bucketId);
        }
    }

    public RetainableByteBuffer acquire(int size, boolean offHeap) {
        int bucketId = Math.floorMod(Thread.currentThread().threadId(), this.numBuckets);
        if (offHeap) {
            return this.offHeapBuckets[bucketId].alloc(size);
        }
        return this.heapBuckets[bucketId].alloc(size);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkMaxMemory(boolean offHeap) {
        long max;
        long l = max = offHeap ? this.maxOffHeapMemory : this.maxHeapMemory;
        if (max <= 0L || !this.evictor.compareAndSet(false, true)) {
            return;
        }
        try {
            ++this.checkCount;
            if (this.checkCount % this.checkMaxMemoryPoint == 0 && this.getMemory(offHeap) > max) {
                this.evict(offHeap);
            }
        }
        finally {
            this.evictor.set(false);
        }
    }

    private void evict(boolean offHeap) {
        if (offHeap) {
            Arrays.stream(this.offHeapBuckets).forEach(ArenaBucket::evict);
        } else {
            Arrays.stream(this.heapBuckets).forEach(ArenaBucket::evict);
        }
    }

    private long getOffHeapMemory() {
        return this.getMemory(true);
    }

    private long getHeapMemory() {
        return this.getMemory(false);
    }

    private long getMemory(boolean offHeap) {
        if (offHeap) {
            return Arrays.stream(this.offHeapBuckets).mapToLong(ArenaBucket::getMemory).sum();
        }
        return Arrays.stream(this.heapBuckets).mapToLong(ArenaBucket::getMemory).sum();
    }

    public void clear() {
        this.evict(true);
        this.evict(false);
    }

    public String toString() {
        return String.format("%s{onHeap=%d/%d,offHeap=%d/%d}", super.toString(), this.getHeapMemory(), this.maxHeapMemory, this.getOffHeapMemory(), this.maxOffHeapMemory);
    }

    static {
        for (int i = 0; i < poolSizeShiftToSize.length; ++i) {
            ConcurrentRetainableBufferPool.poolSizeShiftToSize[i] = 1 << i + 7;
        }
    }

    private class ArenaBucket {
        private Arena sharedArena = Arena.ofShared();
        private final Arena autoArena = Arena.ofAuto();
        private final int bucketId;
        private final boolean offHeap;
        private final List<FixedSizeBufferPool> pools;

        ArenaBucket(boolean offHeap, int bucketId) {
            this.bucketId = bucketId;
            this.offHeap = offHeap;
            this.pools = new ArrayList<FixedSizeBufferPool>();
        }

        synchronized RetainableByteBuffer alloc(int size) {
            int poolSizeShift = this.getPoolSizeShift(size);
            if (poolSizeShift >= this.pools.size()) {
                this.addNewPools(poolSizeShift);
            }
            return this.pools.get(poolSizeShift).allocate(this.offHeap && poolSizeShift == 8 ? this.autoArena : this.sharedArena);
        }

        private int getPoolSizeShift(int size) {
            return Math.max(7, 32 - Integer.numberOfLeadingZeros(size - 1)) - 7;
        }

        private void addNewPools(int poolSizeShift) {
            int newPoolSizeShift;
            for (newPoolSizeShift = this.pools.size(); newPoolSizeShift <= poolSizeShift; ++newPoolSizeShift) {
                this.pools.add(new FixedSizeBufferPool(poolSizeShiftToSize[newPoolSizeShift], this.offHeap));
            }
            this.updateMaxMemoryIfNeeded(poolSizeShiftToSize[newPoolSizeShift] * 16);
        }

        private void updateMaxMemoryIfNeeded(int newMaxSize) {
            if (this.offHeap) {
                if ((long)newMaxSize > ConcurrentRetainableBufferPool.this.maxOffHeapMemory) {
                    ConcurrentRetainableBufferPool.this.maxOffHeapMemory = newMaxSize;
                }
            } else if ((long)newMaxSize > ConcurrentRetainableBufferPool.this.maxHeapMemory) {
                ConcurrentRetainableBufferPool.this.maxHeapMemory = newMaxSize;
            }
        }

        synchronized void evict() {
            boolean canClose = this.offHeap;
            for (FixedSizeBufferPool pool : this.pools) {
                pool.evict();
                canClose &= pool.getBufferCount() == 0;
            }
            if (canClose) {
                this.sharedArena.close();
                this.sharedArena = Arena.ofShared();
            }
        }

        synchronized long getMemory() {
            return this.pools.stream().mapToLong(FixedSizeBufferPool::getMemory).sum();
        }

        public String toString() {
            return String.format("%s{bucketId=%d,offHeap=%b,#pools=%d}", super.toString(), this.bucketId, this.offHeap, this.pools.size());
        }
    }

    private class Buffer
    implements RetainableByteBuffer {
        private final AtomicInteger refCount = new AtomicInteger(1);
        private final MemorySegment buffer;
        private final FixedSizeBufferPool pool;
        private ByteBuffer byteBuffer;

        Buffer(MemorySegment buffer, FixedSizeBufferPool pool) {
            this.buffer = buffer;
            this.pool = pool;
            this.byteBuffer = buffer.asByteBuffer();
            this.byteBuffer.limit(0);
            this.byteBuffer.position(0);
        }

        public void retain() {
            if (this.byteBuffer == null) {
                throw new IllegalStateException("Buffer cannot be retained since already released");
            }
            this.refCount.getAndUpdate(c -> c + 1);
        }

        public boolean release() {
            boolean shouldRelease;
            if (this.byteBuffer == null) {
                return true;
            }
            boolean bl = shouldRelease = this.refCount.updateAndGet(c -> c - 1) == 0;
            if (shouldRelease) {
                this.pool.free(this.buffer);
                this.byteBuffer = null;
                ConcurrentRetainableBufferPool.this.checkMaxMemory(this.pool.isOffHeap());
            }
            return shouldRelease;
        }

        public boolean canRetain() {
            return true;
        }

        public boolean isRetained() {
            return this.refCount.get() > 1;
        }

        public ByteBuffer getByteBuffer() {
            return this.byteBuffer;
        }
    }

    private class FixedSizeBufferPool {
        private final List<MemorySegment> buffers = new ArrayList<MemorySegment>();
        private final int bufferSize;
        private final boolean offHeap;
        private int allocatedBuffers;

        FixedSizeBufferPool(int bufferSize, boolean offHeap) {
            this.bufferSize = bufferSize;
            this.offHeap = offHeap;
        }

        synchronized Buffer allocate(Arena arena) {
            MemorySegment buffer = this.buffers.isEmpty() ? this.allocateNewBuffer(arena) : this.buffers.removeFirst();
            ++this.allocatedBuffers;
            return new Buffer(buffer, this);
        }

        synchronized void free(MemorySegment buffer) {
            if (this.allocatedBuffers == 0) {
                throw new RuntimeException("Pool has already freed all allocated segments");
            }
            --this.allocatedBuffers;
            this.buffers.add(buffer);
        }

        synchronized void evict() {
            this.buffers.clear();
        }

        private MemorySegment allocateNewBuffer(Arena arena) {
            return this.offHeap ? arena.allocate(this.bufferSize, 4L) : MemorySegment.ofArray(new byte[this.bufferSize]);
        }

        long getMemory() {
            return (long)(this.allocatedBuffers + this.buffers.size()) * (long)this.bufferSize;
        }

        int getBufferCount() {
            return this.allocatedBuffers + this.buffers.size();
        }

        boolean isOffHeap() {
            return this.offHeap;
        }
    }
}

